package com.sonicsw.ma.gui.runtime.notifications;

import java.awt.BorderLayout;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.util.ArrayList;

import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JInternalFrame;
import javax.swing.JPopupMenu;
import javax.swing.JToolBar;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.ToolTipManager;
import javax.swing.event.PopupMenuEvent;
import javax.swing.event.PopupMenuListener;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.WorkspaceWindow;
import com.sonicsw.ma.gui.runtime.JAbstractMonitorPanel;
import com.sonicsw.ma.gui.runtime.JMonitorDialog;
import com.sonicsw.ma.gui.runtime.notifications.model.INotificationWatcher;
import com.sonicsw.ma.gui.runtime.notifications.model.NotificationsModel;
import com.sonicsw.ma.gui.runtime.notifications.model.ParentNode;
import com.sonicsw.ma.gui.runtime.util.AbstractNode;
import com.sonicsw.ma.gui.runtime.util.IWatcher;
import com.sonicsw.ma.gui.util.BasicAction;
import com.sonicsw.ma.gui.util.BasicGuiAction;
import com.sonicsw.ma.gui.util.ExtendedJScrollPane;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.JBasicMenu;
import com.sonicsw.ma.gui.util.JBasicMenuItem;
import com.sonicsw.ma.gui.util.JMAInternalFrame;
import com.sonicsw.ma.gui.util.PopupMenuShower;
import com.sonicsw.ma.gui.util.ResourceManager;
import com.sonicsw.ma.plugin.IMonitorPlugin;

public final class JNotificationsPanel extends    JAbstractMonitorPanel
                                       implements TreeModelListener
{
    private BasicAction m_watchAction = new WatchButtonAction();
    private JTree       m_tree;
    private TreeSelectionListener tsl;


    public JNotificationsPanel()
    {
        super();

        m_tree = new JTree((TreeModel)null)
        {
            @Override
            public String getToolTipText(MouseEvent event)
            {
                TreePath path = getPathForLocation(event.getX(), event.getY());
                if (path == null)
                {
                    return null;
                }

                return ((AbstractNode)path.getLastPathComponent()).getDescription();
            }
        };
        m_tree.setRootVisible(false);
        m_tree.setShowsRootHandles(true);
        m_tree.setCellRenderer(new NotificationsTreeCellRenderer());
        m_tree.setVisibleRowCount(10);
        m_tree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
        m_tree.addMouseListener(new PopupMenuShower(m_tree, buildPopupMenu()));
        tsl = new TreeSelectionListener()
                {
                    @Override
                    public void valueChanged(TreeSelectionEvent evt) { enableButtons(); }
                };
        m_tree.addTreeSelectionListener(tsl);

        ExtendedJScrollPane scrollPane = new ExtendedJScrollPane(m_tree);


        JToolBar tb = new JToolBar(JToolBar.HORIZONTAL);
        tb.setFloatable(false);
        tb.setRollover(true);
        tb.add(new JWatchButton(m_watchAction));

        add(tb,         BorderLayout.NORTH);
        add(scrollPane, BorderLayout.CENTER);

        enableButtons();

        expandTree();
    }

    @Override
    public Icon getIcon() { return ResourceManager.getIcon(getClass(), "Alert"); }

    @Override
    public void maInitialize()
    {
        ToolTipManager.sharedInstance().registerComponent(m_tree);
    }

    @Override
    public void maCleanup()
    {
        NotificationsModel model = (NotificationsModel) m_tree.getModel();
        if (model != null)
        {
            model.removeTreeModelListener(this);
        }
        m_tree.removeTreeSelectionListener(tsl);
        m_tree.setModel(null);
        ToolTipManager.sharedInstance().unregisterComponent(m_tree);
    }

    @Override
    public String getTabTitle()
    {
        return "Notifications";
    }

    @Override
    public String getMonitorId()
    {
        return IMonitorPlugin.MONITOR_ID_NOTIFICATIONS;
    }

    @Override
    public void changeModel(IMonitorPlugin plugin)
    {
        NotificationsModel newModel = (NotificationsModel)plugin.getMonitorModel(IMonitorPlugin.MONITOR_ID_NOTIFICATIONS);
        NotificationsModel oldModel = (NotificationsModel)m_tree.getModel();

        if (oldModel != null)
        {
            oldModel.removeTreeModelListener(this);
        }

        m_tree.setModel(newModel);

        if (newModel != null)
        {
            newModel.addTreeModelListener(this);
        }

        expandTree();
    }

    private void expandTree()
    {
        // expand all but instance nodes
        int row = 0;
        while (row < m_tree.getRowCount())
        {
            if (m_tree.getPathForRow(row).getLastPathComponent() instanceof ParentNode)
            {
                m_tree.expandRow(row);
            }
            row++;
        }
    }

    private JPopupMenu buildPopupMenu()
    {
        JPopupMenu menu = new JPopupMenu();

        JBasicMenu watchMenu = new JBasicMenu((BasicAction)m_watchAction);

        watchMenu.add(new JBasicMenuItem(new WatchAction("NotificationsSheet.watch.component", IWatcher.COMPONENT_SCOPE)));
        watchMenu.add(new JBasicMenuItem(new WatchAction("NotificationsSheet.watch.container", IWatcher.CONTAINER_SCOPE)));
        watchMenu.add(new JBasicMenuItem(new WatchAction("NotificationsSheet.watch.domain",    IWatcher.DOMAIN_SCOPE)));

        menu.add(watchMenu);

        return menu;
    }

    private void enableButtons()
    {
        TreePath path = m_tree.getSelectionPath();
        m_watchAction.setEnabled((path == null) ? false : true);
    }

    private AbstractNode getCurrentNode()
    {
        TreePath path = m_tree.getSelectionPath();
        return (AbstractNode)path.getLastPathComponent();
    }

    private void setCurrentNode(AbstractNode node)
    {
        ArrayList path = new ArrayList();
        while (node != null)
        {
            path.add(0, node);
            node = (AbstractNode)node.getParent();
        }

        m_tree.setSelectionPath(new TreePath(path.toArray()));
        enableButtons();
    }

    //-------------------------------------------------------------------------
    //
    // TreeModelListener Implementation
    //
    //-------------------------------------------------------------------------

    @Override
    public void treeNodesChanged(TreeModelEvent evt)     { enableButtons(); }
    @Override
    public void treeNodesInserted(TreeModelEvent evt)    { enableButtons(); }
    @Override
    public void treeNodesRemoved(TreeModelEvent evt)     { enableButtons(); }
    @Override
    public void treeStructureChanged(TreeModelEvent evt) { enableButtons(); }

    //-------------------------------------------------------------------------
    //
    // Inner classes
    //
    //-------------------------------------------------------------------------

    class JWatchButton extends JButton
    {
        public JWatchButton(Action action)
        {
            super(action);
        }

        @Override
        protected void configurePropertiesFromAction(Action a)
        {
            String[] types = { Action.SMALL_ICON,
                               Action.SHORT_DESCRIPTION,
                               Action.LONG_DESCRIPTION,
                               "enabled" };

            Helper.configurePropertiesFromAction(this, a, types);
        }
    }

    private class WatchButtonAction
    extends BasicGuiAction
    {
        public WatchButtonAction()
        {
            super("NotificationsSheet.watch");

//            putValue(Action.SMALL_ICON, ArrowIcon.ARROW_SOUTH);
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            JPopupMenu pm = new JPopupMenu("-");

            pm.add(new JBasicMenuItem(new WatchAction("NotificationsSheet.watch.component", IWatcher.COMPONENT_SCOPE)));
            pm.add(new JBasicMenuItem(new WatchAction("NotificationsSheet.watch.container", IWatcher.CONTAINER_SCOPE)));
            pm.add(new JBasicMenuItem(new WatchAction("NotificationsSheet.watch.domain",    IWatcher.DOMAIN_SCOPE)));
            pm.addPopupMenuListener(new PopupMenuListener()
            {
                @Override
                public void popupMenuCanceled(PopupMenuEvent evt) {}
                @Override
                public void popupMenuWillBecomeVisible(PopupMenuEvent evt) {}
                @Override
                public void popupMenuWillBecomeInvisible(PopupMenuEvent evt)
                {
                      m_watchAction.setSelected(false);
                }
            });

            if (pm.getComponentCount() > 0)
            {
                JButton src = (JButton)evt.getSource();
                Point   pt  = Helper.getPopupMenuOrigin(src, pm);

                pm.show(src, pt.x, pt.y);
            }
        }
    }

    private class WatchAction extends BasicGuiAction
    {
        private short m_scope;

        public WatchAction(String resource, short scope)
        {
            super(resource);

            m_scope = scope;
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            AbstractNode         node    = getCurrentNode();
            NotificationsModel   model   = (NotificationsModel)m_tree.getModel();
            INotificationWatcher watcher = NotificationsWatchWindow.getWatchWindow(m_scope, model.getComponentName());

            if (!((JInternalFrame)watcher).isVisible())
            {
                JMonitorDialog  parent = (JMonitorDialog)SwingUtilities.getAncestorOfClass(JMonitorDialog.class, JNotificationsPanel.this);
                WorkspaceWindow domainParent = parent.getWorkspaceWindow();

                if (domainParent != null)
                {
                    domainParent.addDependantFrame((JMAInternalFrame)watcher);

                    MgmtConsole.getMgmtConsole().showInternalFrame((JInternalFrame)watcher, true);
                }
            }

            // get the remote subscription going if not already subscribed
            try
            {
                model.subscribeNotification(node);
            }
            catch(Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                                       "Failed to subscribe to notifications", e, true);
                return;
            }

            watcher.addWatch(model.getChildNotifications(node), model);
        }
    }
}
