package com.sonicsw.ma.gui.runtime.notifications;

import java.awt.BorderLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Map;
import java.util.Set;

import javax.management.Notification;
import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.UIManager;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.TableColumn;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.table.IModelTableModel;
import com.sonicsw.ma.gui.table.ModelListTableModel;
import com.sonicsw.ma.gui.table.RowTableColumn;
import com.sonicsw.ma.gui.util.BasicGuiAction;
import com.sonicsw.ma.gui.util.ExtendedJScrollPane;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.JButtonPanel;
import com.sonicsw.ma.gui.util.JMADialog;
import com.sonicsw.ma.gui.util.JMAFrame;
import com.sonicsw.ma.gui.util.JPartitionPanel;
import com.sonicsw.ma.gui.util.JRowTable;

import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.jmx.client.MFNotification;

public final class NotificationAttributesSheet extends JMADialog implements ListSelectionListener
{
    private static final String NA           = "n/a";
    private static final String RES_PREVIOUS = "notification.previous";
    private static final String RES_NEXT     = "notification.next";

    private Notification m_notification;
    private int          m_scope;
    private JRowTable    m_parentTable;
    private JTabbedPane  m_tab;
    private JRowTable    m_attrTable;
    private JTextArea    m_attrDetail;

    JTextField m_tfSource       = new JTextField();
    JTextField m_tfHost         = new JTextField();
    JTextArea  m_taNotification = new JTextArea("");
    JTextField m_tfSeverity     = new JTextField();
    JTextField m_tfType         = new JTextField();
    JTextField m_tfSequence     = new JTextField();
    JTextField m_tfDateTime     = new JTextField();


    public NotificationAttributesSheet(JMAFrame     parent,
                                       Notification notification,
                                       int          scope,
                                       JRowTable    parentTable)
    {
        super(parent, "notifications.attributes");

        m_notification = notification;
        m_scope        = scope;
        m_parentTable  = parentTable;
    }

    @Override
    public void maInitialize()
    {
        m_tab = new JTabbedPane(JTabbedPane.TOP);

        getContentPane().add(m_tab);

        m_tab.add(JPartitionPanel.wrap(makeGeneralPanel()), "General");
        m_tab.add(makeAttributesPanel(),                    "Attributes");

        setNotification(m_notification);

        m_parentTable.getSelectionModel().addListSelectionListener(this);

        updatePreviousNextButtons();
    }

    @Override
    public void maCleanup()
    {
        if (m_parentTable != null)
        {
            m_parentTable.getSelectionModel().removeListSelectionListener(this);
            m_parentTable.cleanup();
            m_parentTable = null;
        }

        m_notification = null;

        if (m_attrTable != null)
        {
            m_attrTable.cleanup();
        }
    }

    @Override
    public void valueChanged(ListSelectionEvent evt)
    {
        if (evt.getValueIsAdjusting())
        {
            return;
        }

        int               sel   = m_parentTable.getSelectedRow();
        NotificationValue value = (NotificationValue)((IModelTableModel)m_parentTable.getModel()).getRowModel(sel);

        if ((value != null) && (value.getNotification() != null))
        {
            setNotification(value.getValue());
        }
    }

    protected void updatePreviousNextButtons()
    {
        m_buttonPanel.getAction(RES_PREVIOUS).setEnabled(m_parentTable.getSelectedRow() > 0);
        m_buttonPanel.getAction(RES_NEXT).setEnabled(m_parentTable.getSelectedRow() < (m_parentTable.getRowCount()-1));
    }

    @Override
    protected JPanel createButtonPanel(Action[] action)
    {
        m_buttonPanel = new JButtonPanel(false);

        JButton btnPrevious = new JButton(new PreviousAction());  btnPrevious.setMargin(new Insets(0, 0, 0, 0));
        JButton btnNext     = new JButton(new NextAction());      btnNext.setMargin(new Insets(0, 0, 0, 0));

        m_buttonPanel.add(btnPrevious);
        m_buttonPanel.add(btnNext);
        m_buttonPanel.addRemainder();
        m_buttonPanel.add(new JButton(new CloseAction(getDefaultCancelAction())));

        return m_buttonPanel;
    }

    private JPanel makeGeneralPanel()
    {
        JPartitionPanel panel = new JPartitionPanel("");

        m_tfSource.setEditable(false);
        m_tfHost.setEditable(false);
        m_taNotification.setEditable(false);
        m_taNotification.setLineWrap(true);
        m_taNotification.setWrapStyleWord(true);
        m_taNotification.setRows(3);
        m_taNotification.setBackground(UIManager.getColor("Panel.background"));
        m_tfSeverity.setEditable(false);
        m_tfType.setEditable(false);
        m_tfSequence.setEditable(false);
        m_tfDateTime.setEditable(false);

        panel.addRow("Source",       m_tfSource);
        panel.addRow("Hostname",     m_tfHost);
        panel.addRow("Notification", new JScrollPane(m_taNotification));
        panel.addRow("Severity",     m_tfSeverity);
        panel.addRow("Log Type",     m_tfType);
        panel.addRow("Sequence",     m_tfSequence);
        panel.addRow("Date & Time",  m_tfDateTime);

        return panel;
    }

    private void setNotification(Notification newNotification)
    {
        try
        {
            boolean isMF     = (newNotification instanceof MFNotification);
            String  source   = newNotification.getSource().toString();
            String  host     = isMF ? ((MFNotification)newNotification).getSourceHost() : NA;
            String  text     = newNotification.getType();
            String  severity = getSeverity(newNotification);
            String  logType  = getLogType(newNotification);
            String  sequence = getSequence(newNotification);
            String  dateTime = Helper.getDateTimeString(new Long(newNotification.getTimeStamp()));

            m_tfSource.setText(source);
            m_tfHost.setText(host);
            m_taNotification.setText(text);
            m_tfSeverity.setText(severity);
            m_tfType.setText(logType);
            m_tfSequence.setText(sequence);
            m_tfDateTime.setText(dateTime);

            m_attrDetail.setText("");
            if (newNotification instanceof MFNotification)
            {
                Set attr = ((MFNotification)newNotification).getAttributes().entrySet();

                ((IModelTableModel)m_attrTable.getModel()).setContents(new ArrayList(attr));
            }
            else
            {
                ((IModelTableModel)m_attrTable.getModel()).clear();
            }
            
            m_attrTable.clearSelection(); // Selection not valid after contents have changed
                                          // and the 'value' textarea gets out of sync.
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR, "Failed to set notification in dialog view", e, false);
        }
    }

    private JComponent makeAttributesPanel()
    {
        TableColumn[] tableColumns = new TableColumn[]
        {
            new NameColumn(0,  "Attribute Name", 30),
            new ValueColumn(1, "Value",          70),
        };

        m_attrTable = new JRowTable(tableColumns, getClass().getName());
        m_attrTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        m_attrTable.getSelectionModel().addListSelectionListener(new ListSelectionListener()
        {
            @Override
            public void valueChanged(ListSelectionEvent evt)
            {
                if (evt.getValueIsAdjusting())
                {
                    return;
                }

                int       sel   = m_attrTable.getSelectedRow();
                Map.Entry entry = (Map.Entry)((ModelListTableModel)m_attrTable.getModel()).getRowModel(sel);

                if (entry == null)
                {
                    m_attrDetail.setText(" ");
                }
                else
                if (entry.getValue() == null)
                {
                    m_attrDetail.setText("<null>");
                }
                else
                {
                    boolean isArray = entry.getValue().getClass().isArray();

                    m_attrDetail.setLineWrap(!isArray);
                    m_attrDetail.setWrapStyleWord(!isArray);
                    m_attrDetail.setText(getAttributeValue(entry.getValue(), true));
                }
            }
        });

/* Test entries...
        ((ModelListTableModel)m_attrTable.getModel()).insertRow(new TestMapEntry("test1", "sfsdf sdf sdf sdfsdf sdf sdf werwer dfg dfg dfg ert wer werwerwer  dsfgsdfsdf sdf sdfsdf sdfsdf"), 0);
        ((ModelListTableModel)m_attrTable.getModel()).insertRow(new TestMapEntry("test2", new Object[] { Boolean.FALSE, "Jack", "and", "Jill", new Integer(201) }), 0);
        ((ModelListTableModel)m_attrTable.getModel()).insertRow(new TestMapEntry("test2", null), 0);
*/

        m_attrDetail = new JTextArea(" ");
        m_attrDetail.setEditable(false);
        m_attrDetail.setBackground(UIManager.getColor("Panel.background"));

        JPanel preview = new JPanel(new BorderLayout());
        preview.add(new JLabel("Value:"), BorderLayout.NORTH);
        preview.add(new ExtendedJScrollPane(m_attrDetail), BorderLayout.CENTER);

        JSplitPane sp = new JSplitPane(JSplitPane.VERTICAL_SPLIT, new ExtendedJScrollPane(m_attrTable), preview);
        sp.setOneTouchExpandable(true);
        sp.setResizeWeight(0.6);
        return sp;
    }

/*
    class TestMapEntry implements Map.Entry
    {
        Object key;
        Object value;

        public TestMapEntry(Object key, Object value)
        {
            this.key = key;
            this.value = value;
        }

        public Object getKey() { return key; }
        public Object getValue()  { return value; }
        public Object setValue(Object value)
        {
            Object oldValue = this.value;
            this.value = value;
            return oldValue;
        }
    }
*/

    private static String getAttributeValue(Object object, boolean preview)
    {
        if (object == null)
        {
            return "<null>";
        }

        // special treatment for dates
        if (object instanceof Date)
        {
            return Helper.getDateTimeString(new Long(((Date) object).getTime()));
        }

        // special treatment for dates in GregorianCalendar format
        if (object instanceof GregorianCalendar)
        {
            GregorianCalendar calendar = ((GregorianCalendar) object);
            return Helper.getDateTimeString(calendar.getTimeInMillis()) + " " + calendar.getTimeZone().getDisplayName() ;
        }

        // only other special treatment for arrays only
        if (!object.getClass().isArray())
        {
            return object.toString();
        }

        StringBuffer buffer = new StringBuffer();
        Object[]     array  = (Object[])object;
        for (int i = 0; i < array.length; i++)
        {
            if (i > 0)
            {
                if (preview)
                {
                    buffer.append('\n');
                }
                else
                {
                    buffer.append(", ");
                }
            }

            buffer.append('[').append(i).append(']').append(' ').append(array[i].toString());
        }

        return buffer.toString();
    }

    private String getSeverity(Notification notification)
    {
        if (!(notification instanceof INotification))
        {
            return NA;
        }

        return Level.LEVEL_TEXT[((INotification)notification).getSeverity()];
    }

    private String getLogType(Notification notification)
    {
        if (!(notification instanceof INotification))
        {
            return NA;
        }

        return INotification.LOGTYPE_TEXT[((INotification)notification).getLogType()];
    }

    private String getSequence(Notification notification)
    {
        String res = NA;

        if (notification instanceof INotification)
        {
            res = Long.toString(((INotification)notification).getSequenceNumber());
        }

        return res;
    }

    class CloseAction extends BasicGuiAction
    {
        public CloseAction(Action targetAction)
        {
            super("dialog.close", targetAction);
        }
    }

    class PreviousAction extends BasicGuiAction
    {
        public PreviousAction()
        {
            super(RES_PREVIOUS);
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            int curSel = m_parentTable.getSelectedRow();

            m_parentTable.setRowSelectionInterval(curSel - 1, curSel - 1);

            updatePreviousNextButtons();
        }
    }

    class NextAction extends BasicGuiAction
    {
        public NextAction()
        {
            super(RES_NEXT);
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            int curSel = m_parentTable.getSelectedRow();

            m_parentTable.setRowSelectionInterval(curSel + 1, curSel + 1);

            updatePreviousNextButtons();
        }
    }

    private static class NameColumn extends RowTableColumn
    {
        NameColumn(int modelIndex, String name, int width)
        {
            super(modelIndex, name, width);
        }

        @Override
        public Object getColumnValue(Object rowModel)
        {
            return ((Map.Entry) rowModel).getKey();
        }
    }

    private static class ValueColumn extends RowTableColumn
    {
        ValueColumn(int modelIndex, String name, int width)
        {
            super(modelIndex, name, width);

            setSortable(false);
        }

        @Override
        public Object getColumnValue(Object rowModel)
        {
            Object object = ((Map.Entry) rowModel).getValue();

            return getAttributeValue(object, false);
        }
    }
}
