package com.sonicsw.ma.gui.runtime.notifications;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.NotificationListener;
import javax.management.ObjectName;
import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JSeparator;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.table.TableColumn;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.IWorkspaceWindow;
import com.sonicsw.ma.gui.JPreferencesDialog;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.PreferenceManager;
import com.sonicsw.ma.gui.runtime.notifications.model.AbstractNotificationsModel;
import com.sonicsw.ma.gui.runtime.notifications.model.INotificationWatcher;
import com.sonicsw.ma.gui.runtime.util.IWatcher;
import com.sonicsw.ma.gui.table.IModelTableModel;
import com.sonicsw.ma.gui.table.RowTableColumn;
import com.sonicsw.ma.gui.util.BasicGuiAction;
import com.sonicsw.ma.gui.util.ExtendedJScrollPane;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.JBasicMenu;
import com.sonicsw.ma.gui.util.JBasicMenuItem;
import com.sonicsw.ma.gui.util.JMADialog;
import com.sonicsw.ma.gui.util.JMAFrame;
import com.sonicsw.ma.gui.util.JMAInternalFrame;
import com.sonicsw.ma.gui.util.JRowTable;
import com.sonicsw.ma.gui.util.PopupMenuShower;
import com.sonicsw.ma.gui.util.ResourceManager;
import com.sonicsw.ma.plugin.IPlugin;

import com.sonicsw.mf.common.runtime.impl.CanonicalName;
import com.sonicsw.mf.jmx.client.MFNotification;

public class NotificationsWatchWindow
    extends JMAInternalFrame
    implements INotificationWatcher, NotificationListener, IWorkspaceWindow
{
    private static final Map m_watchWindows = new HashMap();

    private int       m_maxNotifications = 10;
    private short     m_scope;
    private String    m_scopeID;
    private JRowTable m_table;
    private Map       m_data = new HashMap();

    static NotificationsWatchWindow getWatchWindow(short scope, ObjectName scopeName)
    {
        String name = null;

        // first determine the name to use based on the scope
        CanonicalName cName = new CanonicalName(scopeName.toString());

        switch (scope)
        {
            case DOMAIN_SCOPE    : name = cName.getDomainName(); break;
            case CONTAINER_SCOPE : name = cName.getDomainName() + "." + cName.getContainerName(); break;
            case COMPONENT_SCOPE :
            default              : name = cName.getCanonicalName(); break;
        }

        NotificationsWatchWindow window = (NotificationsWatchWindow)m_watchWindows.get(scope + name);
        if (window == null)
        {
            window = new NotificationsWatchWindow(scope, name);
            m_watchWindows.put(scope + name, window);
        }

        return window;
    }

    public static void removeAllNotifications(String componentName)
    {
        Iterator i = m_watchWindows.entrySet().iterator();

        while (i.hasNext())
        {
            Map.Entry                entry  = (Map.Entry)i.next();
            NotificationsWatchWindow window = (NotificationsWatchWindow)entry.getValue();

            window.removeAllWatches(componentName);
        }
    }

    /**
     * Constructor
     */
    private NotificationsWatchWindow(short scope, String scopeID)
    {
        super("notifications.watcher");
        setProperties(scopeID);

        m_scope = scope;
        m_scopeID = scopeID;
    }

    private void setProperties(String scopeID) {
        setTitle(getTitle() + " - " + scopeID);
        setFrameIcon(ResourceManager.getIcon(getClass(), "Alert"));
    }
    
    @Override
    protected void maInitialize()
    {
        // set initial values from preferences
        PreferenceManager pManager = PreferenceManager.getInstance();

        m_maxNotifications = pManager.getInt("preferences.notifications",
                                             JPreferencesDialog.MAX_NOTIFICATIONS,
                                             JPreferencesDialog.DEFAULT_MAX_NOTIFICATIONS);

        initPanels();
    }

    @Override
    protected void maCleanup()
    {
        // remove listener subscriptions and cleanup maps
        Iterator i = m_data.values().iterator();
        while (i.hasNext())
        {
            WatchModel dataModel = (WatchModel)i.next();

            try
            {
                dataModel.removeAllSubscriptions(this);
            }
            catch (Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                    "Failed to remove notification subscriptions", e, true);
            }
        }
        m_data.clear();

        m_watchWindows.remove(m_scope + m_scopeID);

        if (m_table != null)
        {
            m_table.cleanup();
        }
    }

    @Override
    public boolean canClose()
    {
        boolean res = true;

        if ((m_data != null) && (!m_data.isEmpty()))
        {
            res = (JOptionPane.showConfirmDialog(this,
                                                 "Are you sure you want to close watcher:\n" + m_scopeID,
                                                 "Close Notifications Watcher",
                                                 JOptionPane.YES_NO_OPTION) == JOptionPane.YES_OPTION);
        }

        return res;
    }

    @Override
    public JComponent[] getSelectedMenuItems(int type)
    {
        ArrayList menuItems = new ArrayList();

        if (type == IPlugin.PLUGIN_TYPE)
        {
            JBasicMenu clearMenu = new JBasicMenu(new ClearAction());

            clearMenu.add(new JBasicMenuItem(new ClearSelectedAction()));
            clearMenu.add(new JBasicMenuItem(new ClearSelectedTypeAction()));
            clearMenu.add(new JBasicMenuItem(new ClearAllAction()));


            menuItems.add(new JBasicMenuItem(new OptionsAction()));
            menuItems.add(new JSeparator());
            menuItems.add(clearMenu);
            menuItems.add(new JBasicMenuItem(new RemoveWatchAction()));
            menuItems.add(new JSeparator());
            menuItems.add(new JBasicMenuItem(new PropertiesAction()));

            if (MgmtConsole.DEVELOPER_MODE)
            {
                menuItems.add(new JBasicMenuItem(new DebugAction()));
            }
        }

        return (JComponent[])menuItems.toArray(new JComponent[0]);
    }

    @Override
    public int getMaxNotifications()
    {
        return m_maxNotifications;
    }

    @Override
    public void setMaxNotifications(int maxNotifications)
    {
        m_maxNotifications = maxNotifications;
        enforceMaxNotifications(m_maxNotifications);
    }

    private void initPanels()
    {
        super.getContentPane().setLayout(new BorderLayout());
        super.getContentPane().add(makeTablePanel(), BorderLayout.CENTER);
    }

    private JPanel makeTablePanel()
    {
        JPanel panel = new JPanel(new BorderLayout());

        panel.add(makeTable(), BorderLayout.CENTER);

        return panel;
    }

    private JComponent makeTable()
    {
        // only show the source component if this is not a component watcher
        TableColumn[] tableColumns = null;

        if (m_scope < IWatcher.COMPONENT_SCOPE)
        {
            tableColumns = new TableColumn[]
                           {
                           new SourceColumn(0, "Source", 15),
                           new HostnameColumn(1, "Hostname", 15),
                           new NotificationColumn(2, "Notification", 15),
                           new SeverityColumn(3, "Severity", 15),
                           new LogTypeColumn(4, "Log Type", 15),
                           new SequenceColumn(5, "Sequence", 10),
                           new TimeColumn(6, "Date & Time", 15),
            };
        }
        else
        {
            tableColumns = new TableColumn[]
                           {
                           new NotificationColumn(0, "Notification", 30),
                           new SeverityColumn(1, "Severity", 15),
                           new LogTypeColumn(2, "Log Type", 15),
                           new SequenceColumn(3, "Sequence", 20),
                           new TimeColumn(4, "Date & Time", 20),
            };
        }

        m_table = new JRowTable(tableColumns, getClass().getName());
        m_table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

        // put the table in a scroll pane now so that when we add the
        // PopupMenuShower to the table it will also apply to the scroll pane
        JComponent ret = new ExtendedJScrollPane(m_table);

        // add the popup menu
        JPopupMenu popup = new JPopupMenu();

        JComponent[] menuItems = getSelectedMenuItems(IPlugin.PLUGIN_TYPE);

        for (int i = 0; i < menuItems.length; i++)
        {
            popup.add(menuItems[i]);
        }

        m_table.addMouseListener(new LocalMenuShower(m_table, popup));

        return ret;
    }

    @Override
    public synchronized void addWatch(MBeanNotificationInfo      info,
                                      AbstractNotificationsModel model)
    {
        addWatch(new MBeanNotificationInfo[] {info} , model);
    }

    @Override
    public synchronized void addWatch(MBeanNotificationInfo[]    info,
                                      AbstractNotificationsModel model)
    {
        String     componentName = model.getComponentName().getCanonicalName();
        WatchModel dataModel     = (WatchModel)m_data.get(componentName);

        if (dataModel == null)
        {
            dataModel = new WatchModel(model);

            m_data.put(componentName, dataModel);
        }

        try
        {
            dataModel.addSubscriptions(info, this);
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                "Failed to create notification subscription(s)", e, true);
        }
    }

    private void debug()
    {
        Iterator i = m_data.values().iterator();
        while (i.hasNext())
        {
            WatchModel dataModel = (WatchModel)i.next();

            System.err.println("Subscription to '" + dataModel.getComponentName() + "'");

            Iterator iN = dataModel.getSubscriptions().keySet().iterator();
            while (iN.hasNext())
            {
                String eN = (String)iN.next();

                System.err.println("  " + eN);
            }
        }
    }

    private synchronized void removeWatch(NotificationValue notificationValue)
    {
        Notification notification  = notificationValue.getValue();
        String       componentName = notification.getSource().toString();

        WatchModel dataModel = (WatchModel)m_data.get(componentName);

        if (dataModel != null)
        {
            try
            {
                dataModel.removeSubscription(notification.getType(), this);
            }
            catch (Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                    "Failed to remove notification subscriptions", e, true);
            }

            if (!dataModel.isUsed())
            {
                m_data.remove(componentName);
            }
        }
    }

    private synchronized void removeAllWatches(String componentName)
    {
        // remove listener subscriptions and cleanup maps
        WatchModel dataModel = (WatchModel)m_data.get(componentName);

        try
        {
            dataModel.removeAllSubscriptions(this);
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                "Failed to remove all notification subscriptions", e, true);
        }

        m_data.remove(componentName);
    }

    private synchronized void clearSelectedNotification(NotificationValue notificationValue)
    {
        ((IModelTableModel)m_table.getModel()).delete(notificationValue);
    }

    private synchronized void clearSelectedTypeNotifications(NotificationValue notificationValue)
    {
        IModelTableModel tableModel = (IModelTableModel)m_table.getModel();
        String notificationType = notificationValue.getValue().getType();
        for (int i = tableModel.getRowCount() - 1; i >= 0; i--)
        {
            NotificationValue rowValue = (NotificationValue)tableModel.getRowModel(i);
            if (rowValue.getValue().getType().equals(notificationType))
            {
                tableModel.deleteRow(i);
            }
        }
    }

    private synchronized void clearAllNotifications()
    {
        ((IModelTableModel)m_table.getModel()).clear();
    }

    @Override
    public short getWatchScope()
    {
        return m_scope;
    }

    @Override
    public synchronized void handleNotification(Notification notification, Object handback)
    {
        String     componentName = notification.getSource().toString();
        String     type          = notification.getType();
        WatchModel dataModel     = (WatchModel)m_data.get(componentName);

        if ((dataModel == null) || !dataModel.getSubscriptions().containsKey(type))
         {
            return; // to handle timing issues
        }

        NotificationValue notificationValue = new NotificationValue(notification);

        enforceMaxNotifications(m_maxNotifications - 1);
        ((IModelTableModel)m_table.getModel()).addRow(notificationValue);
    }

    private NotificationValue getSelectedRowValue()
    {
        int row = m_table.getSelectedRow();

        if (row == -1)
        {
            return null;
        }

        return (NotificationValue)((IModelTableModel)m_table.getModel()).getRowModel(row);
    }

    private synchronized void enforceMaxNotifications(int maxNotifications)
    {
        IModelTableModel tableModel = (IModelTableModel)m_table.getModel();

        while (tableModel.getRowCount() > maxNotifications)
        {
            // find the oldest and delete that
            NotificationValue oldest = null;
            Iterator iterator = tableModel.getContents().iterator();
            while (iterator.hasNext())
            {
                NotificationValue current = (NotificationValue)iterator.next();
                if (oldest == null || oldest.getValue().getTimeStamp() > current.getValue().getTimeStamp())
                {
                    oldest = current;
                }
            }
            tableModel.delete(oldest);
        }
    }

    //-------------------------------------------------------------------------

    private static class SourceColumn extends RowTableColumn
    {
        SourceColumn(int modelIndex, String name, int width)
        {
            super(modelIndex, name, width);
        }

        @Override
        public Object getColumnValue(Object rowModel)
        {
            return ((NotificationValue)rowModel).getSource();
        }
    }

    private static class HostnameColumn extends RowTableColumn
    {
        HostnameColumn(int modelIndex, String name, int width)
        {
            super(modelIndex, name, width);
        }

        @Override
        public Object getColumnValue(Object rowModel)
        {
            return ((NotificationValue)rowModel).getHost();
        }
    }

    private static class NotificationColumn extends RowTableColumn
    {
        NotificationColumn(int modelIndex, String name, int width)
        {
            super(modelIndex, name, width);
        }

        @Override
        public Object getColumnValue(Object rowModel)
        {
            return ((NotificationValue)rowModel).getNotification();
        }
    }

    private static class SeverityColumn extends RowTableColumn
    {
        SeverityColumn(int modelIndex, String name, int width)
        {
            super(modelIndex, name, width);
        }

        @Override
        public Object getColumnValue(Object rowModel)
        {
            return ((NotificationValue)rowModel).getSeverity();
        }
    }

    private static class LogTypeColumn extends RowTableColumn
    {
        LogTypeColumn(int modelIndex, String name, int width)
        {
            super(modelIndex, name, width);
        }

        @Override
        public Object getColumnValue(Object rowModel)
        {
            return ((NotificationValue)rowModel).getLogType();
        }
    }

    private static class SequenceColumn extends RowTableColumn
    {
        SequenceColumn(int modelIndex, String name, int width)
        {
            super(modelIndex, name, width);

            setColumnClass(Long.class);
        }

        @Override
        public Object getColumnValue(Object rowModel)
        {
            return ((NotificationValue)rowModel).getSequence();
        }
    }

    private static class TimeColumn extends RowTableColumn
    {
        TimeColumn(int modelIndex, String name, int width)
        {
            super(modelIndex, name, width);

            setCellRenderer();
        }

        private void setCellRenderer() {
            setCellRenderer(Helper.getDateTimeTableCellRenderer());
        }
        
        @Override
        public Object getColumnValue(Object rowModel)
        {
            return ((NotificationValue)rowModel).getTime();
        }
    }

    private void showPropertiesDialog()
    {
        try
        {
            JMAFrame parent = (JMAFrame)SwingUtilities.getAncestorOfClass(JMAFrame.class, this);
            JMADialog dialog = new NotificationAttributesSheet(parent, getSelectedRowValue().getValue(), m_scope, m_table);
            dialog.setVisible(true);
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                "Failed to show notification properties", e, true);
        }
    }

    private class PropertiesAction extends BasicGuiAction
    {
        public PropertiesAction()
        {
            super("NotificationsWatchWindow.properties");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            showPropertiesDialog();
        }

        @Override
        public boolean isEnabled()
        {
            NotificationValue notificationValue = getSelectedRowValue();

            if (notificationValue != null)
            {
                if (notificationValue.getValue()instanceof MFNotification)
                {
                    return true;
                }
            }
            return false;
        }
    }

    private class RemoveWatchAction extends BasicGuiAction
    {
        public RemoveWatchAction()
        {
            super("NotificationsWatchWindow.remove");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            NotificationsWatchWindow.this.removeWatch(NotificationsWatchWindow.this.getSelectedRowValue());
        }

        @Override
        public boolean isEnabled()
        {
            NotificationValue notificationValue = getSelectedRowValue();

            if (notificationValue != null)
            {
                Notification notification  = notificationValue.getValue();
                String       componentName = notification.getSource().toString();
                WatchModel   dataModel     = (WatchModel)m_data.get(componentName);

                if ((dataModel != null) && dataModel.isSubscriptionInUse(notification.getType()))
                {
                    return true;
                }
            }
            return false;
        }
    }

    private class ClearAction extends BasicGuiAction
    {
        public ClearAction()
        {
            super("NotificationsWatchWindow.clear");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
        }

        @Override
        public boolean isEnabled()
        {
            return (m_table.getRowCount() > 0);
        }
    }

    private class ClearSelectedAction extends BasicGuiAction
    {
        public ClearSelectedAction()
        {
            super("NotificationsWatchWindow.clearSelected");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            NotificationsWatchWindow.this.clearSelectedNotification(NotificationsWatchWindow.this.getSelectedRowValue());
        }

        @Override
        public boolean isEnabled()
        {
            return (getSelectedRowValue() != null);
        }
    }

    private class ClearSelectedTypeAction extends BasicGuiAction
    {
        public ClearSelectedTypeAction()
        {
            super("NotificationsWatchWindow.clearSelectedType");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            NotificationsWatchWindow.this.clearSelectedTypeNotifications(NotificationsWatchWindow.this.getSelectedRowValue());
        }

        @Override
        public boolean isEnabled()
        {
            return (getSelectedRowValue() != null);
        }
    }

    private class ClearAllAction extends BasicGuiAction
    {
        public ClearAllAction()
        {
            super("NotificationsWatchWindow.clearAll");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            NotificationsWatchWindow.this.clearAllNotifications();
        }

        @Override
        public boolean isEnabled()
        {
            return (m_table.getRowCount() > 0);
        }
    }

    private class OptionsAction extends BasicGuiAction
    {
        public OptionsAction()
        {
            super("NotificationsWatchWindow.options");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            try
            {
                JMAFrame parent = (JMAFrame)SwingUtilities.getAncestorOfClass(JMAFrame.class, NotificationsWatchWindow.this);
                WatchPropSheet dialog = new WatchPropSheet(parent);
                dialog.editInstance(null, dialog.new Model(NotificationsWatchWindow.this), false);
                dialog.setVisible(true);
            }
            catch (Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                    "Failed to edit notification watch properties",
                    e, true);
            }
        }
    }

    private class DebugAction extends BasicGuiAction
    {
        public DebugAction()
        {
            super("NotificationsWatchWindow.options");
            putNameValue();
        }
        
        private void putNameValue() {
            putValue(BasicGuiAction.NAME, "Debug");
        }
        
        @Override
        public void actionPerformed(ActionEvent evt)
        {
            NotificationsWatchWindow.this.debug();
        }
    }

    private class WatchModel
    {
        public  AbstractNotificationsModel m_model;
        private Map                        m_subscriptions;

        public WatchModel(AbstractNotificationsModel model)
        {
            m_model = model;
            m_subscriptions = null;
        }

        public String getComponentName()
        {
            return m_model.getComponentName().toString();
        }

        public Map getSubscriptions()
        {
            return (m_subscriptions != null) ? m_subscriptions : Collections.EMPTY_MAP;
        }

        public void addSubscriptions(MBeanNotificationInfo[] info,
                                     NotificationListener    listener)
            throws Exception
        {
            if (m_subscriptions == null)
            {
                m_subscriptions = new HashMap();
            }

             for (int i = info.length - 1; i >= 0; i--)
             {
                 String type = AbstractNotificationsModel.getTypePrefix(info[i].getNotifTypes());

                 if (m_subscriptions.containsKey(type))
                 {
                    continue; // were already subscribed
                }

                 m_subscriptions.put(type, info[i]);

                 MgmtConsole.displayMessage(MgmtConsole.MESSAGE_INFO,
                    "Adding Notification '" + type + "' from " + m_model.getComponentName().toString(),
                    false);

                 m_model.addNotificationListener(type, listener);
             }
        }

        public MBeanNotificationInfo removeSubscription(String               type,
                                                        NotificationListener listener)
            throws Exception
        {
            MBeanNotificationInfo res = (MBeanNotificationInfo)getSubscriptions().remove(type);

            if (getSubscriptions().isEmpty())
            {
                m_subscriptions = null;
            }

            Helper.logDebugMessage("Removing Notification '" + type + "' from " + m_model.getComponentName().toString());

            m_model.removeNotificationListener(type, listener);

            return res;
        }

        public void removeAllSubscriptions(NotificationListener listener)
            throws Exception
        {
            String[] types = (String[])getSubscriptions().keySet().toArray(new String[0]);

            for (int i = 0; i < types.length; i++)
            {
                removeSubscription(types[i], listener);
            }
        }

        public boolean isSubscriptionInUse(String type)
        {
            return (getSubscriptions().get(type) != null);
        }
        public boolean isUsed()
        {
            return !getSubscriptions().isEmpty();
        }
    }

    private class LocalMenuShower extends PopupMenuShower
    {
        LocalMenuShower(JTable table, JPopupMenu popup)
        {
            super(table, popup);
        }

        @Override
        protected void showMenuIfPopupTrigger(MouseEvent evt)
        {
            JPopupMenu popup = getPopupMenu();

            if (popup != null && popup.isPopupTrigger(evt))
            {
                for (int i = 0; i < popup.getComponentCount(); i++)
                {
                    Component item = popup.getComponent(i);

                    if (item instanceof AbstractButton)
                    {
                        Action action = ((AbstractButton) item).getAction();

                        if (action != null)
                        {
                            item.setEnabled(action.isEnabled());
                        }
                    }
                }
            }

            super.showMenuIfPopupTrigger(evt);
        }

        /**
         * Method to detect a double-click on a selected item.
         *
         * @param event The mouse event that caused the action
         */
        @Override
        public void mouseClicked(MouseEvent event)
        {
            if (event.getClickCount() == 2)
            {
                if (event.getComponent()instanceof JTable && getTable().getSelectedRowCount() == 1)
                {
                    showPropertiesDialog();
                }
            }
        }
    }

}