/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui.runtime.propsheets;

import java.util.HashMap;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.InvalidAttributeValueException;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.ObjectName;
import javax.management.ReflectionException;

import com.sonicsw.mf.common.runtime.IIdentity;
import com.sonicsw.mf.jmx.client.IRemoteMBeanServer;
import com.sonicsw.mf.jmx.client.JMSConnectorClient;


public class RuntimeBean
{
    private IRemoteMBeanServer m_server;
    private ObjectName         m_name;
    private IIdentity          m_identity;
    private HashMap            m_attributes = new HashMap();

    public RuntimeBean(IRemoteMBeanServer server, ObjectName name, IIdentity identity)
    {
        m_server   = server;
        m_name     = name;
        m_identity = identity;
    }

    public IRemoteMBeanServer getServer()
    {
        return m_server;
    }

    public ObjectName getName()
    {
        return m_name;
    }

    public IIdentity getIdentity()
    {
        return m_identity;
    }

    public void cacheAllAttributes()
    throws IllegalArgumentException
    {
        m_attributes.clear();

        try
        {
            // First get the MBeanInfo for this bean
            MBeanInfo info = m_server.getMBeanInfo(m_name);

            if(info != null)
            {
                MBeanAttributeInfo[] attributes = info.getAttributes();
                String[] names = new String[attributes.length];

                // Then get the names of all the attributes in the MBean
                for(int i = 0; i < attributes.length; i++)
                {
                    names[i] = attributes[i].getName();
                }

                AttributeList list = m_server.getAttributes(m_name, names);

                for(int i = 0; i < list.size(); i++)
                {
                    Attribute attrib = (Attribute)list.get(i);
                    m_attributes.put(attrib.getName(), attrib.getValue());
                }
            }
        }
        catch (Exception e)
        {
            throw new IllegalArgumentException("Failed to get all runtime attributes", e);
        }
    }

    public Object getCachedAttribute(String attributeName)
    throws IllegalArgumentException
    {
        Object ret = m_attributes.get(attributeName);

        if(ret == null)
        {
            ret = getAttribute(attributeName);
        }

        return ret;
    }

    public Object getAttribute(String attributeName)
    throws IllegalArgumentException
    {
        Object res = null;

        try
        {
            res = m_server.getAttribute(m_name, attributeName);
        }
        catch (MBeanException e)
        {
            throw new IllegalArgumentException("MBean exception getting atribute <" + attributeName + ">", e);
        }
        catch (ReflectionException e)
        {
            throw new IllegalArgumentException("MBean reflection exception getting attribute <" + attributeName + ">", e);
        }
        catch (AttributeNotFoundException e)
        {
            throw new IllegalArgumentException("MBean <" + m_name + "> has no attribute <" + attributeName + ">", e);
        }
        catch (InstanceNotFoundException e)
        {
            throw new IllegalArgumentException("MBean <" + m_name + "> does not exist in the repository", e);
        }
        catch (Exception e)
        {
            if(e instanceof IllegalArgumentException)
            {
                throw (IllegalArgumentException)e;
            }

            IllegalArgumentException th = new IllegalArgumentException("Failed to get attribute <" + attributeName + "> - " + e.getMessage());
            th.initCause(e);
            throw th;
        }
        return res;
    }

    protected void setAttribute(String attributeName, Object newValue)
    throws IllegalArgumentException
    {
        try
        {
            m_server.setAttribute(m_name, new Attribute(attributeName, newValue));

            // If we have this attribute in the cache then we want to update it
            if(m_attributes.containsKey(attributeName))
            {
                m_attributes.put(attributeName, newValue);
            }
        }
        catch (MBeanException e)
        {
            throw new IllegalArgumentException("MBean exception setting attribute <" + attributeName + ">", e);
        }
        catch (InvalidAttributeValueException e)
        {
            throw new IllegalArgumentException("<" + newValue + "> is not a valid value for attribute <" +
                             attributeName + "> in MBean <" + m_name + ">", e);
        }
        catch (ReflectionException e)
        {
            throw new IllegalArgumentException("MBean reflection exception setting attribute <" + attributeName + ">", e);
        }
        catch (AttributeNotFoundException e)
        {
            throw new IllegalArgumentException("MBean <" + m_name + "> has no attribute <" + attributeName + ">", e);
        }
        catch (InstanceNotFoundException e)
        {
          throw new IllegalArgumentException("MBean <" + m_name + "> not found", e);
        }
        catch (Exception e)
        {
            if(e instanceof IllegalArgumentException)
            {
                throw (IllegalArgumentException)e;
            }

            IllegalArgumentException th = new IllegalArgumentException("Failed to set attribute <" + attributeName + "> - " + e.getMessage());
            th.initCause(e);
            throw th;
        }
    }

    public Object invoke(String methodName, Object[] params, String[] signature)
    throws IllegalArgumentException
    {
        Object res = null;
        ClassLoader loader = this.getClass().getClassLoader();

        try
        {
            res = ((JMSConnectorClient)m_server).invoke(m_name, methodName, params, signature, loader);
        }
        catch (Exception e)
        {
            throw new IllegalArgumentException("Failed to invoke method <" + methodName + "> - " + e.getMessage(), e);
        }
        return res;
    }

    @Override
    public String toString()
    {
        return m_server.toString() + "; " + m_name + "; " + m_identity;
    }

}
