
package com.sonicsw.ma.gui.runtime.propsheets;

import java.beans.PropertyVetoException;

import modelobjects.framework.model.ModelAspectAdapter;
import modelobjects.framework.model.ModelAspectId;
import modelobjects.framework.model.ModelObjectAdapter;

/**
 * <p>Title: AttributeMapAspectAdapter</p>
 * <p>Description: The AttributeMapAspectAdapter is a ModelAspectAdapter that
 * uses the IConfigPath to find an attribute in an IAttributeMap-based data
 * model and allows set and get operations depending on whether the adapter
 * was created as read-only. </p>
 * <p>Copyright: Copyright (c) 2002</p>
 * <p>Company: Sonic Software</p>
 * @author Huw Jones
 * @version 1.0
 */
public class RuntimeModelAspectAdapter extends ModelAspectAdapter
{
    private String  m_name;
    private boolean m_readOnly;

    /**
     * Creates an adapter which can get and set an attribute of class "type"
     * by using the IConfigPath "path" to find the attribute.
     *
     * Adapter assumes underlying model is that of an IAttributeMap.
     *
     * @param path      The IConfigPath to the attribute.
     * @param type      The type of the attribute found by the path.
     * @param readOnly  Indicates whether this attribute can be set.
     */
    public RuntimeModelAspectAdapter(String  name,
                                     Class   type,
                                     boolean readOnly)
    {
       super(ModelAspectId.forName(name), type);

       m_name     = name;
       m_readOnly = readOnly;
    }

    /**
     * Return whether the aspect is a read-only aspect which cannot be
     * assigned a new value, ie. set method fails.
     *
     * @return  True if the aspect is read-only, otherwise false.
     */
    @Override
    public boolean isReadonlyAspect()
    {
        return m_readOnly;
    }

    /**
     * Get the aspect value from the model.
     *
     * @param modelObject
     * @param modelObjectAdapter
     * @return
     */
    @Override
    protected Object getAspectValue(Object             modelObject,
                                    ModelObjectAdapter modelObjectAdapter)
    {
        // Use the cached values from the runtime bean
        return ((RuntimeBean)modelObject).getCachedAttribute(m_name);
    }

    /**
     * Set the aspect value on the model.
     *
     * @param modelObject
     * @param newAspectValue
     * @param modelObjectAdapter
     * @throws PropertyVetoException
     * @throws IllegalArgumentException
     * @throws UnsupportedOperationException
     */
    @Override
    protected void setAspectValue(Object             modelObject,
                                  Object             newAspectValue,
                                  ModelObjectAdapter modelObjectAdapter)
    throws PropertyVetoException,
           IllegalArgumentException,
           UnsupportedOperationException
    {
        if (isReadonlyAspect())
        {
            throw new UnsupportedOperationException("ReadOnly!");
        }

        ((RuntimeBean)modelObject).setAttribute(m_name, newAspectValue);
    }

}
