/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui.table;

import java.awt.BorderLayout;

import javax.swing.AbstractButton;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.table.TableColumn;
import javax.swing.table.TableModel;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.util.BasicAction;
import com.sonicsw.ma.gui.util.ExtendedJScrollPane;
import com.sonicsw.ma.gui.util.FilterTableModel;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.JRowTable;
import com.sonicsw.ma.gui.util.JWaitCursor;
import com.sonicsw.ma.gui.util.PluginMouseClickListener;
import com.sonicsw.ma.plugin.IPlugin;

/**
 * This class is an abstract table implementation of the content pane
 */
public abstract class AbstractTableContentPane extends    JPanel
                                               implements ITableContentPane
{
    protected   ExtendedJScrollPane m_scroller;
    protected JRowTable   m_table;
    private   IPlugin     m_plugin;
    private   Object      m_refreshSync = new Object();

    public AbstractTableContentPane(IPlugin plugin)
        throws IllegalArgumentException
    {
        if (plugin == null)
        {
            throw new IllegalArgumentException("Content Pane must have a plugin (plugin = null)");
        }

        setPluginProperty(plugin);

        initializaUI();

        // ContentPane can now be populated via a call to setTableData. No longer done
        // in this constructor.
    }
    
    private void setPluginProperty(IPlugin plugin) {
        setPlugin(plugin);
    }
    
    private void initializaUI() {
        initUI();
    }
    /**
     * activateFilter
     * @param bActivateFilter defines if the filter is active
     */
    public void activateFilter(boolean bActivateFilter)
    {
        TableModel model = m_table.getModel();

        if (model instanceof FilterTableModel)
        {
            ((FilterTableModel)model).activateFilter(bActivateFilter);
        }
    }

    /**
     * isFilterActivated
     * @return
     */
    public boolean isFilterActivated()
    {
        boolean bIsFilterActivated = false;
        TableModel model = m_table.getModel();

        if (model instanceof FilterTableModel)
        {
            bIsFilterActivated = ((FilterTableModel)model).isFilterActivated();
        }

        return bIsFilterActivated;
    }

    /**
     * filterData
     * @return
     */
    public void filterData(boolean bFilter)
    {
        TableModel model = m_table.getModel();

        if (model instanceof FilterTableModel)
        {
            ((FilterTableModel)model).filterData(bFilter);
        }
    }

    @Override
    public IPlugin getPlugin()
    {
        return m_plugin;
    }

    protected void setPlugin(IPlugin plugin)
    {
        m_plugin = plugin;
    }

    //-------------------------------------------------------------------------
    //
    // Selection support
    //
    //-------------------------------------------------------------------------

    @Override
    public Object[] getSelectedItems()
    {
        int[]            sel   = m_table.getSelectedRows();
        Object[]         res   = new Object[sel.length];
        IModelTableModel model = (IModelTableModel)m_table.getModel();

        for (int i = 0; i < res.length; i++)
        {
            res[i] = model.getRowModel(sel[i]);
        }

        return res;
    }

    @Override
    public Object getSelectedItem()
    {
        Object[] res = getSelectedItems();

        return ((res != null) && (res.length > 0)) ? res[0] : null;
    }

    //-------------------------------------------------------------------------

    @Override
    public void addRowModel(Object object)
    {
        ((IModelTableModel)m_table.getModel()).addRow(object);
    }

    @Override
    public void removeRowModel(Object object)
    {
        ((IModelTableModel)m_table.getModel()).delete(object);
    }

    @Override
    public void refreshRowModel(Object object)
    {
        IModelTableModel model = (IModelTableModel)m_table.getModel();

        int contentIndex = model.getContents().indexOf(object);
        int modelIndex   = model.getIndexOf(object);

        if(contentIndex >= 0)
        {
            if(contentIndex != modelIndex)
            {
                model.deleteRow(contentIndex);
                model.addRow(object);
            }
            else
            {
                model.rowChanged(modelIndex);
            }
        }
    }

    @Override
    public void refresh()
    {
        Helper.logDebugMessage("refresh content pane - " + getClass().getName());

        // Using a regular thread rather than a SwingUtilities.invokeLater to
        // improve perceived user performance...in the case where the Event
        // Dispatch Thread is blocked/busy...then this approach still updates
        // the table.
        //
        // This is not a good approach because now Swing component (table)
        // updates are being made off of the main Event Dispatch Thread.
        //
        Thread t = new Thread(new Runnable()
        {
            @Override
            public void run()
            {
                // Sonic00019790 - refresh can get called multiple times and
                // because the table update is threaded we can get into a state
                // where multiple threads are trying to modify the same table
                // at the same time - not good! So we synchronize on an object
                // before doing the actual table work...
                //
                synchronized (m_refreshSync)
                {

                JWaitCursor wc = new JWaitCursor(MgmtConsole.getMgmtConsole());
                MgmtConsole.getMgmtConsole().getStatusBar().setPersistentMessage("Adding elements to view...");

                try
                {
                    setTableData(getPlugin());
                }
                catch (Exception e)
                {
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR, "Failed to refresh content pane", e, false);
                }
                finally
                {
                    MgmtConsole.getMgmtConsole().getStatusBar().showStatusMessage(null);
                    wc.release();
                }

                } // synchronized
            }
        });

        t.start();
    }

    //-------------------------------------------------------------------------

    /**
     * Subclass should override
     */
    @Override
    public AbstractButton[] getToolbarItems()
    {
        return null;
    }

    /**
     * Subclass should override
     */
    @Override
    public JComponent[] getMenuItems(int type)
    {
        return null;
    }

    @Override
    public BasicAction getDefaultAction()
    {
        Object sel = getSelectedItem();

        if ((sel == null) || !(sel instanceof IPlugin))
        {
            return null;
        }

        return ((IPlugin)sel).getDefaultAction();
    }

    protected void initUI()
    {
        setLayout(new BorderLayout());

        m_table = new JRowTable(new FilterTableModel(getColumns()), getClass().getName());

        m_scroller = new ExtendedJScrollPane();
        add(m_scroller);

        // Setting the table into the scrollpane's viewport has to be done
        // before adding the menu shower because the shower checks the table's
        // parenting and will add an additional mouse listener for the viewport
        //
        m_scroller.setViewportView(m_table);

        // The default row height is 16 pixels....may of our icons are 16 pixels
        // in height so to avoid everything looking squashed we add 2 pixels
        // here to give them a bit more space.
        //
        m_table.setRowHeight(m_table.getRowHeight() + 2);

        // Make the table's mouse events go to the menu shower...
        //
        m_table.addMouseListener(new PluginMouseClickListener(this));
    }

    @Override
    public final JRowTable getTable()
    {
        return m_table;
    }

    @Override
    public JComponent getComponent()
    {
        return this;
    }

    public JScrollPane getScroller()
    {
        return m_scroller;
    }

    /**
     * Get the array of TableColumns to use for the table
     */
    @Override
    public abstract TableColumn[] getColumns();

    @Override
    public abstract void setTableData(IPlugin plugin);
}