/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui.table;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.swing.JProgressBar;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.plugin.AbstractGUIPlugin;
import com.sonicsw.ma.plugin.ConfigBeanModel;
import com.sonicsw.ma.plugin.IPlugin;
import com.sonicsw.mx.config.IAttributeList;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigElement;
import com.sonicsw.mx.config.IConfigServer;

import com.sonicsw.mf.common.config.query.FromElementList;
import com.sonicsw.mf.common.config.query.Query;

/**
 * <p>Title: ConfigQueryTableContentPane</p>
 * <p>Description: A specialized config table content pane that builds its
 *                 row model by performing a DS query.</p>
 * <p>Copyright: Copyright (c) 2004</p>
 * <p>Company: Sonic Software</p>
 * @author hjones
 * @version 1.0
 */
public abstract class ConfigQueryTableContentPane extends ConfigBeanTableContentPane
{
    private static final int QUERY_SIZE = 500;

    private boolean m_enforceKey = false;
    private boolean m_excludeConfigElement = true;
    private Set     m_excludeTypes;

    /**
     * Construct a table content pane that builds its row model using a DS
     * query. No type filtering is performed.
     *
     * @param plugin  The parent plugin that hosts the data model.
     */
    public ConfigQueryTableContentPane(IPlugin plugin)
    {
        this(plugin, new HashSet());
    }

    /**
     * Construct a table content pane that builds its row model using a DS
     * query. By supplying a set of excludable types, data filtering can
     * be achieved.
     *
     * @param plugin        The parent plugin that hosts the data model.
     * @param excludeTypes  A set of types (strings) to exclude if the query
     *                      returns Config Beans (typed Config Elements) that
     *                      match.
     */
    public ConfigQueryTableContentPane(IPlugin plugin, Set excludeTypes)
    {
        super(plugin);

        m_excludeTypes = excludeTypes;
    }

    /**
     * Construct a table content pane that builds its row model using a DS
     * query. By supplying an array of excludable types, data filtering can
     * be achieved.
     *
     * @param plugin        The parent plugin that hosts the data model.
     * @param excludeTypes  An array of types to exclude if the query returns
     *                      Config Beans (typed Config Elements) that match.
     */
    public ConfigQueryTableContentPane(IPlugin plugin, String[] excludeTypes)
    {
        this(plugin, new HashSet(Arrays.asList(excludeTypes)));
    }

    /**
     * Controls whether or not basic (un-typed) Config Elements are allowed
     * in the table model.
     *
     * Note: that this feature is turned ON by default, i.e. Config Elements are
     *       NOT added to the table.
     *
     * @param excludeConfigElement  A boolean value indicating whether or not
     *                              to exclude Config Elements from the table.
     */
    public void setExcludeConfigElement(boolean excludeConfigElement)
    {
        m_excludeConfigElement = excludeConfigElement;
    }

    /**
     * Indicates whether or not Config Elements are excluded from the table.
     *
     * @return  Returns true if Config Elements are excluded, otherwise false.
     */
    public boolean isExcludeConfigElement()
    {
        return m_excludeConfigElement;
    }

    /**
     * When key enforcement is active each element that is returned by the
     * query is matched up with the reference in the parent's bean model.
     * This incurrs a considerable performance hit (not as bad though as not
     * doing the query) but may be required by some implementations if they
     * need this key to perform delete's, etc.
     *
     * Note that this feature is turned OFF by default.
     *
     * @param enforceKey
     */
    public void setEnforceKey(boolean enforceKey)
    {
        m_enforceKey = enforceKey;
    }

    /**
     * Indicates whether or not the Enforce Key feature is active.
     *
     * @return  If active then returns true, otherwise false (inactive).
     */
    public boolean isEnforceKey()
    {
        return m_enforceKey;
    }

    /**
     * Method fetches the table data from its plugin source and sets it into
     * the table model and ultimately, the table's view.
     *
     * This class optimizes the building of the table model by performing a
     * query on the DS rather than iterating through each map or list attribute.
     *
     * @param plugin  The parent plugin for which the content pane is being
     *                built.
     */
    @Override
    public void setTableData(IPlugin plugin)
    {
        JProgressBar    bar       = MgmtConsole.getMgmtConsole().getStatusBar().getProgressBar();
        IConfigServer   server    = getPlugin().getPluginContext().getConfigContext().getConfigServer();
        ConfigBeanModel beanModel = (ConfigBeanModel)getRowModel(plugin);
        Query           query     = new Query();

        if (bar != null)
        {
            bar.setIndeterminate(true);
            bar.setVisible(true);
        }

        try
        {
            ArrayList newModel = new ArrayList();

            List ceList = new ArrayList(server.listConfigElements(getQueryFolder()));
            int  ceSize = ceList.size();

            int iStart = 0;

            while (iStart < ceSize)
            {
                String[] ceFetch = getStringArray(ceList, iStart, QUERY_SIZE);
                query.setFrom(new FromElementList(ceFetch));

                Set      set = server.loadConfigElements(query);
                Iterator i   = set.iterator();

                iStart += ceFetch.length;

                while (i.hasNext())
                {
                    IConfigElement element = (IConfigElement)i.next();

                    if (element instanceof IConfigBean)
                    {
                        String elementType = ((IConfigBean)element).getConfigType().getName();

                        if (m_excludeTypes.contains(elementType))
                        {
                            continue;
                        }
                    }
                    else
                    if (isExcludeConfigElement())
                    {
                        continue;
                    }

                    String key = null;

                    if (isEnforceKey())
                    {
                        // We've got to find the bean in the plugin model...otherwise
                        // the element shouldn't be displayed (because its not in
                        // the map or list.
                        //
                        key = getKey(beanModel, element);
                        if (key == null)
                        {
                            continue;
                        }
                    }

                    ConfigBeanModel rowModel = new ConfigBeanModel(element, beanModel.getType(), key);

                    if (canAddRowToModel(rowModel))
                    {
                        newModel.add(rowModel);
                    }
                }
            }

            IModelTableModel tableModel = (IModelTableModel)m_table.getModel();
            tableModel.setContents(newModel);
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                                      "Failed to load table row data", e, false);
        }
        finally
        {
            if (bar != null)
            {
	            bar.setIndeterminate(false);
	            bar.setVisible(false);
            }
            MgmtConsole.getMgmtConsole().getStatusBar().releaseProgressBar();
        }
    }

    private String[] getStringArray(List list, int start, int size)
    {
        String[] res = new String[Math.min(Math.max(list.size() - start, 0), size)];

        int iEnd   = Math.min(start + size, list.size());
        int iStart = Math.min(start, iEnd);

        for (int i = iStart; i < iEnd; i++)
        {
            res[i - iStart] = (String)list.get(i);
        }

        return res;
    }

    /**
     * The elements for this content pane were loaded using a query and might
     * not 'actually' be part of the parent model's map or list. So we use
     * this method to match an element loaded by the query to an actual
     * attribute reference made by the parent model.
     *
     * If the element is found then its key is returned.
     *
     * @param model    The parent model in which we hope contains a reference
     *                 attribute to the element.
     * @param element  The element for which we are searching for a key.
     * @return         The string key (unique name for map, unique number for
     *                 list) that indicates that the element was found,
     *                 otherwise null if the element was not found.
     */
    protected String getKey(ConfigBeanModel model, IConfigElement element)
    {
        if (model != null)
        {
            if (model.getData() instanceof IAttributeMap)
            {
                IAttributeMap beanMap = model.getData();
                Iterator      i       = beanMap.keySet().iterator();

                while (i.hasNext())
                {
                    String        name = (String)i.next();
                    IAttributeMap map  = (IAttributeMap)beanMap.getAttribute(name);

                    if (map == element)
                    {
                        return name;
                    }
                }
            }
            else
            if (model.getDataList() instanceof IAttributeList)
            {
                IAttributeList beanList = model.getDataList();
                Iterator       i        = beanList.iterator();
                int            count    = 0;

                while (i.hasNext())
                {
                    Object listItem = i.next();
                    count++;

                    if (listItem == element)
                    {
                        return Integer.toString(count);
                    }
                }
            }
        }

        return null;
    }

    /**
     * Default operation returns the parent plugin's internal path representation.
     *
     * Should be overridden by subclasses to target specific logical folder.
     *
     * @return  The logical folder under which the query is to be executed.
     */
    protected String getQueryFolder()
    {
        return ((AbstractGUIPlugin)getPlugin())._getPluginPath();
    }

}
