package com.sonicsw.ma.gui.table;

import java.awt.Component;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.ArrayList;
import java.util.List;

import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;

import modelobjects.framework.EditRule;
import modelobjects.framework.ModelEditMediator;
import modelobjects.framework.ViewAspectAdapter;
import modelobjects.framework.model.ModelAspectId;
import modelobjects.util.ObjectCopier;

import com.sonicsw.ma.gui.util.ExtendedJTable;

/**
 *  JRowTableAspectAdapter is a special kind of ViewAspectAdapter that
 *  is used for List-valued model-aspects that are edited in JTables.
 *  This class should be used only with care, since it must make a complete
 *  copy of the model-aspect List and of the contents of the List in order
 *  to be sure that edits made in the table (inserts, deletes, cell-edits)
 *  are both discardable in the case of reset, and undoable and redoable.
 *
 *  Note that JRowTableAspectAdapter can only be used with JRowTable's that
 *  use the IModelTableModel for their TableModels.
 *
 *  JRowTableAspectAdapter supports PropertyChangeListeners that may want
 *  to track when it is editable in order to enable/disable editing actions.
 */
public class JRowTableAspectAdapter extends    ViewAspectAdapter
                                    implements TableModelListener
{
    public static final String EDITABLE_PROPERTY = "editable";

    protected ExtendedJTable      table;
    protected ObjectCopier        rowModelCopier;
    private boolean               editable;
    private PropertyChangeSupport propertyChangeSupport;

    public JRowTableAspectAdapter(ModelAspectId     modelAspectId,
                                  ExtendedJTable    table,
                                  ObjectCopier      rowModelCopier,
                                  EditRule          editRule,
                                  ModelEditMediator modelEditMediator)
    {
        super(modelAspectId, editRule, null, modelEditMediator);
        table.setName(modelAspectId.toString());
        this.table          = table;
        this.rowModelCopier = rowModelCopier;

        table.getModel().addTableModelListener(this);
        propertyChangeSupport = new PropertyChangeSupport(this);
    }

    @Override
    public Component getViewComponent()
    {
        return(table);
    }

    @Override
    public boolean isEditable()
    {
        return(editable);
    }

    @Override
    public void setEditable(boolean editable)
    {
        boolean wasEditable = this.editable;
        this.editable = editable;
        if (editable != wasEditable)
        {
            propertyChangeSupport.firePropertyChange(EDITABLE_PROPERTY,
                                                     wasEditable, editable);
        }
    }

    @Override
    public Object getViewAspectValue()
    {
        return ((IModelTableModel)table.getModel()).getContents();
    }

    @Override
    public String getViewAspectStringValue()
    {
        throw(new IllegalArgumentException("String values not supported"));
    }

    @Override
    protected void setViewAspectValue(Object viewAspectValue)
        throws IllegalArgumentException
    {
        List elems = (List)viewAspectValue;
        int numElems = ((elems == null) ? 0 : elems.size());

        ArrayList contents = new ArrayList(numElems);

        for (int i = 0; i < numElems; i++)
        {
            contents.add(rowModelCopier.copyObject(elems.get(i)));
        }

        ((IModelTableModel)table.getModel()).setContents(contents);
    }

    @Override
    protected void setViewAspectStringValue(String viewStringValue)
        throws IllegalArgumentException
    {
        throw(new IllegalArgumentException("String values not supported"));
    }

    @Override
    public void tableChanged(TableModelEvent evt)
    {
        // ignore TableStructureChange event - data hasn't changed
        if ((evt.getFirstRow() == TableModelEvent.HEADER_ROW) &&
            (evt.getLastRow() == TableModelEvent.HEADER_ROW) &&
            (evt.getColumn() == TableModelEvent.ALL_COLUMNS))
        {
            return;
        }

        fireChangeEvent();
    }

    public void addPropertyChangeListener(PropertyChangeListener listener)
    {
        propertyChangeSupport.addPropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener)
    {
        propertyChangeSupport.removePropertyChangeListener(listener);
    }
}
