package com.sonicsw.ma.gui.table;

import java.text.Collator;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.TableColumnModelEvent;
import javax.swing.event.TableColumnModelListener;
import javax.swing.event.TableModelEvent;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;

/**
 *  ModelListTableModel is the central class of a framework for building
 *  highly flexible, model-object-oriented TableModels for JTables.
 *  The contents of a ModelListTableModel is a List of Model Objects mapped
 *  one to each row of the table.
 */
public class ModelListTableModel
    extends AbstractTableModel
    implements IModelTableModel, TableColumnModelListener
{
    private static final boolean DEBUG = false;

    protected int m_columnCount;
    protected TableColumn[] m_tableColumns;
    protected List m_listView;

    // sorting support
    protected Comparator sortComparator;
    protected int sortColumn;
    protected boolean sortedAscending = true;

    // provide minimal caching of cell values
    protected int lastRow = -1;
    protected int lastCol = -1;
    protected Object lastVal = null;
    protected boolean m_bSortIsActive = false;

    public ModelListTableModel(TableColumn[] tableColumns)
    {
        this(tableColumns, new ArrayList());
    }

    public ModelListTableModel(TableColumn[] tableColumns, List contents)
    {
        this(tableColumns, new ArrayList(), true);
    }

    public ModelListTableModel(TableColumn[] tableColumns, List contents, boolean bSortIsActive)
    {
        m_tableColumns = tableColumns;
        m_columnCount = tableColumns.length;
        m_bSortIsActive = bSortIsActive;

        setContents(contents);
    }

    @Override
    public void activateSort(boolean bSortIsActive)
    {
        m_bSortIsActive = bSortIsActive;
    }

    /**
     * isSortActivated
     * @return
     */
    @Override
    public boolean isSortActivated()
    {
        return m_bSortIsActive;
    }

    /**
     *  Return the model List contents of this ModelListTableModel.
     */
    @Override
    public List getContents()
    {
        return m_listView;
    }

    /**
     *  Assign the model List contents of this ModelListTableModel.
     */
    @Override
    public final void setContents(List contents)
    {
        m_listView = contents;

        invalidateCellCache();

        sortByColumn(getSortColumn(), isSortedInAscendingOrder());

        fireTableStructureChanged();
    }

    /**
     *  Return this ModelListTableModel's array of TableColumnAdapters.
     */
    @Override
    public TableColumn[] getColumns()
    {
        return m_tableColumns;
    }

    /**
     *  Return the TableColumnAdapter for the specified model column index.
     */
    @Override
    public TableColumn getColumn(int modelColumnIndex)
    {
        return m_tableColumns[modelColumnIndex];
    }

    /**
     *  Override to invalidate cell cache on every change.
     */
    @Override
    public void fireTableChanged(TableModelEvent event)
    {
        invalidateCellCache();

        super.fireTableChanged(event);
    }

    /**
     *  Return the number of row model objects in this ModelListTableModel.
     */
    @Override
    public int getRowCount()
    {
        return m_listView.size();
    }

    /**
     *  Return the number of columns in this ModelListTableModel.
     */
    @Override
    public int getColumnCount()
    {
        // JSP: Keep an eye on for impact.
        return m_tableColumns.length;
    }

    @Override
    public String getColumnName(int columnIndex)
    {
        Object value = getColumn(columnIndex).getHeaderValue();

        return (value != null) ? value.toString() : "";
    }

    /**
     *  Return the row model object at the specified index of this
     *  ModelListTableModel.
     */
    @Override
    public Object getRowModel(int row)
    {
        return ((row >= 0) && (row < m_listView.size())) ? m_listView.get(row) : null;
    }

    /**
     *  Return whether the cell at the specified row and column is editable.
     */
    @Override
    public boolean isCellEditable(int row, int column)
    {
        return ((com.sonicsw.ma.gui.table.RowTableColumn) getColumn(column)).isEditable();
    }

    /**
     * Returns the row index of the first match in the contents List if
     * the value is present.
     *
     * @param  startRow     the row to start the search
     * @param  columnIndex  the column inwhich to search
     * @param  value        the value to find
     * @return the row index of the first match or -1 if no match is found
     */
    @Override
    public int findCell(int startRow, int columnIndex, Object value)
    {
        for (int rowIndex = startRow; rowIndex < getRowCount(); rowIndex++)
        {
            Object cellValue = getValueAt(rowIndex, columnIndex);

            if ((cellValue == null) && (value == null))
            {
                return rowIndex;
            }

            if ((cellValue != null) && (value != null) && value.equals(cellValue))
            {
                return rowIndex;
            }
        }

        return -1;
    }

    /**
     * Hacked version of the java.util.Arrays binarySearch method that tries
     * to find an exact match on the keyed object.
     */
    private int binarySearch(Object[] a, Object key, Comparator c)
    {
        int mid = Arrays.binarySearch(a, key, c);

        if (mid < 0)
        {
            return mid;
        }

        // Double check to make sure we really found the right object,
        // and not just another that equivalent by the comparator, since
        // some comparators might treat many objects as equivalent.\

        if (a[mid].equals(key))
        {
            return mid;
        }

        // scan backward
        for (int i = mid - 1; i >= 0; i--)
        {
            if (c.compare(key, a[i]) != 0)
            {
                break;
            }

            if (key.equals(a[i]))
            {
                return i;
            }
        }

        // scan forward
        for (int i = mid + 1; i < a.length; i++)
        {
            if (c.compare(key, a[i]) != 0)
            {
                break;
            }

            if (key.equals(a[i]))
            {
                return i;
            }
        }

        // Ok, so we couldn't find a direct match but we did get a match
        // on the comparator so the mid-point should be the insertion point.
        return -(mid + 1);
    }

    /**
     *  Return the index of the specified object in the contents List if it
     *  is present, or the index at which to insert the object otherwise,
     *  encoded as (-1 - insertionIndex).
     *  If the contents are sorted, a binary search is used.  Otherwise a
     *  linear search is used and if the object is not present the insertion
     *  index is at the end of the list.
     */
    @Override
    public int getIndexOf(Object obj)
    {
        if (isSortActivated() == false || sortComparator == null)
        {
            return m_listView.indexOf(obj);
        }

        return binarySearch(m_listView.toArray(), obj, sortComparator);
    }

    /**
     *  Insert the specified row model object into this ModelListTableModel
     *  at the specified row index.
     */
    @Override
    public void insertRow(Object rowModel, int index)
    {
        debugPrintln("insertRow " + index);

        m_listView.add(index, rowModel);
        fireTableRowsInserted(index, index);
        recheckSortedOrder(index);
    }

    @Override
    public int addRow(Object rowModel)
    {
        int insertIndex = getRowCount();

        if (!isSortActivated())
        {
            insertRow(rowModel, insertIndex);
        }
        else
        {
            int index = getIndexOf(rowModel);

            if (index < 0)
            {
                // convert "not-present" index into real index
                //
                index = -1 - index;

                if (index < insertIndex)
                {
                    insertIndex = index;
                }

                m_listView.add(insertIndex, rowModel);

                fireTableRowsInserted(insertIndex, insertIndex);
            }
            else
            {
                // The row model already exists in the table model so don't
                // insert it again...just refresh the row.
                //
                insertIndex = -1;

                fireTableRowsUpdated(index, index);
            }
        }

        debugPrintln("addRow " + insertIndex);

        return insertIndex;
    }

    /**
     *  Insert the specified row model object into this ModelListTableModel
     *  maintaining sorted order if the model is currently sorted, or at the
     *  end otherwise.  It is illegal to insert a row model object that is
     *  already present.
     *
     * @deprecated  As of 2.1.1 release should use addRow instead.
     */
    @Override
    public void insertRowMaintainSort(Object rowModel)
    {
        addRow(rowModel);
    }

    /**
     *  Delete the row at the specified index.
     */
    @Override
    public void deleteRow(int index)
    {
        m_listView.remove(index);
        fireTableRowsDeleted(index, index);
    }

    /**
     *  Delete the specified object from this ModelListTableModel if it is
     *  present.
     */
    @Override
    public void delete(Object obj)
    {
        int row = getIndexOf(obj);
        if (row >= 0)
        {
            deleteRow(row);
        }
    }

    protected void debugPrintln(String str)
    {
        if (DEBUG)
        {
            System.err.println("## " + hashCode() + " > " + str);
        }
    }

    /**
     *  Clear the contents of this ModelListTableModel.
     */
    @Override
    public void clear()
    {
        if (getRowCount() == 0)
        {
            return;
        }

        debugPrintln("clear");

        m_listView.clear();

        fireTableStructureChanged();
    }

    /**
     *  Notify TableModelListeners that the specified row has changed.
     */
    @Override
    public void rowChanged(int index)
    {
        fireTableRowsUpdated(index, index);
        recheckSortedOrder(index);
    }

    /**
     *  Return the value of the cell at the specified row and column.
     */
    @Override
    public Object getValueAt(int row, int column)
    {
        if ((row == lastRow) && (column == lastCol))
        {
            return lastVal;
        }

        Object value = ((com.sonicsw.ma.gui.table.RowTableColumn) m_tableColumns[column]).getColumnValue(getRowModel(row));

        // save last computed value since this method gets call a lot
        lastRow = row;
        lastCol = column;
        lastVal = value;

        return value;
    }

    /**
     *  Assign a new value to the cell at the specified row and column.
     */
    @Override
    public void setValueAt(Object value, int row, int column)
    {
        invalidateCellCache();
        ((com.sonicsw.ma.gui.table.RowTableColumn) m_tableColumns[column]).setColumnValue(getRowModel(row), value);

        recheckSortedOrder(row);
    }

    /**
     *  Return the TableCellRenderer for the specified column.
     */
    public TableCellRenderer getColumnCellRenderer(int columnIndex)
    {
        return m_tableColumns[columnIndex].getCellRenderer();
    }

    /**
     *  Assign the TableCellRenderer for the specified column.
     */
    public void setColumnCellRenderer(int columnIndex,
                                      TableCellRenderer cellRenderer)
    {
        m_tableColumns[columnIndex].setCellRenderer(cellRenderer);
    }

    /**
     *  Return the TableCellEditor for the specified column.
     */
    public TableCellEditor getColumnCellEditor(int columnIndex)
    {
        return m_tableColumns[columnIndex].getCellEditor();
    }

    /**
     *  Assign the TableCellEditor for the specified column.
     */
    public void setColumnCellEditor(int columnIndex, TableCellEditor cellEditor)
    {
        m_tableColumns[columnIndex].setCellEditor(cellEditor);
    }

    /**
     *  Return the model Class of the the specified column in this
     *  ModelListTableModel.
     */
    @Override
    public Class getColumnClass(int columnIndex)
    {
        return ((com.sonicsw.ma.gui.table.RowTableColumn) m_tableColumns[columnIndex]).getColumnClass();
    }

    /**
     *  Create the TableColumnModel for this ModelListTableModel based on the
     *  array of TableColumnAdapters.
     */
    /*
      private DefaultTableColumnModel makeColumnModel()
      {
        DefaultTableColumnModel tcm = new DefaultTableColumnModel();
        for (int i = 0; i < m_columnAdapters.length; i++)
        {
          TableColumnAdapter columnAdapter = m_columnAdapters[i];
          TableColumn        column        = columnAdapter.makeModelPropertyTableColumn(i);
          column.setHeaderRenderer(new HeaderCellRenderer(this, columnAdapter));
          tcm.addColumn(column);
        }
        return tcm;
      }
     */

    /**
     *  Invalidate the unit cell cache used for improving the performance of
     *  the getValueAt(int,int) method.
     */
    private void invalidateCellCache()
    {
        lastRow = -1;
        lastCol = -1;
        lastVal = null;
    }

    /////////////////////////////////////////////////////////////////////////////
    ///
    /// sorting support
    ///
    /////////////////////////////////////////////////////////////////////////////

    /**
     *  Sort this ModelListTableModel by the specified model column.  If the
     *  table is not currently sorted by that column, the column's normal
     *  Comparator is used to perform the sort.  Otherwise, the column's
     *  reverse comparator is used.
     */
    @Override
    public void sortByColumn(int modelColumnIndex)
    {
        if (isSortActivated())
        {
            sortByColumn(modelColumnIndex, ((sortColumn != modelColumnIndex) || !sortedAscending));
        }
    }

    /**
     *  Sort this ModelListTableModel by the specified model column, specifying
     *  whether to sort in ascending or descending order.
     *  <p>
     *  The ModelListTableModel is sorted by extracting the complete contents
     *  List into an object array, sorting the array using the Comparator,
     *  and then replacing the contents of the list with the sorted array.
     */
    @Override
    public void sortByColumn(int modelColumnIndex, boolean ascending)
    {
        if (isSortActivated())
        {
            debugPrintln("sortByColumn " + modelColumnIndex + " " + (String)(ascending ? "ASCENDING" : "DESCENDING"));

            //IL(can cause array out of bound exception if column index is -1)
            if (modelColumnIndex == -1)
            {
                return;
            }

            RowTableColumn column = (RowTableColumn) m_tableColumns[modelColumnIndex];

            if (!column.isSortable())
            {
                return;
            }

            /*
                  System.out.println("sorting by model column " + modelColumnIndex);
                  System.out.println("  column name = " + columnAdapter.getHeaderLabel());
                  System.out.println("  ascending = " + ascending);
             */

            Comparator comparator = getComparator(column, ascending);

            Object[] modelArray = m_listView.toArray();
            Arrays.sort(modelArray, comparator);

            // clear the contents and re-load it from the sorted array
            m_listView.clear();
            for (int i = 0, n = modelArray.length; i < n; i++)
            {
                m_listView.add(i, modelArray[i]);
            }

            setSortedByColumn(modelColumnIndex, ascending);

            fireTableDataChanged();
        }
    }

    protected Comparator getComparator(RowTableColumn column, boolean ascending){
         return new DefaultComparator(column, ascending);
    }
    /**
     *  Assert that the table is sorted by the specified column's Comparator,
     *  and indicate whether it's sorted ascending or not.  If the sortColumn
     *  and/or sortComparator are changed by this method, PropertyChangeEvents
     *  are fired to notify listeners about the changes.
     */
    @Override
    public void setSortedByColumn(int modelColumnIndex, boolean ascending)
    {
        if (isSortActivated())
        {
            if (modelColumnIndex >= m_tableColumns.length)
            {
                // if the columnIndex is invalid, sort on the first column...
                modelColumnIndex = 0;
            }

            Comparator oldComparator = sortComparator;
            int oldSortColumn = sortColumn;

            this.sortColumn = modelColumnIndex;
            this.sortedAscending = ascending;

            if (modelColumnIndex == -1)
            {
                this.sortComparator = null;
            }
            else {
                this.sortComparator = new DefaultComparator((RowTableColumn) m_tableColumns[modelColumnIndex], ascending);
            //((RowTableColumn)m_tableColumns[modelColumnIndex]).getRowComparator(ascending);
            }

            // The table needs to re-paint its header because the sort column has changed
            // or the direction of sort has changed.
            //
            if ((oldSortColumn != sortColumn) || (oldComparator != sortComparator))
            {
                fireTableStructureChanged();
            }
        }
    }

    /**
     *  Return whether this ModelListTableModel is current sorted.
     */
    @Override
    public boolean isSorted()
    {
        return (sortComparator != null);
    }

    /**
     *  Check to make sure the specified row is still in sorted order with
     *  respect to its previous and next rows.  If it's not, mark the table
     *  as being unsorted.
     */
    void recheckSortedOrder(int rowNumber)
    {
        if (isSortActivated() && (sortComparator != null) && (rowNumber >= 0))
        {
            debugPrintln("recheckSortedOrder " + rowNumber);

            Object rowModel = getRowModel(rowNumber);

            //!! System.out.println("rechecking sortedness of row " + rowNumber);

            boolean sorted = true;

            if (rowNumber > 0)
            {
                //!! System.out.println("rechecking sortedness against previous row");
                Object prevRowModel = getRowModel(rowNumber - 1);
                if (sortComparator.compare(prevRowModel, rowModel) > 0)
                {
                    sorted = false;
                }
            }

            if (sorted && ((rowNumber + 1) < m_listView.size()))
            {
                //!! System.out.println("rechecking sortedness against next row");
                Object nextRowModel = getRowModel(rowNumber + 1);
                if (sortComparator.compare(rowModel, nextRowModel) > 0)
                {
                    sorted = false;
                }
            }

            if (!sorted)
            {
                //!! System.out.println("Not sorted any more!");
                setSortedByColumn( -1, false);
            }
        }
    }

    /**
     *  Return the model column by which the table is sorted, or -1 if it's
     *  not currently sorted.
     */
    @Override
    public int getSortColumn()
    {
        return sortColumn;
    }

    /**
     *  Return whether this ModelListTableModel is sorted in ascending order.
     *  The results of this method are meaningless if it is not sorted at all.
     */
    @Override
    public boolean isSortedInAscendingOrder()
    {
        return sortedAscending;
    }

    //-------------------------------------------------------------------------
    //
    // TableColumnModelListener implementation
    //
    //-------------------------------------------------------------------------

    @Override
    public void columnAdded(TableColumnModelEvent evt)
    {
        m_columnCount++;
    }

    @Override
    public void columnRemoved(TableColumnModelEvent evt)
    {
        m_columnCount--;

        if (sortColumn >= m_columnCount)
        {
            sortColumn = -1;
        }
    }

    @Override
    public void columnMoved(TableColumnModelEvent evt)
    {}

    @Override
    public void columnSelectionChanged(ListSelectionEvent evt)
    {}

    @Override
    public void columnMarginChanged(ChangeEvent evt)
    {}

    //-------------------------------------------------------------------------

    class DefaultComparator implements Comparator
    {
        private RowTableColumn m_tableColumn;
        private boolean m_ascending;

        public DefaultComparator(RowTableColumn tableColumn, boolean ascending)
        {
            m_tableColumn = tableColumn;
            m_ascending   = ascending;
        }

        @Override
        public int compare(Object row1, Object row2)
        {
            Object cell1 = m_tableColumn.getColumnValue(row1);
            Object cell2 = m_tableColumn.getColumnValue(row2);

            if (m_tableColumn.getCellRenderer() instanceof ConfigBeanTableCellRenderer) {
                cell1 = ConfigBeanTableCellRenderer.getCaptionText(cell1);
                cell2 = ConfigBeanTableCellRenderer.getCaptionText(cell2);
            }

            int res = doCompare(cell1, cell2);

            return m_ascending ? res : -res;
        }

        private int doCompare(Object a, Object b)
        {
            if (a == b)
            {
                return 0;
            }

            if (a == null)
            {
                return -1;
            }

            if (b == null)
            {
                return 1;
            }

            Collator collator = Collator.getInstance();

            if ((a instanceof String) && (b instanceof String))
            {
                return collator.compare(a, b);
            }

            if (a instanceof Comparable)
            {
                return ((Comparable) a).compareTo(b);
            }

            if (b instanceof Comparable)
            {
                return -((Comparable) b).compareTo(a);
            }

            return collator.compare(a.toString(), b.toString());
        }
    }
}