package com.sonicsw.ma.gui.util;

import java.awt.event.InputEvent;
import java.util.StringTokenizer;

import javax.swing.Action;
import javax.swing.KeyStroke;

/**
 * Represents an action that loads its name, tooltip, and other
 * properties from a resource bundle.
 */
public class BasicResourceAction extends BasicAction
{
    private static final String ICON_LOCATION = "com/sonicsw/ma/gui/icons/";

    private static final String[] ACTION_KEYS = new String[]
    {
        Action.NAME,
        Action.SHORT_DESCRIPTION,
        Action.LONG_DESCRIPTION,
        Action.MNEMONIC_KEY,
        Action.ACCELERATOR_KEY,
        Action.SMALL_ICON,
        BasicAction.LARGE_ICON,
    };

    private static final String ACTION_PREFIX = "action";

    private static final String[] ACTION_KEY_TAGS = new String[]
    {
        "name",
        "short_description",
        "long_description",
        "mnemonic",
        "accelerator",
        "small_icon",
        "large_icon",
    };

    protected String m_resourceId = null;

    /**
     * Constructs the action with no resource file...the action is essentially
     * un-initialized.
     *
     * This constructor can be used to delay the configuration of the action
     * but it is expected that configureAction is called at some point.
     */
    public BasicResourceAction()
    {
        this(null, null);
    }

    /**
     * Constructs the action using the resource file.
     *
     * @param resourceName the name of the resource file to load to configure the action.
     * @param resourceId used to locate the correct tags in the resource file.
     */
    public BasicResourceAction(String resourceId,
                               Action targetAction)
    {
        super(targetAction);

        m_resourceId = resourceId;
    }

    /**
     * Returns an object representing the correct type based on the key.
     */
    private Object getActionObject(String actionKey, String val)
    {
        Object res = val;

        if (actionKey.equals(Action.ACCELERATOR_KEY))
        {
            char accelerator = val.charAt(val.lastIndexOf("+") + 1);

            int modifiers = 0;
            StringTokenizer t = new StringTokenizer(val,"+");
            while(t.hasMoreTokens())
            {
                String modifier = t.nextToken();
                if (modifier.equals("CTRL_MASK"))
                {
                    modifiers |= InputEvent.CTRL_MASK;
                }
                else if (modifier.equals("SHIFT_MASK"))
                {
                    modifiers |= InputEvent.SHIFT_MASK;
                }
                else if (modifier.equals("ALT_MASK"))
                {
                    modifiers |= InputEvent.ALT_MASK;
                }
            }

            res = KeyStroke.getKeyStroke(accelerator,modifiers);
        }
        else if (actionKey.equals(Action.MNEMONIC_KEY))
        {
            res = new Integer((int)val.toCharArray()[0]);
        }
        else if (actionKey.equals(Action.SMALL_ICON) || actionKey.equals(BasicAction.LARGE_ICON))
        {
            res = ResourceManager.getIcon(getClass(), val);
        }

        return res;
    }

    @Override
    public Object getValue(String key)
    {
        Object res = null;

        if (key.equals(Action.ACTION_COMMAND_KEY))
        {
            if (m_resourceId != null)
            {
                return m_resourceId;
            }

            return super.getValue(key);
        }

        // get the value directly from the action
        res = super.getValue(key);

        if ((res == null) && (m_resourceId != null))
        {
            String tag = keyToTag(key);
            String id  = new StringBuffer(ACTION_PREFIX).append('.').append(m_resourceId).append('.').append(tag).toString();
            String val = ResourceManager.getString(getClass(), id);

            if ((val != null) && (val.length() > 0))
            {
                res = getActionObject(key, val);
            }
        }

        return res;
    }

    private String keyToTag(String key)
    {
        for (int i = 0; i < ACTION_KEYS.length; i++)
        {
            if (key.equals(ACTION_KEYS[i]))
            {
                return ACTION_KEY_TAGS[i];
            }
        }

        return null;
    }

}
