/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui.util;

import java.awt.Component;
import java.awt.Container;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.util.List;

import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JSeparator;
import javax.swing.JTable;
import javax.swing.JTree;
import javax.swing.JViewport;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreePath;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.table.IContentPane;
import com.sonicsw.ma.gui.table.ModelListTableModel;
import com.sonicsw.ma.plugin.IPlugin;

/**
 *  ContentPaneMenuShower looks at mouse events to decide when it's time to show
 *  a JPopupMenu, and then adjusts the position of the popup if it wasn't
 *  completely on screen.
 */
public abstract class CommonPopupMenuShower extends PopupMenuShower
{
    public CommonPopupMenuShower(JTable table, JPopupMenu popup)
    {
        super(table, popup);
    }

    public CommonPopupMenuShower(JTree tree, JPopupMenu popup)
    {
        super(tree, popup);
    }

    @Override
    protected void showMenuIfPopupTrigger(MouseEvent evt)
    {
        Component  component = evt.getComponent();
        JPopupMenu popup     = getPopupMenu();

        if (popup == null)
        {
            return;
        }

        if (evt.getComponent() instanceof JViewport)
        {
            // mouse is pressed on the JViewport so we want to remove the
            // current selection prior to displaying a menu
            if (((JViewport)component).getView() == getTable())
            {
                getTable().clearSelection();
            }
        }

        if (evt.isPopupTrigger())
        {
            JWaitCursor wc = new JWaitCursor(MgmtConsole.getMgmtConsole());

            try
            {
                List list = null;

                popup.removeAll();

                if (component instanceof JTree)
                {
                    JTree c = (JTree)component;

                    int selRow = c.getRowForLocation(evt.getX(), evt.getY());

                    // if we currently have no selection then select the row
                    // under the cursor
                    if (c.getSelectionCount() == 0 && selRow >= 0)
                    {
                        c.setSelectionRow(selRow);
                    }
                    // else if we currently have a selection but it isn't the
                    // tree node under the cursor
                    else if (c.getSelectionCount() == 1 && selRow >= 0 && !c.isRowSelected(selRow))
                    {
                        c.setSelectionRow(selRow);
                    }

                    if (!c.isRowSelected(selRow))
                    {
                        return;
                    }

                    list = getMenuItems();
                }
                else
                if (component instanceof JViewport)
                {
                    Component c = ((JViewport)component).getView();

                    if (c == getTable())
                    {
                        list = getParentMenuItems();
                    }
                }
                else
                if (component instanceof JTable)
                {
                    JTable c = (JTable)component;
                    Point pt = new Point(evt.getX(), evt.getY());
                    int row = c.rowAtPoint(evt.getPoint());

                    if (row >= 0)
                    {
                        if (!c.isRowSelected(row))
                        {
                            c.setRowSelectionInterval(row, row);
                        }

                        list = getMenuItems();
                    }
                }

                if (list == null)
                {
                    return;
                }

                for (int i = 0; i < list.size(); i++)
                {
                    addMenuComponent(popup, (JComponent)list.get(i));
                }

                if (popup.getComponentCount() > 0)
                {
                    popup.show(evt.getComponent(), evt.getX(), evt.getY());
                    ensurePopupFullyOnScreen(evt);
                }
            }
            finally
            {
                wc.release();
            }
        }
    }

    protected void addMenuComponent(JPopupMenu popup, JComponent menuComponent)
    {
        if (menuComponent instanceof JMenuItem)
        {
            JMenuItem mi = (JMenuItem)menuComponent;
            Action action = mi.getAction();

            if (action instanceof BasicAction)
            {
                if (((BasicAction)action).canDisplay() == false)
                {
                    // not allowed to display action...
                    return;
                }
            }
        }
        else if (menuComponent instanceof JSeparator)
        {
            addSeparator(popup, (JSeparator)menuComponent);
            return;
        }

        popup.add(menuComponent);
    }

    /**
     * addSeparator eliminates double separators.
     * @param popup
     * @param menuSeparator
     */
    public static void addSeparator (JPopupMenu popup, JSeparator menuSeparator)
    {
        // If there is something here...
        if (popup != null && popup.getComponentCount() > 0)
        {
            // And the last component is not a separator.
            Component c = popup.getComponent (popup.getComponentCount() - 1);
            if (!(c instanceof JSeparator))
            {
                popup.add (menuSeparator);
            }
        }
    }

    protected abstract List getMenuItems();

    protected List getParentMenuItems()
    {
        return null;
    }

    /**
     * Method to detect a double-click on a selected item.
     *
     * @param event The mouse event that caused the action
     */
    @Override
    public void mouseClicked(MouseEvent evt)
    {
        Component component = evt.getComponent();

        // Check for a double-click on the selected item.
        if (evt.getClickCount() == 2)
        {
            // Manage tree node (plugin) selection.
            if (evt.getSource() == getTree())
            {
                JTree tree = (JTree)evt.getSource();

                TreePath mousePath = tree.getClosestPathForLocation(evt.getX(), evt.getY());
                TreePath selPath   = tree.getSelectionPath();

                if (!mousePath.equals(selPath))
                {
                    return;
                }

                DefaultMutableTreeNode node = (DefaultMutableTreeNode)tree.getSelectionPath().getLastPathComponent();

                if (node.isLeaf() && (node.getUserObject() instanceof IPlugin))
                {
                    IPlugin plugin = (IPlugin)node.getUserObject();

                    displayPropSheet(plugin.getDefaultAction());
                }
            }
            // Manage table row selection.
            else
            if ((evt.getSource() == getTable()) &&
                (getTable() instanceof JRowTable) &&
                (getTable().getSelectedRowCount() == 1))
            {
                JRowTable           table      = (JRowTable)evt.getSource();
                ModelListTableModel tableModel = (ModelListTableModel)table.getModel();
                int                 selRow     = table.getSelectedRow();
                Object              rowModel   = tableModel.getRowModel(selRow);

                if (rowModel != null)
                {
                    Container parent = getTable();

                    while (parent != null)
                    {
                        if (parent instanceof IContentPane)
                        {
                            displayPropSheet(((IContentPane)parent).getDefaultAction());
                            break;
                        }
                        parent = parent.getParent();
                    }
                }
            }
        }
    }

    /**
     * Method to display a property sheet for a selected  plugin.
     *
     * @param plugin The selected plugin
     */
    protected void displayPropSheet(BasicAction action)
    {
        Helper.logDebugMessage("displayPropSheet : " + action);

        if (action != null)
        {
            Object source = (getTree() != null) ? (Object)getTree() : (Object)getTable();

            action.actionPerformed(new ActionEvent(source, 0, action.getActionCommand()));
        }
    }
}