package com.sonicsw.ma.gui.util;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.Action;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JToggleButton;
import javax.swing.border.EmptyBorder;

public class EditableComponentPanel
    extends JPanel
{
    protected ExtendedJScrollPane m_scroller = null;
    protected JComponent m_component = null;
    protected JPanel m_panelButtons = null;
    protected JButton m_buttonAdd = null;
    protected JButton m_buttonRemove = null;
    protected JButton m_buttonEdit = null;
    protected Icon m_iconObject = null;

    protected boolean m_bAddAction = false;
    protected boolean m_bEditAction = false;
    protected boolean m_bRemoveAction = false;

    public static final int ICONS_AND_LABELS = 0;
    public static final int ICONS_ONLY = 1;
    public static final int LABELS_ONLY = 2;

    public EditableComponentPanel()
    {
        this(true, true, true);
    }

    public EditableComponentPanel(boolean bAdd, boolean bEdit, boolean bRemove)
    {
        super();

        m_bAddAction = bAdd;
        m_bEditAction = bEdit;
        m_bRemoveAction = bRemove;
    }

    public void setComponent(JComponent component)
    {
        m_component = component;
    }

    public final void setObjectIcon(Icon icon)
    {
        m_iconObject = icon;
    }

    /**
     * Overrides JComponent#setEnabled in order to enable/disable the components within this panel.
     * @param bEnabled
     */
    @Override
    public void setEnabled(boolean bEnabled)
    {
        m_component.setEnabled(bEnabled);
        if (m_buttonAdd != null)
        {
            m_buttonAdd.setEnabled(bEnabled);
        }

        if (m_buttonRemove != null)
        {
            m_buttonRemove.setEnabled(bEnabled);
        }

        if (m_buttonEdit != null)
        {
            m_buttonEdit.setEnabled(bEnabled);
        }

        super.setEnabled(bEnabled);
    }

    protected void init(String nBorder, int nButtonStyle)
    {
        init(nBorder, true, nButtonStyle, true);
    }

    protected void init(String nBorder, boolean bCreateScroller, int nButtonStyle)
    {
        init(nBorder, bCreateScroller, nButtonStyle, true);
    }

    protected void init(String nBorder, int nButtonStyle, boolean bAutoEdit)
    {
        init(nBorder, true, nButtonStyle, bAutoEdit);
    }

    protected void init(String nBorder, boolean bCreateScroller, int nButtonStyle, boolean bAutoEdit)
    {
        setLayout(new BorderLayout());

        if (bCreateScroller)
        {
            // Create a scroller for the component.
            m_scroller = new ExtendedJScrollPane(m_component);
            add(m_scroller, BorderLayout.CENTER);
        }
        else
        {
            add(m_component, BorderLayout.CENTER);
        }

        if (m_bAddAction || m_bEditAction || m_bRemoveAction)
        {
            // Create the buttons for the panels.
            if (m_bAddAction)
            {
                m_buttonAdd = addButton(new AddAction(), nBorder, nButtonStyle);
            }

            if (m_bEditAction)
            {
                m_buttonEdit = addButton(new EditAction(), nBorder, nButtonStyle);
            }

            if (m_bRemoveAction)
            {
                m_buttonRemove = addButton(new RemoveAction(), nBorder, nButtonStyle);
            }

            if (bAutoEdit && m_bEditAction)
            {
                // Add a double click listener for editing.
                m_component.addMouseListener(new MouseAdapter()
                {
                    @Override
                    public void mouseClicked(MouseEvent e)
                    {
                        if (e.getClickCount() == 2 && canEnableEdit())
                        {
                            onEdit();
                        }
                    }
                });
            }

            addButtons();

            // Make it look perty.
            Helper.equalize(m_panelButtons);
        }
    }

    // A convenient place to add additional buttons.
    protected void addButtons()
    {
    }

    protected void createButtonPanel(String nBorder)
    {
        // Create the bottom (button) panel.
        if (m_panelButtons == null)
        {
            m_panelButtons = new JPanel();
            m_panelButtons.setBorder(new EmptyBorder(JPartitionPanel.BORDER_INSETS));
            add(m_panelButtons, nBorder);

            // Set the layout based on the direction of preference.
            if (nBorder == BorderLayout.EAST || nBorder == BorderLayout.WEST)
            {
                m_panelButtons.setLayout(new BoxLayout(m_panelButtons, BoxLayout.Y_AXIS));
            }
            else
            {
                m_panelButtons.setLayout(new FlowLayout(FlowLayout.RIGHT, 5, 5));
            }
        }
    }

    protected JButton addButton(BasicResourceAction a, String nBorder, int nButtonStyle)
    {
        // Create the panel if not already created.
        createButtonPanel(nBorder);

        JButton button = null;

        if (a.canDisplay())
        {
            button = addButton((Action) a, nBorder, nButtonStyle);
        }

        return (button);
    }

    protected JButton addButton(Action a, String nBorder, int nButtonStyle)
    {
        // Create the panel if not already created.
        createButtonPanel(nBorder);

        JButton button = createButton(a);

        // Add it to the panel.
        addComponent(button, nBorder);

        // If there are no icons, turn it off.
        if (nButtonStyle == LABELS_ONLY)
        {
            button.setIcon(null);
        }

        // Otherwise if there are no labels, turn them off.
        else if (nButtonStyle == ICONS_ONLY)
        {
            // Clear the text and set the border to make the button look
            // like a toolbar button.
            button.setText(null);
        }

        return (button);
    }

    protected JButton createButton(Action a)
    {
        return new ExtendedJButton(a);
    }

    protected JToggleButton addToggleButton(Action a, String nBorder, int nButtonStyle)
    {
        JToggleButton button = new JToggleButton(a);

        // Add it to the panel.
        addComponent(button, nBorder);

        // If there are no icons, turn it off.
        if (nButtonStyle == LABELS_ONLY)
        {
            button.setIcon(null);
        }

        // Otherwise if there are no labels, turn them off.
        else if (nButtonStyle == ICONS_ONLY)
        {
            // Clear the text and set the border to make the button look
            // like a toolbar button.
            button.setText(null);
        }

        return (button);
    }

    protected void addComponent(Component c, String nBorder)
    {
        if (m_panelButtons == null)
        {
            createButtonPanel(nBorder);
        }

        // Add it to the panel.
        m_panelButtons.add(c);

        // If it is in a vertical alignment, we need a strut also.
        if (nBorder == BorderLayout.EAST || nBorder == BorderLayout.WEST)
        {
            m_panelButtons.add(Box.createVerticalStrut(5));
        }
    }

    protected void setUIState()
    {
        // Set the state of the add / remove / edit buttons.
        if (m_bAddAction && m_buttonAdd != null)
        {
            if (m_buttonAdd.getAction()instanceof BasicResourceAction)
            {
                m_buttonAdd.setEnabled(((BasicResourceAction) m_buttonAdd.getAction()).isEnabled());
            }
            else
            {
                m_buttonAdd.setEnabled(canEnableAdd());
            }
        }

        if (m_bEditAction && m_buttonEdit != null)
        {
            if (m_buttonEdit.getAction()instanceof BasicResourceAction)
            {
                m_buttonEdit.setEnabled(((BasicResourceAction) m_buttonEdit.getAction()).isEnabled());
            }
            else
            {
                m_buttonEdit.setEnabled(canEnableEdit());
            }
        }

        if (m_bRemoveAction && m_buttonRemove != null)
        {
            if (m_buttonRemove.getAction()instanceof BasicResourceAction)
            {
                m_buttonRemove.setEnabled(((BasicResourceAction) m_buttonRemove.getAction()).isEnabled());
            }
            else
            {
                m_buttonRemove.setEnabled(canEnableRemove());
            }
        }
    }

    // Give access to our components for easier override.
    public JScrollPane getScroller()
    {
        return (m_scroller);
    }

    public JComponent getComponent()
    {
        return (m_component);
    }

    public JPanel getButtonPanel()
    {
        return (m_panelButtons);
    }

    public JButton getAddButton()
    {
        return (m_buttonAdd);
    }

    public JButton getEditButton()
    {
        return (m_buttonEdit);
    }

    public JButton getRemoveButton()
    {
        return (m_buttonRemove);
    }

    protected void onAdd()
    {
    }

    protected void onRemove()
    {
    }

    protected void onEdit()
    {
    }

    protected boolean canEnableAdd()
    {
        return true;
    }

    protected boolean canEnableEdit()
    {
        return true;
    }

    protected boolean canEnableRemove()
    {
        return true;
    }

    protected boolean canDisplayAdd()
    {
        return true;
    }

    protected boolean canDisplayEdit()
    {
        return true;
    }

    protected boolean canDisplayRemove()
    {
        return true;
    }

    //////////////////////////////////////////////////////////////////////////
    //
    // Inner Classes

    class AddAction
        extends BasicGuiAction
    {
        public AddAction()
        {
            super("editable_component.add");

            if (m_iconObject != null)
            {
                putSmallIconValue();
            }
        }

        private void putSmallIconValue() {
            putValue(Action.SMALL_ICON, new ExtendedImageIcon(m_iconObject, ResourceManager.getIcon(getClass(), "newoverlay")));
        }
        
        @Override
        public boolean isEnabled()
        {
            return canEnableAdd();
        }

        @Override
        public boolean canDisplay()
        {
            return canDisplayAdd();
        }

        @Override
        public void actionPerformed(ActionEvent e)
        {
            onAdd();

            setUIState();
        }
    }

    class EditAction
        extends BasicGuiAction
    {
        public EditAction()
        {
            super("editable_component.edit");

            if (m_iconObject != null)
            {
                putSmallIconValue();
            }
        }
        
        private void putSmallIconValue() {
            putValue(Action.SMALL_ICON, new ExtendedImageIcon(m_iconObject, ResourceManager.getIcon(getClass(), "editoverlay")));
        }
        
        @Override
        public boolean isEnabled()
        {
            return canEnableEdit();
        }

        @Override
        public boolean canDisplay()
        {
            return canDisplayEdit();
        }

        @Override
        public void actionPerformed(ActionEvent e)
        {
            onEdit();

            setUIState();
        }
    }

    class RemoveAction
        extends BasicGuiAction
    {
        public RemoveAction()
        {
            super("editable_component.remove");

            if (m_iconObject != null)
            {
                putSmallIconValue();
            }
        }
        
        private void putSmallIconValue() {
            putValue(Action.SMALL_ICON, new ExtendedImageIcon(m_iconObject, ResourceManager.getIcon(getClass(), "deleteoverlay")));
        }

        @Override
        public boolean isEnabled()
        {
            return canEnableRemove();
        }

        @Override
        public boolean canDisplay()
        {
            return canDisplayRemove();
        }

        @Override
        public void actionPerformed(ActionEvent e)
        {
            onRemove();

            setUIState();
        }
    }
}