package com.sonicsw.ma.gui.util;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.text.DateFormat;
import java.util.Date;
import java.util.Vector;

import javax.swing.Icon;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.ToolTipManager;
import javax.swing.UIManager;
import javax.swing.event.TableModelEvent;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

import com.sonicsw.ma.gui.MgmtConsole;

public class ExtendedJTable extends JTable
{
    public ExtendedJTable ()
    {
        this(null);
        commonInit();
    }

    public ExtendedJTable(TableModel dm, TableColumnModel cm)
    {
        super(dm, cm);
        commonInit();
    }

    public ExtendedJTable(TableModel dm, TableColumnModel cm, ListSelectionModel sm)
    {
        super(dm, cm, sm);
        commonInit();
    }

    public ExtendedJTable(int numRows, int numColumns)
    {
        super(numRows, numColumns);
        commonInit();
    }

    public ExtendedJTable(final Vector rowData, final Vector columnNames)
    {
        super(rowData, columnNames);
        commonInit();
    }

    public ExtendedJTable(final Object[][] rowData, final Object[] columnNames)
    {
        super(rowData, columnNames);
        commonInit();
    }

    public ExtendedJTable (TableModel model)
    {
        super(model);
        commonInit();
    }

    public final void commonInit()
    {
        //Enable tool tips.
        ToolTipManager.sharedInstance().registerComponent(this);

        getTableHeader().setDefaultRenderer(new ExtendedTableHeaderRenderer());
        setDefaultRenderer(Object.class, new ExtendedTableCellRenderer());
    }

    @Override
    public void setShowHorizontalLines(boolean state)
    {
        if (state)
        {
            setIntercellSpacing(new Dimension(1, getShowVerticalLines() ? 1 : 0));
        }
        else
        {
            setIntercellSpacing(new Dimension(0, getShowVerticalLines() ? 1 : 0));
        }

        super.setShowHorizontalLines(state);
    }

    @Override
    public void setShowVerticalLines(boolean state)
    {
        if (state)
        {
            setIntercellSpacing(new Dimension(getShowHorizontalLines() ? 1 : 0, 1));
        }
        else
        {
            setIntercellSpacing(new Dimension(getShowHorizontalLines() ? 1 : 0, 0));
        }

        super.setShowVerticalLines(state);
    }

    public void stopEditing()
    {
        if (getCellEditor() != null)
        {
            getCellEditor().stopCellEditing();
        }
    }

    public int getViewportWidth()
    {
        int nWidthTable = getSize().width;

        if (getScroller() != null)
        {
            nWidthTable = getScroller().getViewport().getWidth();
        }

        return nWidthTable;
    }

    public JScrollPane getScroller()
    {
        return (JScrollPane)SwingUtilities.getAncestorOfClass(JScrollPane.class, this);
    }

    // The override for being told when a cell is being edited. Works in conjunction
    // with editingStopped() and editingCanceled().
    public void editingStarted ()
    {
    }

    // this overrides the stupid code in the JDK that clears the table selection
    // on a table change.  -jsp
    @Override
    public void tableChanged(TableModelEvent e)
    {
        if (e == null || e.getFirstRow() == TableModelEvent.HEADER_ROW) {

            if (getAutoCreateColumnsFromModel())
            {
                // This will effect invalidation of the JTable and JTableHeader.
                createDefaultColumnsFromModel();
                return;
            }
        }

        resizeAndRepaint();
        return;
    }

    public boolean hasSelection ()
    {
        return getSelectedRowCount() > 0 || getSelectedColumnCount() > 0;
    }

    public Color getBackgroundColorBySelectionState(boolean bSelected)
    {
        Color bg = UIManager.getColor("TextField.background");

        if (bSelected)
        {
            if (hasFocus() == false)
            {
                bg = UIManager.getColor("ComboBox.disabledBackground");
            }
            else
            {
                bg = UIManager.getColor("Tree.selectionBackground");
            }
        }

        return bg;
    }

    public Color getForegroundColorBySelectionState(boolean bSelected)
    {
        Color fg = UIManager.getColor("Tree.textForeground");

        if (bSelected)
        {
            if (hasFocus() == false)
            {
                fg = UIManager.getColor("Tree.textForeground");
            }
            else
            {
                fg = UIManager.getColor("Tree.selectionForeground");
            }
        }

        return fg;
    }

    @Override
    public boolean isEditing()
    {
        boolean bIsEditing = super.isEditing();

        return bIsEditing;
    }

    public int getNumberSelections()
    {
        int[] arRows = getSelectedRows();

        return arRows.length;
    }

    /////////////////////////////////////////////////////////////////////////////////////
    // Inner Classes

    public static class ExtendedTableCellRenderer
        extends DefaultTableCellRenderer
        implements SwingConstants
    {
        private int m_gray = 0;


        public ExtendedTableCellRenderer (int nAlignment)
        {
            super();

            setCellHorizontalAlignment (nAlignment);
        }

        private void setCellHorizontalAlignment(int nAlignment) {
            setHorizontalAlignment(nAlignment);
        }
        
        public ExtendedTableCellRenderer ()
        {
            this(JLabel.LEFT);
        }

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value,
            boolean isSelected, boolean hasFocus, int row, int column)
        {
            Component c = super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);

            setGrayIcon(0);

            // Make sure the value isn't null.
            if (value != null)
            {
                // Only gray the icon if the plugin on the clipboard has been cut...
                //
                if (MgmtConsole.isInitialized() && MgmtConsole.getMgmtConsole().getClipboardAction() == MgmtConsole.CLIP_ACTION_CUT)
                {
                    Object clipboardObject = MgmtConsole.getMgmtConsole().getClipboardObject();

                    if ((clipboardObject != null) && clipboardObject.equals(value))
                    {
                        setGrayIcon(40);
                    }
                }


                String strTip = null;

                if (value instanceof Date)
                {
                    strTip = DateFormat.getDateTimeInstance().format((Date) value);
                }
                else
                {
                    strTip = (String)value.toString();
                }

                setToolTipText(c, table, strTip, column);
            }
            else
            {
                ((JComponent)c).setToolTipText(null);
            }

            ((JComponent) c).setBackground(((ExtendedJTable) table).getBackgroundColorBySelectionState(table.isRowSelected(row)));
            ((JComponent) c).setForeground(((ExtendedJTable) table).getForegroundColorBySelectionState(table.isRowSelected(row)));

            return c;
        }

        public void setToolTipText(Component c, JTable table, String strText, int column)
        {
            int nStringWidth = SwingUtilities.computeStringWidth(table.getFontMetrics(table.getFont()), strText);
            int nIconWidth = getIcon() == null ? 0 : getIcon().getIconWidth();

            if (table.getColumnModel().getColumn(column).getWidth() < nStringWidth + nIconWidth)
            {
                // display the tip if the value is truncated...
                if (strText != null && strText.length() > 0)
                {
                    ((JComponent)c).setToolTipText(Helper.formatToHTMLForWrapping(strText, 128));
                }
            }
            else
            {
                // turn the tip off...
                ((JComponent)c).setToolTipText(null);
            }
        }

        public void setGrayIcon(int percentage)
        {
            m_gray = percentage;
        }

        @Override
        public Icon getIcon()
        {
            try
            {
                Icon icon = super.getIcon();

                if ((icon != null) && (m_gray > 0))
                {
                    icon = Helper.createDisabledIcon(icon, m_gray);
                }

                return icon;
            }
            catch (Exception e)
            {
                e.printStackTrace();
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
            }

            return null;
        }

    }

    public static class ExtendedTableHeaderRenderer
        extends ExtendedTableCellRenderer
    {
        public ExtendedTableHeaderRenderer ()
        {
            super ();
        }

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value,
            boolean isSelected, boolean hasFocus, int row, int column)
        {
            Component c = super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);

            if (table != null)
            {
                JTableHeader header = table.getTableHeader();
                if (header != null)
                {
                    ((JComponent)c).setForeground(header.getForeground());
                    ((JComponent)c).setBackground(header.getBackground());
                    ((JComponent)c).setFont(header.getFont());
                }
            }
            setText((value == null) ? "" : value.toString());
            ((JComponent)c).setBorder(UIManager.getBorder("TableHeader.cellBorder"));
            setHorizontalAlignment(JLabel.CENTER);
            setHorizontalTextPosition(JLabel.LEADING);

            return c;
        }
    }
}
