package com.sonicsw.ma.gui.util;

import java.util.ArrayList;
import java.util.Iterator;

import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;

import com.sonicsw.ma.gui.MgmtConsole;

/**
 * <p>Title: FilterTreeModel</p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2002, Sonic Software Corporation</p>
 * @author Jeffrey S. Pace
 * @version 1.0
 */
public class FilterTreeModel
    extends DefaultTreeModel
{
    protected boolean m_bFilterIsActive;

    /**
     * FilterTreeModel
     * @param root
     */
    public FilterTreeModel(TreeNode root)
    {
        this(root, false);
    }

    /**
     * FilterTreeModel
     * @param root
     * @param asksAllowsChildren
     */
    public FilterTreeModel(TreeNode root, boolean asksAllowsChildren)
    {
        this(root, false, false);
    }

    /**
     * FilterTreeModel
     * @param root
     * @param asksAllowsChildren
     * @param filterIsActive defines if the filter is active
     */
    public FilterTreeModel(TreeNode root, boolean asksAllowsChildren, boolean bFilterIsActive)
    {
        super(root, asksAllowsChildren);
        m_bFilterIsActive = bFilterIsActive;
    }

    /**
     * activateFilter
     * @param filterIsActive defines if the filter is active
     */
    public void activateFilter(boolean bFilterIsActive)
    {
        m_bFilterIsActive = bFilterIsActive;
    }

    /**
     * isActivatedFilter
     * @return
     */
    public boolean isActivatedFilter()
    {
        return m_bFilterIsActive;
    }

    /**
     *
     * @param node
     * @return
     */
    public boolean isNodeFiltered(Object node)
    {
        boolean bIsFilterable = node != null && node instanceof IFilterable;
        return bIsFilterable ? !((IFilterable)node).canDisplay(this) : false;
    }

    /**
     * Invoke this method after you've changed how node is to be represented
     * in the tree.
     *
     * @param node
     */
    @Override
    public void nodeChanged(TreeNode node)
    {
        if (isActivatedFilter())
        {
            // filter is active...
            if (isNodeFiltered(node))
            {
                // node is filtered...
                return;
            }
        }

        super.nodeChanged(node);
    }

    /**
     * Invoke this method after you've changed how the children identified by
     * childIndicies are to be represented in the tree.
     *
     * @param node
     * @param childIndices
     */
    @Override
    public void nodesChanged(TreeNode node, int[] childIndices)
    {
        if (isActivatedFilter())
        {
            // filter is active...
            if (isNodeFiltered(node))
            {
                // node is filtered...
                return;
            }

            super.nodesChanged(node, getUnfilteredChildInidicies(node, childIndices));
            return;
        }

        super.nodesChanged(node, childIndices);
    }

    /**
     * Invoke this method if you've totally changed the children of node and
     * its childrens children... This will post a treeStructureChanged event.
     *
     * @param node
     */
    @Override
    public void nodeStructureChanged(TreeNode node)
    {
        if (isActivatedFilter())
        {
            // filter is active...
            if (isNodeFiltered(node))
            {
                // node is filtered...
                return;
            }
        }

        super.nodeStructureChanged(node);
    }

    /**
     * Invoke this method after you've inserted some TreeNodes into node.
     * childIndices should be the index of the new elements and must be sorted
     * in ascending order.
     *
     * @param node
     * @param childIndices
     */
    @Override
    public void nodesWereInserted(TreeNode node, int[] childIndices)
    {
        if (isActivatedFilter())
        {
            // filter is active...
            if (isNodeFiltered(node))
            {
                // node is filtered...
                return;
            }

            super.nodesWereInserted(node, getUnfilteredChildInidicies(node, childIndices));
            return;
        }

        super.nodesWereInserted(node, childIndices);
    }

    /**
     * Invoke this method after you've removed some TreeNodes from node.
     * childIndices should be the index of the removed elements and must be
     * sorted in ascending order. And removedChildren should be the array of
     * the children objects that were removed.
     *
     * @param node
     * @param childIndices
     * @param removedChildren
     */
    @Override
    public void nodesWereRemoved(TreeNode node, int[] childIndices, Object[] removedChildren)
    {
        if (isActivatedFilter())
        {
            // filter is active...
            if (isNodeFiltered(node))
            {
                // node is filtered...
                return;
            }
        }

        super.nodesWereRemoved(node, childIndices, removedChildren);
    }

    /**
     * Returns an unfiltered child for the specified parent node and index.
     *
     * @param parent parent node
     * @param index index of requested child.
     * @return unfiltered child node.
     */
    @Override
    public Object getChild(Object parent, int index)
    {
        if (isActivatedFilter())
        {
            if (parent instanceof FilterTreeNode)
            {
                return ((FilterTreeNode)parent).getChildAt(index, isActivatedFilter());
            }
        }
        return ((TreeNode)parent).getChildAt(index);
    }

    /**
     * Returns a count of unfiltered children for the specified parent node.
     *
     * @param parent parent node
     * @return count of unfiltered children.
     */
    @Override
    public int getChildCount(Object parent)
    {
        if (isActivatedFilter())
        {
            if (parent instanceof FilterTreeNode)
            {
                return ((FilterTreeNode)parent).getChildCount(isActivatedFilter());
            }
        }
        return ((TreeNode)parent).getChildCount();
    }

    /**
     * Returns an int array of unfiltered child indicies.
     *
     * @param nodeParent the parent of the child nodes.
     * @param childIndices the indicies of the children
     * @return array of unfiltered child indicies.
     */
    protected int[] getUnfilteredChildInidicies(TreeNode nodeParent, int[] childIndices)
    {
        ArrayList listIndicies = new ArrayList();

        for (int i = 0; i < childIndices.length; i++)
        {
            TreeNode nodeChild = null;

            try
            {
                if (nodeParent instanceof FilterTreeNode)
                {
                    nodeChild = ((FilterTreeNode)nodeParent).getChildAt(childIndices[i],
                        isActivatedFilter());
                }
                else
                {
                    nodeChild = nodeParent.getChildAt(childIndices[i]);
                }
            }
            catch (ArrayIndexOutOfBoundsException e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
            }

            // filter is active...
            if (isNodeFiltered(nodeChild) == false)
            {
                // node is not filtered...
                listIndicies.add(new Integer(childIndices[i]));
            }
        }

        // convert to int array...
        int[] arIndicies = new int[listIndicies.size()];
        Iterator iter = listIndicies.iterator();
        int i = 0;

        while (iter.hasNext())
        {
            arIndicies[i++] = ((Integer)iter.next()).intValue();
        }

        return arIndicies;
    }
}