package com.sonicsw.ma.gui.util;

import java.text.Collator;
import java.util.Enumeration;

import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreeNode;

/**
 * <p>Title: FilterTreeNode</p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2002, Sonic Software Corporation</p>
 * @author Jeffrey S. Pace
 * @version 1.0
 */
public class FilterTreeNode
    extends DefaultMutableTreeNode
    implements IFilterable
{
    protected boolean m_bCanDisplay;

    /**
     * FilterTreeNode
     */
    public FilterTreeNode()
    {
        this(null);
    }

    /**
     * FilterTreeNode
     * @param userObject
     */
    public FilterTreeNode(Object userObject)
    {
        this(userObject, true, true);
    }

    /**
     * FilterTreeNode
     * @param userObject
     * @param allowsChildren
     * @param bCanDisplay
     */
    public FilterTreeNode(Object userObject, boolean allowsChildren, boolean bCanDisplay)
    {
        super(userObject, allowsChildren);
        m_bCanDisplay = bCanDisplay;
    }

    /**
     * getInsertIndex
     * @param childName
     * @return
     */
    public int getInsertIndex(String childName)
    {
        for (int i = 0; i < super.getChildCount(); i++)
        {
            TreeNode node = getChildAt(i);

            if (Collator.getInstance().compare(childName, node.toString()) <= 0)
            {
                return i;
            }
        }
        return super.getChildCount();
    }

    /**
     * getChildAt
     * @param index an index into this node's child array
     * @param filterIsActive defines if the filter is active
     * @return the TreeNode in this filtered node's child array at the specified index
     */
    public TreeNode getChildAt(int nIndex, boolean bFilterIsActive)
    {
        if (!bFilterIsActive)
        {
            // default behavior...
            return super.getChildAt(nIndex);
        }
        if (children == null)
        {
            // bad mojo...
            throw new ArrayIndexOutOfBoundsException("node has no children");
        }

        // iterate through those nodes displayed until we match on our index...
        int nRealIndex = -1;
        int nVisibleIndex = -1;
        Enumeration en = children.elements();
        while (en.hasMoreElements())
        {
            TreeNode node = (TreeNode)en.nextElement();

            if (node instanceof IFilterable)
            {
                IFilterable nodeFiltered = (IFilterable)node;
                if (nodeFiltered.canDisplay())
                {
                    nVisibleIndex++;
                }
            }
            else
            {
                // not filtered - so it must be visible.
                nVisibleIndex++;
            }
            nRealIndex++;

            if (nVisibleIndex == nIndex)
            {
                // we found our index.
                return (TreeNode)children.elementAt(nRealIndex);
            }
        }

        throw new ArrayIndexOutOfBoundsException("index unmatched: visible index = " +
            nVisibleIndex + ", index = " + nIndex);
    }

    /**
     * getChildCount
     * @param filterIsActive defines if the filter is active
     * @return the number of visible filtered child nodes
     */
    public int getChildCount(boolean bFilterIsActive)
    {
        if (!bFilterIsActive)
        {
            // default behavior...
            return super.getChildCount();
        }
        if (children == null)
        {
            return 0;
        }

        // only count those nodes that are filtered and visible or not filtered...
        int nCount = 0;
        Enumeration en = children.elements();
        while (en.hasMoreElements())
        {
            TreeNode node = (TreeNode)en.nextElement();

            if (node instanceof IFilterable)
            {
                IFilterable nodeFiltered = (IFilterable)node;
                if (nodeFiltered.canDisplay())
                {
                    nCount++;
                }
            }
            else
            {
                // not a filtered node - so it must be displayed...
                nCount++;
            }
        }

        return nCount;
    }

    /**
     * setDisplayable
     * @param bCanDisplay sets if this node can display.
     */
    @Override
    public final void setDisplayable(boolean bCanDisplay)
    {
        m_bCanDisplay = bCanDisplay;
    }

    /**
     * canDisplay
     * @objModel objModel sets if this node can display in this model.
     * @return value specifying if this node can be displayed.
     */
    @Override
    public boolean canDisplay(Object objModel)
    {
        return m_bCanDisplay;
    }

    /**
     * canDisplay
     * @return value specifying if this node can be displayed.
     */
    @Override
    public boolean canDisplay()
    {
        return m_bCanDisplay;
    }

    /**
     * getVisibleChildIndex
     * @param aNode
     * @return
     */
    public int getVisibleChildIndex(TreeNode aNode)
    {
        // iterate through those nodes displayed until we match on our index...
        int nRealIndex = -1;
        int nVisibleIndex = -1;
        Enumeration en = children.elements();
        while (en.hasMoreElements())
        {
            TreeNode node = (TreeNode)en.nextElement();

            if (node instanceof IFilterable)
            {
                IFilterable nodeFiltered = (IFilterable)node;
                if (nodeFiltered.canDisplay())
                {
                    nVisibleIndex++;
                }
            }
            else
            {
                // not filtered - so it must be visible.
                nVisibleIndex++;
            }
            nRealIndex++;

            if (node.equals(aNode))
            {
                // we found our index.
                return nVisibleIndex;
            }
        }

        return -1;
    }
}