package com.sonicsw.ma.gui.util;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.GraphicsConfiguration;
import java.awt.GridBagConstraints;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.image.FilteredImageSource;
import java.awt.image.ImageProducer;
import java.io.IOException;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.StringTokenizer;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.GrayFilter;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JSeparator;
import javax.swing.JTable;
import javax.swing.KeyStroke;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.table.TableCellRenderer;
import javax.swing.text.JTextComponent;

import com.sonicsw.ma.gui.MgmtConsole;

public class Helper
{
	static private final ThreadLocal<DateFormat> DATE_PARSER_THREAD_LOCAL = new ThreadLocal<DateFormat>() {
        @Override
        protected DateFormat initialValue() {
            return DateFormat.getDateTimeInstance();
        }
    };
    
    static private final String PROLOG = "<HTML>";
    static private final String EPILOG = "</HTML>";

    public final static String TIP_PROLOG         = "<HTML><BODY><TABLE CELLPADDING=1>";
    public final static String TABLE_LEFT         = "<tr><td><b>";
    public final static String TABLE_MID          = "</b></td><td>";
    public final static String TABLE_RIGHT        = "</td></tr>";
    public final static String FONT_STATUS_UP     = "<FONT face=\"Arial\" SIZE=\"2\" color=\"#009966\">";
    public final static String FONT_STATUS_DOWN   = "<FONT face=\"Arial\" SIZE=\"2\" color=\"#FF0000\">";
    public final static String FONT_END           = "</font>";
    public final static String TIP_EPILOG         = "</TABLE></BODY></HTML>";

    private static final ExecutorService threadPool = Executors.newCachedThreadPool();
    
    private static class TableCellRendererHolder
    {
        final static private TableCellRenderer INSTANCE = new DateTimeTableCellRenderer();
    }

    /**
     * Creates a "disabled" version of the supplied icon by turning it into
     * a grayscale image and brightening the pixels in the image.
     *
     * @param
     * @param percentage
     * @return
     * @deprecated  Use createDisabledIcon(Icon i, int percentage) instead.
     */
    public static final Icon createDisabledIcon(ImageIcon i, int percentage)
    {
        return createDisabledIcon((Icon)i, percentage);
    }

    public static final Icon createDisabledIcon(Icon i, int percentage)
    {
        Icon iconResult = i;

        if (i != null)
        {
            if (i instanceof ImageIcon)
            {
                GrayFilter filter = new GrayFilter(true, percentage);
                ImageProducer prod = new FilteredImageSource(((ImageIcon)i).getImage().getSource(), filter);
                Image grayImage = Toolkit.getDefaultToolkit().createImage(prod);
                iconResult = new ImageIcon(grayImage);
            }
        }

        return iconResult;
    }

    public static Logger getLogger()
    {
        return Logger.global;
    }

    public static void logDebugMessage(String strMessage)
    {
        if (getLogger().getLevel() == Level.ALL)
        {
            logInfoMessage(strMessage);
        }
    }

    public static void logInfoMessage(String strMessage)
    {
        getLogger().info(strMessage);
    }

    public static void logWarningMessage(String strMessage)
    {
        getLogger().warning(strMessage);
    }

    public static void logErrorMessage(String strMessage)
    {
        getLogger().severe(strMessage);
    }

    /**
     * Looks for a button under the supplied Container (Component) that has matches
     * the action. An easy way to match actions with they're component without
     * having to keep references to all the J-stuff...only need to hold onto the
     * action.
     *
     * @param container  The container underwhich the action's component should
     *                   be found.
     * @param action     The action we're looking to match up with a button
     * @return           The action-matching AbstractButton
     */
    public static final AbstractButton getDescendentButtonWithAction(Container container,
        Action action)
    {
        AbstractButton res = null;

        if (container instanceof AbstractButton)
        {
            AbstractButton btn = (AbstractButton) container;

            if (btn.getAction() == action)
            {
                res = btn;
            }
        }
        else
        {
            for (int i = 0; i < container.getComponentCount(); i++)
            {
                if ((res = getDescendentButtonWithAction((Container) container.getComponent(i), action)) != null)
                {
                    break;
                }
            }
        }

        return res;
    }

    public static Container getDescendentNamed(String name, Container comp)
    {
        Container res = null;

        if ((comp.getName() != null) && comp.getName().equalsIgnoreCase(name))
        {
            res = comp;
        }
        else
        {
            for (int i = 0; i < comp.getComponentCount(); i++)
            {
                if ((res = getDescendentNamed(name, (Container) comp.getComponent(i))) != null)
                {
                    break;
                }
            }
        }

        return res;
    }

    public static Container getDescendentClass(Class compClass, Container comp)
    {
        Container res = null;

        if (comp.getClass().equals(compClass))
        {
            res = comp;
        }
        else
        {
            for (int i = 0; i < comp.getComponentCount(); i++)
            {
                if ((res = getDescendentClass(compClass, (Container) comp.getComponent(i))) != null)
                {
                    break;
                }
            }
        }

        return res;
    }

    public static final String arrayToString(String[] array)
    {
        StringBuffer sb = new StringBuffer();

        if (array != null)
        {
            for (int i = 0; i < array.length; i++)
            {
                if (i > 0)
                {
                    sb.append(";");
                }
                sb.append(array[i]);
            }
        }

        return sb.toString();
    }

    public static final String[] stringToArray(String value)
    {
        StringTokenizer st = new StringTokenizer((value == null) ? "" : value, ";");
        ArrayList list = new ArrayList(st.countTokens());

        while (st.hasMoreTokens())
        {
            list.add(st.nextToken());
        }

        return (String[]) list.toArray(new String[list.size()]);
    }

    public static JComponent[] combineMenu(JComponent[] array1,
                                           JComponent[] array2,
                                           boolean useSeparator)
    {
        ArrayList list = new ArrayList();

        if (array1 != null)
        {
            for (int i = 0; i < array1.length; i++)
            {
                list.add(array1[i]);
            }
        }

        if (useSeparator)
        {
            list.add(new JSeparator());
        }

        if (array2 != null)
        {
            for (int i = 0; i < array2.length; i++)
            {
                list.add(array2[i]);
            }
        }

        return (JComponent[]) list.toArray(new JComponent[list.size()]);
    }

    /**
     * Computes the origin for the popup menu.
     */
    public static Point getPopupMenuOrigin(JComponent origin, JComponent popupMenu)
    {
        int x = 0;
        int y = 0;
        // Figure out the sizes needed to caclulate the menu position
        Dimension screenSize;
        Dimension s = origin.getSize();
        Dimension pmSize = popupMenu.getSize();

        // The first time the menu is popped up, the size has not been initiated
        if (pmSize.width == 0)
        {
            pmSize = popupMenu.getPreferredSize();
        }

        Point position = origin.getLocationOnScreen();
        GraphicsConfiguration gc = origin.getGraphicsConfiguration();

        if (gc == null)
        {
            screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        }
        else
        {
            Rectangle sBounds = gc.getBounds();

            screenSize = sBounds.getSize();
            position.x -= sBounds.x;
            position.y -= sBounds.y;
        }

        Container parent = origin.getParent();

        if (parent instanceof JPopupMenu)
        {
            // We are a submenu (pull-right)
            int xOffset = UIManager.getInt("Menu.submenuPopupOffsetX");
            int yOffset = UIManager.getInt("Menu.submenuPopupOffsetY");

            if (origin.getComponentOrientation().isLeftToRight())
            {
                // First determine x:
                x = s.width + xOffset; // Prefer placement to the right
                if (position.x + x + pmSize.width >= screenSize.width &&
                    // popup doesn't fit - place it wherever there's more room
                    screenSize.width - s.width < 2 * position.x)
                {
                    x = 0 - xOffset - pmSize.width;
                }
            }
            else
            {
                // First determine x:
                x = 0 - xOffset - pmSize.width; // Prefer placement to the left
                if (position.x + x < 0 &&
                    // popup doesn't fit - place it wherever there's more room
                    screenSize.width - s.width > 2 * position.x)
                {
                    x = s.width + xOffset;
                }
            }
            // Then the y:
            y = yOffset; // Prefer dropping down
            if (position.y + y + pmSize.height >= screenSize.height &&
                // popup doesn't fit - place it wherever there's more room
                screenSize.height - s.height < 2 * position.y)
            {
                y = s.height - yOffset - pmSize.height;
            }
        }
        else
        {
            // We are a toplevel menu (pull-down)
            int xOffset = UIManager.getInt("Menu.menuPopupOffsetX");
            int yOffset = UIManager.getInt("Menu.menuPopupOffsetY");

            if (origin.getComponentOrientation().isLeftToRight())
            {
                // First determine the x:
                x = xOffset; // Extend to the right
                if (position.x + x + pmSize.width >= screenSize.width &&
                    // popup doesn't fit - place it wherever there's more room
                    screenSize.width - s.width < 2 * position.x)
                {
                    x = s.width - xOffset - pmSize.width;
                }
            }
            else
            {
                // First determine the x:
                x = s.width - xOffset - pmSize.width; // Extend to the left
                if (position.x + x < 0 &&
                    // popup doesn't fit - place it wherever there's more room
                    screenSize.width - s.width > 2 * position.x)
                {
                    x = xOffset;
                }
            }
            // Then the y:
            y = s.height + yOffset; // Prefer dropping down
            if (position.y + y + pmSize.height >= screenSize.height &&
                // popup doesn't fit - place it wherever there's more room
                screenSize.height - s.height < 2 * position.y)
            {
                y = 0 - yOffset - pmSize.height; // Otherwise drop 'up'
            }
        }

        return new Point(x, y);
    }

    /**
     * For some reason this method is defined in AbstractButton but given
     * pacakge scoping so we can't use it in our subclasses!!!
     */
    public static final void configurePropertiesFromAction(AbstractButton btn, Action a, String[] types)
    {
        if (types == null)
        {
            String[] alltypes =
            {
                Action.MNEMONIC_KEY,
                Action.NAME,
                Action.SHORT_DESCRIPTION,
                Action.SMALL_ICON,
                Action.ACTION_COMMAND_KEY,
                "enabled"
            };
            types = alltypes;
        }

        for (int i = 0; i < types.length; i++)
        {
            String type = types[i];

            if (type == null)
            {
                continue;
            }

            if (type.equals(Action.MNEMONIC_KEY))
            {
                Integer n = (a == null) ? null : (Integer) a.getValue(type);
                btn.setMnemonic((n == null) ? '\0' : n.intValue());
            }
            else if (type.equals(Action.NAME))
            {
                btn.setText(a != null ? (String) a.getValue(type) : null);
            }
            else if (type.equals(Action.SHORT_DESCRIPTION))
            {
                btn.setToolTipText(a != null ? (String) a.getValue(type) : null);
            }
            else if (type.equals(Action.SMALL_ICON))
            {
                btn.setIcon(a != null ? (Icon) a.getValue(type) : null);
            }
            else if (type.equals(Action.ACTION_COMMAND_KEY))
            {
                btn.setActionCommand(a != null ? (String) a.getValue(type) : null);
            }
            else if (type.equals(Action.ACCELERATOR_KEY))
            {
                KeyStroke ks = (a == null) ? null : (KeyStroke) a.getValue(type);

                if ((btn instanceof JMenuItem) && (!(btn instanceof JMenu)))
                {
                    ((JMenuItem) btn).setAccelerator((ks == null) ? null : ks);
                }
            }
            else if (type.equals("enabled"))
            {
                btn.setEnabled(a != null ? a.isEnabled() : true);
            }
        }
    }

    public static final String getDateTimeString(Long value)
    {
        return getDateTimeString(value, "n/a");
    }

    public static final String getDateTimeString(Long value, String defValue)
    {
        if ((value == null) || (value.longValue() == 0) || (value.longValue() == -1))
        {
            return defValue;
        }

        return DATE_PARSER_THREAD_LOCAL.get().format(value);
    }

    public static final TableCellRenderer getDateTimeTableCellRenderer()
    {
        return TableCellRendererHolder.INSTANCE;
    }

    public static GridBagConstraints setConstraints(GridBagConstraints gbc,
                                             int gridx,
                                             int gridy,
                                             int gridwidth,
                                             int gridheight,
                                             double weightx,
                                             double weighty,
                                             int anchor,
                                             int fill)
    {
        return setConstraints(gbc, gridx, gridy, gridwidth, gridheight,
                              weightx, weighty,
                              anchor, fill, gbc.insets, gbc.ipadx, gbc.ipady);
    }

    public static GridBagConstraints setConstraints(GridBagConstraints gbc,
                                             int gridx,
                                             int gridy,
                                             int gridwidth,
                                             int gridheight,
                                             double weightx,
                                             double weighty,
                                             int anchor,
                                             int fill,
                                             Insets insets,
                                             int ipadx,
                                             int ipady)
    {
        gbc.gridx = gridx;
        gbc.gridy = gridy;
        gbc.gridwidth = gridwidth;
        gbc.gridheight = gridheight;
        gbc.weightx = weightx;
        gbc.weighty = weighty;
        gbc.anchor = anchor;
        gbc.fill = fill;
        gbc.insets = insets;
        gbc.ipadx = ipadx;
        gbc.ipady = ipady;

        return gbc;
    }

    public static String toHtml(String sText)
    {
        StringBuffer sbValue = new StringBuffer();

        sbValue.append(PROLOG);
        sbValue.append(sText);
        sbValue.append(EPILOG);

        return sbValue.toString();
    }
    public static Frame getFrame (Component c)
    {
        if (c == null)
        {
            return (null);
        }

        Component frame = (c == null ? null : c);

        while ((frame != null) && !(frame instanceof Frame))
        {
            frame = frame.getParent();
        }

        return (Frame) frame;
    }

    public static JFrame getJFrame (Component c)
    {
        if (c == null)
        {
            return (null);
        }

        Component frame = (c == null ? null : c);

        while ((frame != null) && !(frame instanceof JFrame))
        {
            frame = frame.getParent();
        }

        return (JFrame) frame;
    }

    public static Window getWindow (Component c)
    {
        if (c == null)
        {
            return (null);
        }

        Component frame = (c == null ? null : c);

        while ((frame != null) && !(frame instanceof Window))
        {
            frame = frame.getParent();
        }

        return (Window) frame;
    }

    public static JDialog getJDialog (Component c)
    {
        Component dialog = c;
        do
        {
            dialog = dialog.getParent();
        }
        while ((dialog != null) && !(dialog instanceof JDialog));

        return (JDialog) dialog;
    }

    public static Dialog getDialog (Component c)
    {
        if (c == null)
        {
            return (null);
        }

        Component dialog = c;
        do
        {
            dialog = dialog.getParent();
        }
        while ((dialog != null) && !(dialog instanceof Dialog));

        return (Dialog) dialog;
    }

    public static String getTitle (Component c)
    {
        if (c == null)
        {
            return ("");
        }

        Component frame = (c == null ? null : c);

        while ((frame != null) && !(frame instanceof Frame) && !(frame instanceof Dialog))
        {
            frame = frame.getParent();
        }

        if (frame == null)
        {
            return (c.getName() == null ? "" : c.getName());
        }

        if (frame instanceof Frame)
        {
            return (((Frame)frame).getTitle());
        }
        else
        {
            return (((Dialog)frame).getTitle());
        }
    }

    public static void enableTextComponent(JTextComponent t, boolean bEnable)
    {
        t.setEditable(bEnable);
        if(bEnable)
        {
            t.setBackground(UIManager.getColor("TextField.background"));
        }
        else
        {
            t.setBackground(UIManager.getColor("Label.background"));
        }
    }

    public static void equalizeByClass (Container c)
    {
        Component[] aComponents = c.getComponents();
        int         ns = aComponents.length;
        HashMap     map = new HashMap();

        // First, find all the unique classes.
        for (int n = 0; n < ns; n++)
        {
            Object array;

            // Make sure to skip labels.
            if (aComponents[n] instanceof JLabel)
            {
                if (((JLabel)aComponents[n]).getLabelFor() != null)
                {
                    continue;
                }
            }

            if ((array = map.get (aComponents[n].getClass())) == null)
            {
                ArrayList a = new ArrayList();
                a.add (aComponents[n]);
                map.put (aComponents[n].getClass(), a);
            }
            else
            {
                ArrayList a = (ArrayList)array;
                a.add (aComponents[n]);
            }
        }

        // Now, iterate the collections.
        Iterator i = map.values().iterator();
        while (i.hasNext())
        {
            ArrayList a = (ArrayList)i.next();
            equalize (a);
        }

        // Now do the overall layout.
        c.doLayout();
    }

    public static void equalize (Container c)
    {
        equalize(c.getComponents());

        // Now tell the layout manager to redo its deal.
        c.doLayout();
    }

    public static void equalize (Component[] components)
    {
        int         nPreferredWidth = 0;

        // First, find the max prefered width.
        int nComponents = components.length;
        for (int nComponent = 0; nComponent < nComponents; nComponent++)
        {
            Dimension d = components[nComponent].getPreferredSize();

            if (d.getWidth() > nPreferredWidth)
            {
                nPreferredWidth = (int)d.getWidth();
            }
        }

        // Now set the width in all of them.
        for (int nComponent = 0; nComponent < nComponents; nComponent++)
        {
            Dimension d = components[nComponent].getPreferredSize();

            Dimension dNew = new Dimension (nPreferredWidth, (int)d.getHeight());

            sizeComponent (components[nComponent], dNew);
        }
    }

    public static void equalize (ArrayList components)
    {
        int         nPreferredWidth = 0;

        // First, find the max prefered width.
        int nComponents = components.size();
        for (int nComponent = 0; nComponent < nComponents; nComponent++)
        {
            Dimension d = ((Component)components.get(nComponent)).getPreferredSize();

            if (d.getWidth() > nPreferredWidth)
            {
                nPreferredWidth = (int)d.getWidth();
            }
        }

        // Now set the width in all of them.
        for (int nComponent = 0; nComponent < nComponents; nComponent++)
        {
            Dimension d = ((Component)components.get (nComponent)).getPreferredSize();

            Dimension dNew = new Dimension (nPreferredWidth, (int)d.getHeight());

            sizeComponent (((Component)components.get (nComponent)), dNew);
        }
    }

    public static void sizeComponent (Component c, Dimension d)
    {
        if (c instanceof JComponent)
        {
            JComponent j = (JComponent)c;
            j.setPreferredSize (d);
            j.setMinimumSize (d);
            j.setMaximumSize (d);
        }
        else
        {
            c.setSize (d);
        }
    }

    public static void equalize (Component c1, Component c2)
    {
        // First, find the max prefered width.
        Dimension d1 = c1.getPreferredSize();
        Dimension d2 = c2.getPreferredSize();
        int nPreferredWidth = Math.max (d1.width, d2.width);

        // Now size them.
        sizeComponent (c1, new Dimension (nPreferredWidth, d1.height));
        sizeComponent (c2, new Dimension (nPreferredWidth, d2.height));
    }

    static public JLabel getLabelFor (Container c, Component comp)
    {
        Component [] aComponents = c.getComponents();
        for (int n = 0; n < aComponents.length; n++)
        {
            Component comp1 = aComponents[n];
            if (comp1 instanceof JLabel &&
                ((JLabel)comp1).getLabelFor() == comp)
            {
                return ((JLabel)comp1);
            }
        }

        return (null);
    }

    static public void invoke (Runnable r, boolean bWait)
    {
        try
        {
            if (bWait)
            {
                if (SwingUtilities.isEventDispatchThread())
                {
                    r.run();
                }
                else
                {
                    SwingUtilities.invokeAndWait (r);
                }
            }
            else
            {
                SwingUtilities.invokeLater (r);
            }
        }
        catch (Exception ex)
        {
            ex.printStackTrace(System.err);
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, ex.getMessage(), ex, false);    // Log the error msg.
        }
    }

    static public void invoke (Runnable r)
    {
        invoke(r, false);
    }

    /**
     * Executes the runnable asynchronously in a <b>non UI<b> thread from SMC's general purpose thread pool.
     * If the runnble should be run in the UI thread, use Helper.invoke(...) instead 
     * @param r
     */
    static public void asyncExec(Runnable r)
    {
        threadPool.execute(r);
    }

    public static String formatToHTMLForWrapping(String strInput, int nMaxWidth)
    {
        StringBuffer sbOutput = new StringBuffer();

        if (strInput.length() > nMaxWidth)
        {
            StringBuffer sbTemp = new StringBuffer();
            StringTokenizer st = new StringTokenizer(strInput, " ");

            while (st.hasMoreTokens())
            {
                String strToken = st.nextToken();

                if (sbTemp.length() + strToken.length() > nMaxWidth)
                {
                    sbTemp.append("<BR>");
                    sbOutput.append(sbTemp.toString());
                    sbTemp = new StringBuffer();
                }

                if (sbTemp.length() != 0)
                {
                    sbTemp.append(" ");
                }

                sbTemp.append(strToken);
            }

            if (sbTemp.length() > 0)
            {
                // no more tokens, temp is the completion.
                sbOutput.append(sbTemp.toString());
            }
        }
        else
        {
            sbOutput.append(strInput);
        }

        return toHtml(sbOutput.toString());
    }

    static class DateTimeTableCellRenderer
        extends ExtendedJTable.ExtendedTableCellRenderer
    {
        public DateTimeTableCellRenderer()
        {
            super();
            setHorizontalAlignment();
        }

        private void setHorizontalAlignment() {
            setHorizontalAlignment(SwingConstants.LEFT);
        }
        
        @Override
        public Component getTableCellRendererComponent(JTable table,
            Object value,
            boolean isSelected,
            boolean hasFocus,
            int row,
            int column)
        {
            String stringValue = null;

            try
            {
                stringValue = Helper.getDateTimeString((Long) value);
            }
            catch (Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
            }

            return super.getTableCellRendererComponent(table, stringValue,
                isSelected, hasFocus, row, column);
        }

    }

    /**
     * A simple, static class to display a URL in the system browser.
     *
     * Under Unix, the system browser is hard-coded to be 'netscape'.
     * Netscape must be in your PATH for this to work.  This has been
     * tested with the following platforms: AIX, HP-UX and Solaris.
     *
     * Under Windows, this will bring up the default browser under windows,
     * usually either Netscape or Microsoft IE.  The default browser is
     * determined by the OS.  This has been tested under Windows 95/98/NT.
     *
     * Examples:
     * BrowserControl.displayURL("http://www.javaworld.com")
     * BrowserControl.displayURL("file://c:\\docs\\index.html")
     * BrowserContorl.displayURL("file:///user/joe/index.html");
     *
     * Note - you must include the url type -- either "http://" or * "file://".
     */
    public static final void displayURL(String url)
    {
        boolean windows = isWindowsPlatform();
        String[] cmd = null;

        if (windows)
        {
            cmd = new String[]
                  {WIN_PATH, WIN_FLAG, url};
        }
        else
        {
            cmd = new String[]
                  {UNIX_PATH, UNIX_FLAG + "(" + url + ")"};
        }

        try
        {
            Runtime.getRuntime().exec(cmd);
        }
        catch (IOException e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
        }
    }

    /**
     * Try to determine whether this application is running under Windows
     * or some other platform by examing the "os.name" property.
     *
     * @return true if this application is running under a Windows OS
     */
    public static boolean isWindowsPlatform()
    {
        String os = System.getProperty("os.name");

        return ((os != null) && os.startsWith("Windows"));
    }

    // The default system browser under windows.
    private static final String WIN_PATH = "rundll32";
    // The flag to display a url.
    private static final String WIN_FLAG = "url.dll,FileProtocolHandler";
    // The default browser under unix.
    private static final String UNIX_PATH = "netscape";
    // The flag to display a url.
    private static final String UNIX_FLAG = "-remote openURL";
}