/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corporation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui.util;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.ContainerEvent;
import java.awt.event.ContainerListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.image.BufferedImage;
import java.beans.PropertyVetoException;
import java.util.ArrayList;
import java.util.Vector;

import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.DefaultDesktopManager;
import javax.swing.DefaultListCellRenderer;
import javax.swing.DesktopManager;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComponent;
import javax.swing.JDesktopPane;
import javax.swing.JInternalFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JViewport;
import javax.swing.ListSelectionModel;
import javax.swing.Scrollable;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.event.InternalFrameAdapter;
import javax.swing.event.InternalFrameEvent;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import com.sonicsw.ma.gui.MgmtConsole;

public class JMADesktopPane
    extends    JDesktopPane
    implements ComponentListener, ContainerListener, Scrollable
{
  private final static int LAYOUT_NONE     = 0;
  private final static int LAYOUT_CASCADE  = 1;
  private final static int LAYOUT_TILE_HORIZONTAL = 2;
  private final static int LAYOUT_TILE_VERTICAL   = 3;
  private final static int DEFAULT_OFFSETX = 24;
  private final static int DEFAULT_OFFSETY = 24;

  private JScrollPane   m_scrollpane;
  private int           nextX;
  private int           nextY;
  private int           currentLayoutSetting = LAYOUT_NONE;
  private boolean       recomputeSizeInhibited = false;
  private ArrayList     internalFrames;
  private JMenu         m_menuWindow;
  private Vector        m_fileList = new Vector();
  private TileHorizontalAction m_tileHorizontalAction;
  private TileVerticalAction   m_tileVerticalAction;
  private CascadeAction m_cascadeAction;
  private InternalFramesWatcher internalFramesWatcher;


  public JMADesktopPane()
  {
    super();
    prepareDesktopPane();
    
  }
  private void prepareDesktopPane() {
      
    addContainerListener(this);
    internalFrames = new ArrayList();

    setBackground((Color)Toolkit.getDefaultToolkit().getDesktopProperty("win.mdi.backgroundColor"));

    m_menuWindow = new JMenu("Window")
    {
        @Override
        public JMenuItem add(JMenuItem menuItem)
        {
            if (getItemCount() < 14)
            {
                // The following test assumes that the basic Window menu only has
                // two menu items, two for Tile and one for Cascade.
                if (getItemCount() == 3)
                {
                    addSeparator();
                }

                if (menuItem.getAction() != null)
                {
                    if (menuItem.getAction() instanceof OpenInternalFrameAction)
                    {
                        OpenInternalFrameAction action = (OpenInternalFrameAction)menuItem.getAction();
                        JBasicCheckBoxMenuItem checkBoxMenuItem = new JBasicCheckBoxMenuItem(action);
                        checkBoxMenuItem.setUserObject(action.getInternalFrame());
                        checkBoxMenuItem.setIcon(action.getInternalFrame().getFrameIcon());
                        checkBoxMenuItem.addItemListener(new ItemListener()
                        {
                            @Override
                            public void itemStateChanged(ItemEvent event)
                            {
                                // If the user selects an already active frame from
                                // the window list, leave it's menu item selected.
                                JBasicCheckBoxMenuItem menuItem = (JBasicCheckBoxMenuItem)event.getItem();
                                JInternalFrame frame = (JInternalFrame)menuItem.getUserObject();
                                if (frame.isSelected() && event.getStateChange() == ItemEvent.DESELECTED)
                                {
                                    menuItem.setSelected(true);
                                }
                            }
                        });

                        m_fileList.add(checkBoxMenuItem);

                        return super.add(checkBoxMenuItem);
                    }
                    else
                    {
                        return super.add(menuItem);
                    }
                }
                else
                {
                    return super.add(menuItem);
                }
            }
            else
            {
                OpenInternalFrameAction action = (OpenInternalFrameAction)menuItem.getAction();
                JBasicCheckBoxMenuItem checkBoxMenuItem = new JBasicCheckBoxMenuItem(action);
                checkBoxMenuItem.setUserObject(action.getInternalFrame());
                checkBoxMenuItem.addItemListener(new ItemListener()
                {
                    @Override
                    public void itemStateChanged(ItemEvent event)
                    {
                        // If the user selects an already active frame from
                        // the window list, leave it's menu item selected.
                        JBasicCheckBoxMenuItem menuItem = (JBasicCheckBoxMenuItem)event.getItem();
                        JInternalFrame frame = (JInternalFrame)menuItem.getUserObject();
                        if (frame.isSelected() && event.getStateChange() == ItemEvent.DESELECTED)
                        {
                            menuItem.setSelected(true);
                        }
                    }
                });

                m_fileList.add(checkBoxMenuItem);

                if (getItemCount() == 14)
                {
                    return super.add(new JBasicMenuItem(new WindowFileListAction()));
                }

                return null;
            }
        }

        @Override
        public void remove(JMenuItem menuItem)
        {
            if (isMenuComponent(menuItem))
            {
                super.remove(menuItem);
            }

            m_fileList.remove(menuItem);

            // If we only have 10 JInternalFrame windows left after the remove,
            // rebuild the Window menu File List without the "More..." menu item.
            if (m_fileList.size() == 10 || getMenuComponents().length < 15)
            {
                // Leave Tile & Cascade on Window menu.
                Component[] components = getMenuComponents();
                for (int i = 0; i < components.length; i++)
                {
                    if (components[i] instanceof JPopupMenu.Separator ||
                        components[i] instanceof JBasicCheckBoxMenuItem)
                    {
                        super.remove(components[i]);
                    }
                    else
                    {
                        Action action = ((JMenuItem)components[i]).getAction();
                        if (action != null)
                        {
                            String key = (String)action.getValue(Action.ACTION_COMMAND_KEY);
                            if (key != null && key.equals("JMADesktopPane.WindowFileList"))
                            {
                                super.remove(components[i]);
                            }
                        }
                    }
                }

                Vector oldFileList = (Vector)m_fileList.clone();
                m_fileList.removeAllElements();

                for (int i = 0; i < oldFileList.size(); i++)
                {
                    add((JCheckBoxMenuItem)oldFileList.get(i));
                }
            }
            // The following test assumes that the basic Window menu only has
            // four menu items, two for Tile, one for Cascade, and one for the
            // separator added when the first internal frame was added.
            else if (getItemCount() == 4)
            {
                Component[] components = getMenuComponents();

                for (int i = components.length - 1; i >= 0; i--)
                {
                    if (components[i] instanceof JPopupMenu.Separator)
                    {
                        super.remove(components[i]);
                    }
                }
            }
        }
    };

    m_tileHorizontalAction = new TileHorizontalAction();
    m_tileHorizontalAction.setEnabled(false);
    m_menuWindow.add(new JBasicMenuItem(m_tileHorizontalAction));

    m_tileVerticalAction = new TileVerticalAction();
    m_tileVerticalAction.setEnabled(false);
    m_menuWindow.add(new JBasicMenuItem(m_tileVerticalAction));

    m_cascadeAction = new CascadeAction();
    m_cascadeAction.setEnabled(false);
    m_menuWindow.add(new JBasicMenuItem(m_cascadeAction));

    internalFramesWatcher = new InternalFramesWatcher();
    setDoubleBuffered(true);

    maInitialize();

    // Prohibit dragging an internal frame window completely off the edge of the
    // desktop pane.  Since JScrollPane won't handle scrolling to the left or top
    // properly, we won't use scrollbars at all.  The user can still move the
    // internal frame mostly off the edge of the desktop pane but not completely.
    // Note: This mimics the MDI behavior in Jasc Paint Shop Pro 7.0...
    setDesktopManager(new DefaultDesktopManager()
    {
        @Override
        public void dragFrame(JComponent comp,int x,int y)
        {
            Rectangle viewPort = m_scrollpane.getViewport().getBounds();
            Dimension size = comp.getSize();

            if (x > viewPort.x + viewPort.width - 22)
            {
                x = viewPort.x + viewPort.width - 22;
            }
            else if (x + size.width < viewPort.x + 22)
            {
                x = x + ((viewPort.x + 22) - (x + size.width));
            }

            if (y > viewPort.y + viewPort.height - 22)
            {
                y = viewPort.y + viewPort.height - 22;
            }
            else if (y < viewPort.y - 22)
            {
                y = viewPort.y - 22;
            }

            super.dragFrame(comp,x,y);
        }
    });
  }

  public void updateWindowMenuFileList()
  {
  }

  public void maInitialize()
  {
  }

  public void maCleanup()
  {
      JInternalFrame[] frames = getAllFrames();

      for(int i = 0; i < frames.length; i++)
      {
          if (frames[i] instanceof JMAInternalFrame)
        {
            ((JMAInternalFrame)frames[i]).internalFrameClosed(null);
        }
      }
  }

  public JInternalFrame getFrameWithTitle(String title)
  {
      JInternalFrame[] frames = getAllFrames();

      for (int i = 0; i < frames.length; i++)
    {
        if (frames[i].getTitle().equals(title))
        {
            return frames[i];
        }
    }

      return null;
  }

  public JMenu getInternalFramesMenu()
  {
    return m_menuWindow;
  }

  /**
   * Method to select/deselect the menu item for the active internal frame.
   *
   * @param active Flag indicating activate or deactivate
   * @param frame The frame being activated or deactivated
   */
  public void setActive(boolean active,JInternalFrame frame)
  {
    for (int i = 0; i < m_menuWindow.getItemCount(); i++)
    {
      if (m_menuWindow.getItem(i) instanceof JBasicCheckBoxMenuItem)
      {
        JBasicCheckBoxMenuItem menuItem = (JBasicCheckBoxMenuItem)m_menuWindow.getItem(i);
        JInternalFrame userObject = (JInternalFrame)menuItem.getUserObject();
        if (userObject != null && userObject == frame)
        {
            // Set the active window in the Window menu File List.
            menuItem.setSelected(active);
            break;
        }
      }
    }

    for (int i = 0; i < m_fileList.size(); i++)
    {
      if (m_fileList.get(i) instanceof JBasicCheckBoxMenuItem)
      {
        JBasicCheckBoxMenuItem menuItem = (JBasicCheckBoxMenuItem)m_fileList.get(i);
        JInternalFrame userObject = (JInternalFrame)menuItem.getUserObject();
        if (userObject != null && userObject == frame)
        {
            // Set the active window in the Window dialog File List.
            menuItem.setSelected(active);
        }
      }
    }
  }

  /**
   * This method allows child frames to be added with automatic cascading.
   */
  public void addCascaded(Component comp)
  {
    // First add the component in the correct layer
    super.add(comp);

    // Now do the cascading
    if (comp instanceof JInternalFrame)
    {
        this.cascade(comp);
    }

    // Move it to the front
    this.moveToFront(comp);
  }

  // Layout all of the children of this container so that they are cascaded.
  public void cascadeAll()
  {
    Component[] comps = getComponents();
    int count = comps.length;

    nextX = 0;
    nextY = 0;

    for (int i = count - 1; i >= 0; i--)
    {
      Component comp = comps[i];

      if (comp instanceof JInternalFrame && comp.isVisible())
      {
        try
        {
          // Restore frame from maximized or iconified state before cascading,
          // otherwise bad things will happen.
          if (((JInternalFrame)comp).isMaximum())
        {
            ((JInternalFrame)comp).setMaximum(false);
        }

          if (((JInternalFrame)comp).isIcon())
        {
            ((JInternalFrame)comp).setIcon(false);
        }

          cascade(comp);
        }
        catch (PropertyVetoException e)
        {
          MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR, "Error while cascading components.", e, true);
        }
      }
    }
    setCurrentLayoutSetting(LAYOUT_CASCADE);
  }

  /**
   * Layout all of the children of this container
   * so that they are tiled.
   */
  public void tileAll(boolean vertical)
  {
    DesktopManager manager = getDesktopManager();
    if (manager == null)
    {
      // No desktop manager - do nothing
      return;
    }

    Component[] comps = getComponents();
    Component comp;
    int count = 0;

    // Count and handle only the internal frames
    for (int i = 0; i < comps.length; i++)
    {
      comp = comps[i];

      if (comp instanceof JInternalFrame && comp.isVisible())
      {
        try
        {
          // Restore frame from maximized or iconified state before cascading,
          // otherwise bad things will happen.
          if (((JInternalFrame)comp).isMaximum())
        {
            ((JInternalFrame)comp).setMaximum(false);
        }

          if (((JInternalFrame)comp).isIcon())
        {
            ((JInternalFrame)comp).setIcon(false);
        }
        }
        catch (PropertyVetoException e)
        {
          MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR, "Error while tiling components.", e, true);
        }
        count++;
      }
    }

    if (count != 0)
    {
      int rows    = (int)Math.sqrt((double)count);
      int columns = count/rows;
      int spares = count - (columns * rows);

      Dimension paneSize = m_scrollpane.getViewport().getSize();

      if (vertical == false)
      {
          int columnHeight = paneSize.height/columns;
          int availableWidth = paneSize.width;
          int mainWidth = availableWidth/rows;
          int smallerWidth = availableWidth/(rows + 1);
          int rowWidth = mainWidth;
          int x = 0;
          int y = 0;
          int thisRow = rows;
          int normalColumns = columns - spares;

          for (int i = comps.length - 1; i >= 0; i--)
          {
            comp = comps[i];

            if (comp instanceof JInternalFrame && comp.isVisible())
            {
              manager.setBoundsForFrame((JComponent)comp, x, y, rowWidth, columnHeight);
              x += rowWidth;
              if (--thisRow == 0)
              {
                // Filled the row
                y += columnHeight;
                x = 0;

                // Switch to smaller rows if necessary
                if (--normalColumns <= 0)
                {
                  thisRow = rows + 1;
                  rowWidth = smallerWidth;
                }
                else
                {
                  thisRow = rows;
                }
              }
            }
          }
      }
      else
      {
          int columnWidth = paneSize.width/columns;

          int availableHeight = paneSize.height;
          int mainHeight = availableHeight/rows;
          int smallerHeight = availableHeight/(rows + 1);
          int rowHeight = mainHeight;
          int x = 0;
          int y = 0;
          int thisRow = rows;
          int normalColumns = columns - spares;

          for (int i = comps.length - 1; i >= 0; i--)
          {
            comp = comps[i];

            if (comp instanceof JInternalFrame && comp.isVisible())
            {
              manager.setBoundsForFrame((JComponent)comp, x, y, columnWidth, rowHeight);
              y += rowHeight;
              if (--thisRow == 0)
              {
                // Filled the row
                y = 0;
                x += columnWidth;

                // Switch to smaller rows if necessary
                if (--normalColumns <= 0)
                {
                  thisRow = rows + 1;
                  rowHeight = smallerHeight;
                }
                else
                {
                  thisRow = rows;
                }
              }
            }
          }
        }
    }

    setCurrentLayoutSetting(vertical ? LAYOUT_TILE_VERTICAL : LAYOUT_TILE_HORIZONTAL);
  }

  /**
   *  Place a component so that it is cascaded
   *  relative to the previous one
   */
  public void cascade(Component comp)
  {
    Dimension paneSize = m_scrollpane.getViewport().getSize();

    int targetWidth  = 3 * paneSize.width/4;
    int targetHeight = 3 * paneSize.height/4;

    DesktopManager manager = getDesktopManager();

    if (manager == null)
    {
      comp.setBounds(0, 0, targetWidth, targetHeight);
      return;
    }

    if (nextX + targetWidth > paneSize.width || nextY + targetHeight > paneSize.height)
    {
      nextX = 0;
      nextY = 0;
    }

    manager.setBoundsForFrame((JComponent)comp, nextX, nextY, targetWidth, targetHeight);

    nextX += DEFAULT_OFFSETX;
    nextY += DEFAULT_OFFSETY;
  }

  /**
   * This method is called when the desktop pane is resized or moved, as well as
   * when an internal frame is added, removed, resized, moved, hidden, or shown.
   * It resizes the viewport of the desktop pane accordingly to accommodate all
   * it's visible internal frame window sizes and positions.
   */
  protected void recomputeSize()
  {
    if (!recomputeSizeInhibited)
    {
      recomputeSizeInhibited = true;

      Dimension viewportSize = ((m_scrollpane == null) ? new Dimension(0,0) :
        m_scrollpane.getViewport().getSize());

      int width  = viewportSize.width;
      int height = viewportSize.height;

      Component comps[] = getComponents();

      boolean maximized = false;

      for (int i = 0, n = comps.length; i < n; i++)
      {
        Component comp = comps[i];

        if (comp.isVisible())
        {
          // If the internal frame is maximized, force the desktop pane, as well
          // as the internal frame, to resize.
          if (comp instanceof JInternalFrame && ((JInternalFrame)comp).isMaximum())
          {
            width = viewportSize.width;
            height = viewportSize.height;
            maximized = true;
            break;
          }
          // Calculate the new width and height of the desktop pane.
          else
          {
            Rectangle compRect = comp.getBounds();
            int compRightX = compRect.x + compRect.width;
            int compBottomY = compRect.y + compRect.height;

            Rectangle parentRect = getBounds();
            int parentRightX = parentRect.x + parentRect.width;
            int parentBottomY = parentRect.y + parentRect.height;

            if (parentRect.contains(compRect) && (parentRect.width != getSize().width)
                && (parentRect.height != getSize().height))
            {
                maximized = true;
            }
            else
            {
              if (compRightX > parentRightX)
            {
                width += (compRightX - parentRightX);
            }
            else if (compRect.x < parentRect.x)
            {
                width += (parentRect.x - compRect.x);
            }
            else
            {
                width = Math.max(width,compRect.x + compRect.width);
            }

              if (compBottomY > parentBottomY)
            {
                height += (compBottomY - parentBottomY);
            }
            else if (compRect.y < parentRect.y)
            {
                height += (parentRect.y - compRect.y);
            }
            else
            {
                height = Math.max(height,compRect.y + compRect.height);
            }
            }
          }
        }
      }

      Dimension size = getSize();

      if (maximized || (width != size.width) || (height != size.height))
      {
        Dimension newSize = new Dimension(width, height);
        setSize(newSize);
        setPreferredSize(newSize);
      }

      recomputeSizeInhibited = false;
    }
  }

    //-------------------------------------------------------------------------
    //
    // ContainerListener implementation
    //
    //-------------------------------------------------------------------------

    @Override
    public void componentAdded(ContainerEvent event)
    {
        Component comp = event.getChild();
        comp.addComponentListener(this);

        if (comp instanceof JInternalFrame)
        {
            if (!internalFrames.contains(comp))
            {
                JInternalFrame iframe = (JInternalFrame)comp;
                internalFrames.add(iframe);

                m_menuWindow.add(new OpenInternalFrameAction(iframe));

                iframe.addInternalFrameListener(internalFramesWatcher);
            }
        }

        recomputeSize();

        m_tileHorizontalAction.setEnabled(true);
        m_tileVerticalAction.setEnabled(true);
        m_cascadeAction.setEnabled(true);
    }

    @Override
    public void componentRemoved(ContainerEvent event)
    {
        Component comp = event.getChild();
        comp.removeComponentListener(this);
        recomputeSize();

        if (getAllFrames().length < 1)
        {
            m_tileHorizontalAction.setEnabled(false);
            m_tileVerticalAction.setEnabled(false);
            m_cascadeAction.setEnabled(false);
        }

    }

    //-------------------------------------------------------------------------
    //
    // ComponentListener implementation
    //
    //-------------------------------------------------------------------------

    @Override
    public void componentResized(ComponentEvent evt)
    {
        Component c = evt.getComponent();

        if (c instanceof JViewport)
        {
            recomputeSize();
        }

        // Force a maximized internal frame to resize to accommodate the current
        // size of the desktop pane.
        if (c instanceof JInternalFrame)
        {
            if (((JInternalFrame)c).isMaximum())
            {
                if (m_scrollpane != null)
                {
                    Dimension viewPortSize = m_scrollpane.getViewport().getSize();
                    Dimension compSize = c.getSize();
                    if (compSize.width > viewPortSize.width || compSize.height >
                        viewPortSize.height)
                    {
                        ((JInternalFrame)c).setMaximumSize(viewPortSize);
                    }
                }
            }
        }
    }

    @Override
    public void componentMoved(ComponentEvent evt)
    {
        componentHidden(evt);
    }

    @Override
    public void componentShown(ComponentEvent evt)
    {
        componentHidden(evt);
    }

    @Override
    public void componentHidden(ComponentEvent evt)
    {
        if (evt.getComponent() instanceof JViewport)
        {
            recomputeSize();
        }
    }

    //-------------------------------------------------------------------------
    //
    // Scrollable implementation
    //
    //-------------------------------------------------------------------------

    @Override
    public Dimension getPreferredScrollableViewportSize()
    {
        return new Dimension(700, 500);
    }

    @Override
    public int getScrollableUnitIncrement(Rectangle visibleRect,
                                          int orientation, int direction)
    {
        boolean vertical = (orientation == SwingConstants.VERTICAL);

        return (vertical ? 25 : 40);
    }

    @Override
    public int getScrollableBlockIncrement(Rectangle visibleRect,
                                           int orientation, int direction)
    {
        int fullSize = ((orientation == SwingConstants.VERTICAL) ?
                        visibleRect.height : visibleRect.width);

        return Math.max(10, (int)(fullSize * 0.9));
    }

    @Override
    public boolean getScrollableTracksViewportWidth()
    {
        return false;
    }

    @Override
    public boolean getScrollableTracksViewportHeight()
    {
        return false;
    }

    //-------------------------------------------------------------------------

    @Override
    public void setBounds(int x, int y, int w, int h)
    {
        if ((x != getX()) || (y != getY()) || (w != getWidth()) || (h != getHeight()))
        {
            super.setBounds(x, y, w, h);

            recomputeSize();
        }
    }

  private void refreshSettings()
  {
    if (getCurrentLayoutSetting() == LAYOUT_CASCADE)
    {
        cascadeAll();
    }
    else
    if (getCurrentLayoutSetting() == LAYOUT_TILE_HORIZONTAL)
    {
        tileAll(false);
    }
    else
    if (getCurrentLayoutSetting() == LAYOUT_TILE_VERTICAL)
    {
        tileAll(true);
    }
  }

  private void setCurrentLayoutSetting(int setting)
  {
    currentLayoutSetting = setting;
  }

  private int getCurrentLayoutSetting()
  {
    return currentLayoutSetting;
  }

  @Override
public void addNotify()
  {
    super.addNotify();

    // if we're in a JScrollPane (and we probably are) keep track of it
    JScrollPane sp = (JScrollPane)SwingUtilities.getAncestorOfClass(JScrollPane.class, this);

    if ((sp != null) && (getParent().getParent() == sp))
    {
      m_scrollpane = sp;
      m_scrollpane.getViewport().addComponentListener(this);

      // Don't use the scrollbars for the desktop pane's scrollpane as JScrollPane
      // does not handle scrolling an MDI internal frame off the left or top of
      // the desktop pane correctly.
      m_scrollpane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
      m_scrollpane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_NEVER);
    }
  }

  private void removeMenuItemFor(JInternalFrame internalFrame)
  {
    boolean found = false;

    for (int i = 0; i < m_menuWindow.getItemCount(); i++)
    {
      if (m_menuWindow.getItem(i) instanceof JBasicCheckBoxMenuItem)
      {
        JBasicCheckBoxMenuItem menuItem = (JBasicCheckBoxMenuItem)m_menuWindow.getItem(i);
        JInternalFrame userObject = (JInternalFrame)menuItem.getUserObject();
        if (userObject != null && userObject == internalFrame)
        {
            m_menuWindow.remove(menuItem);
            found = true;
            break;
        }
      }
    }

    // If not found in the Window menu File List, search the Window dialog File List.
    if (!found)
    {
      for (int i = 0; i < m_fileList.size(); i++)
      {
        if (m_fileList.get(i) instanceof JBasicCheckBoxMenuItem)
        {
          JBasicCheckBoxMenuItem menuItem = (JBasicCheckBoxMenuItem)m_fileList.get(i);
          JInternalFrame userObject = (JInternalFrame)menuItem.getUserObject();
          if (userObject != null && userObject == internalFrame)
          {
            m_menuWindow.remove(menuItem);
            break;
          }
        }
      }
    }
  }

  class InternalFramesWatcher extends InternalFrameAdapter
  {
    @Override
    public void internalFrameActivated(InternalFrameEvent evt)
    {
        MgmtConsole console = MgmtConsole.getMgmtConsole();
        console.getDesktopPane().setActive(true,evt.getInternalFrame());
    }

    @Override
    public void internalFrameDeactivated(InternalFrameEvent evt)
    {
        MgmtConsole console = MgmtConsole.getMgmtConsole();
        console.getDesktopPane().setActive(false,evt.getInternalFrame());
    }

    @Override
    public void internalFrameClosed(InternalFrameEvent e)
    {
        JInternalFrame iframe = e.getInternalFrame();
        internalFrames.remove(iframe);
        removeMenuItemFor(iframe);
        //Fix for bug Sonic00010013
        JMADesktopPane.this.remove(iframe);
    }
  }

    class TileHorizontalAction extends BasicGuiAction
    {
        public TileHorizontalAction()
        {
            super("JMADesktopPane.tile.horizontal");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            tileAll(false);
        }
    }

    class TileVerticalAction extends BasicGuiAction
    {
        public TileVerticalAction()
        {
            super("JMADesktopPane.tile.vertical");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            tileAll(true);
        }
    }

    class CascadeAction extends BasicGuiAction
    {
        public CascadeAction()
        {
            super("JMADesktopPane.cascade");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            cascadeAll();
        }
    }

    //-------------------------------------------------------------------------

    class OpenInternalFrameAction extends BasicGuiAction
    {
        OpenInternalFrameAction(JInternalFrame internalFrame)
        {
            super("JMADesktopPane.open");

            putNameValue(internalFrame);

            this.internalFrame = internalFrame;
        }
        
        private void putNameValue(JInternalFrame internalFrame) {
            putValue(Action.NAME,internalFrame.getTitle());
        }

        @Override
        public void actionPerformed(ActionEvent event)
        {
            try
            {
                internalFrame.setIcon(false);
                internalFrame.show();
                internalFrame.setSelected(true);
            }
            catch (PropertyVetoException e)
            {
                // cannot uniconify
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
            }
        }

        public JInternalFrame getInternalFrame()
        {
            return internalFrame;
        }

        private JInternalFrame internalFrame;
    }

    //-------------------------------------------------------------------------

    class WindowFileListAction extends BasicGuiAction
    {
        public WindowFileListAction()
        {
            super("JMADesktopPane.WindowFileList");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            WindowFileListDialog dialog = new WindowFileListDialog(MgmtConsole.getMgmtConsole(),"WindowFileListDialog");
            dialog.setVisible(true);

            if (dialog.getCloseCommand() == JMADialog.CLOSE_OK)
            {
                JList list = dialog.getWindowFileList();
                if (list != null && list.getSelectedValue() != null)
                {
                    ((JBasicCheckBoxMenuItem)list.getSelectedValue()).doClick();
                }
            }
        }
    }

    //-------------------------------------------------------------------------

    public class WindowFileListDialog extends JMADialog implements ListSelectionListener
    {
        protected JList m_list;

        public WindowFileListDialog(JMAFrame parent,String name)
        {
            super(parent,name);
        }

        public WindowFileListDialog(JMADialog parent, String name)
        {
            super(parent,name);
        }

        @Override
        protected void maInitialize()
        {
            getContentPane().add(createListPanel(),BorderLayout.CENTER);

            // Overridden behavior to prevent the button panel from being added to the dialog
            // until the other components are added, so that focus traversal works correcty.
            super.getContentPane().add(createButtonPanel(getButtonActions()),BorderLayout.SOUTH);
        }

        protected JPanel createListPanel()
        {
            JPartitionPanel outerPanel = new JPartitionPanel(true,"p,r","");

            outerPanel.add(new JLabel("Select a window to activate:"));

            m_list = new JList(m_fileList);
            m_list.setBorder(BorderFactory.createLoweredBevelBorder());
            m_list.setCellRenderer(new FileNameRenderer());
            m_list.getSelectionModel().setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
            m_list.addListSelectionListener(this);
            m_list.addMouseListener(new MouseAdapter()
            {
                @Override
                public void mouseClicked(MouseEvent e)
                {
                    if (e.getClickCount() == 2)
                    {
                        Action action = getDefaultOKAction();
                        if (action != null)
                        {
                            JButton button = ((JButtonPanel)getButtonPanel()).
                                getButton((String)action.getValue(Action.ACTION_COMMAND_KEY));

                            if (button != null)
                            {
                                button.doClick();
                            }
                        }
                    }
                }
            });

            outerPanel.add(new ExtendedJScrollPane(m_list));

            getDefaultOKAction().setEnabled(false);

            return outerPanel;
        }

        /**
         * Overridden method to prevent the button panel from being added to the dialog
         * until the other components are added, so that focus traversal works correcty.
         *
         * @param action Array of actions for the buttons
         * @return The button panel
         */
        @Override
        protected JPanel createButtonPanel(Action[] action)
        {
            return (m_list != null) ? super.createButtonPanel(action) : null;
        }

        /**
         * Re-implemented method to process event caused when a configuration is selected.
         *
         * @param event Event caused when a configuration is selected
         */
        @Override
        public void valueChanged(ListSelectionEvent event)
        {
            if (event.getSource() == m_list)
            {
                enableButtons();
            }
        }

        /**
         * Method to enable the Ok Button based on the configuration selection.
         */
        protected void enableButtons()
        {
            getDefaultOKAction().setEnabled(!m_list.isSelectionEmpty());
        }

        @Override
        protected void maCleanup()
        {
        }

        public JList getWindowFileList() { return m_list; }
    }

    /**
     *
     * <p>Title: FileNameRenderer</p>
     * <p>Description: Custom renderer to display a frames title.</p>
     * <p>Copyright: Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.</p>
     * <p>Company: Sonic Software Corporation</p>
     * @author Charlie Cocchiaro
     * @version 1.0
     */
    class FileNameRenderer extends DefaultListCellRenderer
    {
        /**
         * Overridden method to display an icon and text string for a plugin.
         *
         * @param list The JComboBox component
         * @param value The plugin factory
         * @param index Index of the item in the list
         * @param isSelected Selection indicator
         * @param cellHasFocus Focus indicator
         * @return The JLabel component being drawn
         */
        @Override
        public Component getListCellRendererComponent(JList list,Object value,int index,
            boolean isSelected,boolean cellHasFocus)
        {
            if (isSelected)
            {
                setBackground(list.getSelectionBackground());
                setForeground(list.getSelectionForeground());
            }
            else
            {
                setBackground(list.getBackground());
                setForeground(list.getForeground());
            }

            Icon icon = ResourceManager.getIcon(getClass(),"CheckMark");
            JBasicCheckBoxMenuItem menuItem = (JBasicCheckBoxMenuItem)value;

            if ( getSelectedFrame() == null)
            {
                setIcon(null);
            }
            else
            {
                // If the menu item corresponds to the active window, set a check icon.
                if (menuItem.isSelected())
                {
                    setIcon(icon);
                // Otherwise, create an icon with the background color so all the
                // list entries line up left justified.
                }
                else
                {
                    BufferedImage bufferedImage = new BufferedImage(icon.getIconWidth(),
                        icon.getIconHeight(),BufferedImage.TYPE_INT_ARGB);

                    Graphics graphics = bufferedImage.createGraphics();
                    graphics.setColor(getBackground());
                    graphics.fillRect(0,0,bufferedImage.getWidth(),bufferedImage.getHeight()-1);
                    setIcon(new ImageIcon(bufferedImage));
                }
            }

            setText(menuItem.getText());

            return this;
        }
    }
}
