/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpocration. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui.util;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.event.ActionEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.util.ArrayList;
import java.util.Iterator;

import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.text.JTextComponent;

import com.sonicsw.ma.gui.MgmtConsole;

public abstract class JMADialog extends JDialog implements WindowListener
{
    public static final int CLOSE_OK = 0;
    public static final int CLOSE_CANCEL = -1;

    private final BasicAction defaultOKAction = new OkAction("dialog.ok");
    private final BasicAction defaultCancelAction = new CancelAction("dialog.cancel");
    private final BasicAction defaultHelpAction = new HelpAction("dialog.help");

    protected JButtonPanel m_buttonPanel;

    private boolean m_initialized = false;
    protected String m_name;
    protected int m_closeCommand = CLOSE_CANCEL;
    private String m_helpId;
    private java.util.List keyEventProviders = new ArrayList();
    private DefaultsKeyAdapter keyAdaptor = new DefaultsKeyAdapter(this);

    public JMADialog(JMAFrame parent, String name)
    {
        super(parent);

        initializeDialog(name);
    }

    public JMADialog(JMADialog parent, String name)
    {
        super(parent);

        initializeDialog(name);
    }
    
    private void initializeDialog(String name) {
        initDialog(name);
    }

    protected void initDialog(String name)
    {
        m_name = name;

        if ((m_name == null) || (m_name.trim().length() == 0))
        {
            setTitle(ResourceManager.getString(getClass(), "JMADialog.unknownTitle"));
        }
        else
        {
            setTitle(ResourceManager.getString(getClass(), "dialog." + m_name + ".title"));
        }

        internalInitialize();
        setModal(true); // Default modality.
    }

    public String getTitleKey()
    {
        return m_name;
    }

    public void setTitleKey(String key)
    {
        m_name = key;
    }

    protected abstract void maInitialize();

    protected abstract void maCleanup();

    @Override
    public void setVisible(boolean newState)
    {
        if (!m_initialized)
        {
            // Add the button panel to the dialog if not null.  It can be null
            // when a subclass overrides the createButtonPanel method to call it
            // after it adds it's other components to the dialog, so that focus
            // traversal will work correctly.
            Action[] actions = getButtonActions();
            JPanel panel = null;

            if ((actions != null) && (actions.length > 0))
            {
                panel = createButtonPanel(actions);
            }

            if (panel != null)
            {
                super.getContentPane().add(panel, BorderLayout.SOUTH);

                if (panel instanceof JButtonPanel)
                {
                    JButton defBtn = ((JButtonPanel)panel).getButton("dialog.ok");

                    if (defBtn == null)
                    {
                        defBtn = ((JButtonPanel)panel).getButton("dialog.cancel");
                    }

                        // If there was no ok or cancel button then we go for the
                        // first button (action) in the action list.
                        //
                    if ((defBtn == null) && (actions != null) && (actions.length > 0))
                    {
                        String defCmd = (String)actions[0].getValue(Action.ACTION_COMMAND_KEY);

                        defBtn = ((JButtonPanel)panel).getButton(defCmd);
                    }

                    if (defBtn != null)
                    {
                        getRootPane().setDefaultButton(defBtn);
                    }
                }
            }

            m_initialized = true;

            // Initialize the custom components in the dialog.
            maInitialize();

            // Initially, we pack the dialog to its preferred size
            pack();

            // if there is a size stored in the preferences then use it
            if (isResizable())
            {
                SizeHelper.restoreWindowSize(this);
            }

            if (!SizeHelper.restoreWindowPosition(this))
            {
                setLocationRelativeTo(getParent());
            }
        }

        setKeyListener(this);

        // If the dialog is being made visible, attempt to set the focus on the
        // first editable/focusable component in the dialog.
        if (newState)
        {
            Helper.invoke(new Runnable()
            {
                @Override
                public void run()
                {
                    Component component = findFocusableComponent(getContentPane().getComponents());

                    if (component != null)
                    {
                        component.requestFocus();
                    }
                }
            });
        }

        super.setVisible(newState);
    }

    /**
     * Find the first editable/focusable component in the dialog and set the
     * input focus on it.
     *
     * @param components The containers' components
     * @return The component to set the focus on
     */
    protected Component findFocusableComponent(Component[] components)
    {
        Component component = null;

        if (components != null)
        {
            for (int i = 0; i < components.length; i++)
            {
                if (components[i].isLightweight())
                {
                    if (components[i] instanceof JComponent)
                    {
                        JComponent jcomp = (JComponent)components[i];

                        // Check for these type of components first before checking
                        // whether they have sub-components (because they all do) so
                        // that the focus can be set on them instead of their sub-components.
                        if (jcomp instanceof JComboBox || jcomp instanceof JList ||
                            jcomp instanceof JSlider)
                        {
                            if (jcomp.isFocusable() && jcomp.isEnabled())
                            {
                                component = jcomp;
                                break;
                            }
                        }
                        // If the component has sub-components (i.e. JPanel),
                        // drill down until we find the first sub-component.
                        else if (jcomp.getComponentCount() > 0)
                        {
                            // If the component is a JTabbedPane, only iterate
                            // through the selected tab's sub-components as we
                            // don't want to set the focus on a sub-component in
                            // in one of the unselected tabs if nothing is found
                            // in the selected one.
                            if (jcomp instanceof JTabbedPane)
                            {
                                Component selTab = ((JTabbedPane)jcomp).getSelectedComponent();
                                if (selTab == null)
                                {
                                    selTab = ((JTabbedPane)jcomp).getComponentAt(0);
                                }

                                if (selTab != null)
                                {
                                    component = findFocusableComponent(((JComponent)selTab).
                                        getComponents());
                                }

                                if (component == null)
                                 {
                                    component = jcomp; // Select the tab as a default.
                                }

                                break;
                            }
                            else
                            {
                                component = findFocusableComponent(jcomp.getComponents());
                                if (component != null)
                                {
                                    break;
                                }
                            }
                        }
                        else if (jcomp.isFocusable() && jcomp.isEnabled())
                        {
                            // If the component is a JTextComponent and it's editable,
                            // set focus on it.  This test is needed because even when
                            // the text field adapter sets the state to not editable,
                            // the focusable state and enabled state is still true and
                            // that causes the focus to be set on it, which is not what
                            // we want.
                            if (jcomp instanceof JTextComponent)
                            {
                                if (((JTextComponent)jcomp).isEditable())
                                {
                                    component = jcomp;
                                    break;
                                }
                            }
                            // Check for JButton, JRadioButton, JCheckBox, etc...
                            else if (jcomp instanceof AbstractButton || jcomp instanceof JTree ||
                                jcomp instanceof JTable)
                            {
                                component = jcomp;
                                break;
                            }
                        }
                    }
                }
            }
        }

        return (component);
    }

    /*
     * Method to assign a key listener to all the components contained within
     * this dialog so that the Escape key is be detected properly.
     * Note: It will not work if we only assign a key listener to the dialog, as
     * the keyboard events are not passed to the dialog from the components.
     * The Enter (Default action) is NOT done here. It is handled by calling
     * setDefaultButton() on the JRootPane.
     * @param container The container of components to add a key listener to
     */
    public void setKeyListener(Container container)
    {
        for (int i = 0; i < container.getComponentCount(); i++)
        {
            Component childComp = container.getComponent(i);

            if (childComp instanceof JComponent){
                childComp.addKeyListener(keyAdaptor);
                keyEventProviders.add(childComp);
            }

            if (childComp instanceof Container)
            {
                setKeyListener((Container)childComp);
            }
        }
    }

    /**
     * Accessor to return the button panel containing the default buttons
     * in the dialog, i.e. OK and Cancel.
     *
     * @return The panel containing the buttons
     */
    public JPanel getButtonPanel()
    {
        return (m_buttonPanel);
    }

    protected void internalInitialize()
    {
        setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);

        addWindowListener(this);
    }

    protected boolean isHelpEnabled()
    {
        if (m_helpId == null)
        {
            if ((m_name != null) && (m_name.trim().length() > 0))
            {
                m_helpId = ResourceManager.getString(getClass(), "dialog." + m_name + ".help");
            }
        }

        return (m_helpId != null);
    }

    protected String getHelpId()
    {
        return (isHelpEnabled() ? m_helpId : null);
    }

    public Action[] getButtonActions()
    {
        boolean hasHelp = isHelpEnabled();
        Action[] res = new Action[hasHelp ? 3 : 2];
        res[0] = defaultOKAction;
        res[1] = defaultCancelAction;

        if (hasHelp)
        {
            res[2] = defaultHelpAction;
        }

        return res;
    }

    protected boolean canClose()
    {
        return true;
    }

    protected JPanel createButtonPanel(Action[] action)
    {
        m_buttonPanel = new JButtonPanel(false);

        m_buttonPanel.addRemainder();

        for (int i = 0; i < action.length; i++)
        {
            if (action[i] != null)
            {
                m_buttonPanel.addButton(action[i]);
            }
        }

        Helper.equalize(m_buttonPanel);
        return m_buttonPanel;
    }

    public BasicAction getDefaultOKAction()
    {
        return defaultOKAction;
    }

    public BasicAction getDefaultCancelAction()
    {
        return defaultCancelAction;
    }

    public BasicAction getDefaultHelpAction()
    {
        return defaultHelpAction;
    }

    public int getCloseCommand()
    {
        return m_closeCommand;
    }

    //-------------------------------------------------------------------------
    //
    // WindowListener implementation
    //
    //-------------------------------------------------------------------------

    @Override
    public void windowActivated(WindowEvent evt)
    {
    }

    @Override
    public void windowClosed(WindowEvent evt)
    {
        if (m_initialized)
        {
            // we have to call them both since only saveWindowPosition will
            // flush the preference to backing store
            if (isResizable())
            {
                SizeHelper.saveWindowSize(this);
            }

            SizeHelper.saveWindowPosition(this);
        }

        try
        {
            maCleanup();
        }
        catch (Exception e)
        {
            MgmtConsole.displayMessage(MgmtConsole.MESSAGE_ERROR, "Error closing dialog.", e, true);
        }

        for (Iterator iterator = keyEventProviders.iterator(); iterator.hasNext();) {
            JComponent component = (JComponent) iterator.next();
            try {
                component.removeKeyListener(keyAdaptor);
            } catch (Exception e) {
                MgmtConsole.displayMessage(MgmtConsole.MESSAGE_ERROR, "Error closing dialog.", e, true);
            }
        }
        keyEventProviders.clear();
    }

    @Override
    public void windowClosing(WindowEvent evt)
    {
        if (canClose())
        {
            dispose();
        }
    }

    @Override
    public void windowDeactivated(WindowEvent evt)
    {
    }

    @Override
    public void windowDeiconified(WindowEvent evt)
    {
    }

    @Override
    public void windowIconified(WindowEvent evt)
    {
    }

    @Override
    public void windowOpened(WindowEvent evt)
    {
    }

    //-------------------------------------------------------------------------

    class OkAction
        extends BasicGuiAction
    {
        public OkAction(String resourceId)
        {
            super(resourceId);
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            super.actionPerformed(evt);

            Component source = (Component)evt.getSource();

            JMADialog dialog = (JMADialog)SwingUtilities.getAncestorOfClass(JDialog.class, source);

            //if (dialog != null)
            {
                dialog.m_closeCommand = CLOSE_OK;
                dialog.dispose();
            }
        }
    }

    class CancelAction
        extends BasicGuiAction
    {
        public CancelAction(String resourceId)
        {
            super(resourceId);
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            super.actionPerformed(evt);

            Component source = (Component)evt.getSource();
            JMADialog dialog = (JMADialog)SwingUtilities.getAncestorOfClass(JDialog.class, source);

            if ((dialog != null) && dialog.canClose())
            {
                dialog.m_closeCommand = CLOSE_CANCEL;
                dialog.dispose();
            }
        }
    }

    class HelpAction
        extends BasicGuiAction
    {
        public HelpAction(String resourceId)
        {
            super(resourceId);
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            super.actionPerformed(evt);

            Helper.displayURL(getHelpId());
        }
    }

    static class DefaultsKeyAdapter
        extends KeyAdapter
    {
        JMADialog m_dialog;

        /**
         * Constructor to create the keyboard listener.
         *
         * @param dialog The dialog containing this listener
         */
        public DefaultsKeyAdapter(JMADialog dialog)
        {
            m_dialog = dialog;
        }

        /**
         * Overridden method to detect a key pressed event.
         * Note: KeyEvent.getKeyCode() always returns 0 in keyTyped() event
         * so detect the keyboard event using this method instead.
         *
         * @param e The keyboard event that caused the action
         */
        @Override
        public void keyPressed(KeyEvent evt)
        {
            Action action = null;

            if (evt.getKeyCode() == KeyEvent.VK_ESCAPE)
            {
                action = m_dialog.getDefaultCancelAction();
            }

            if (action != null)
            {
                JButton button = null;
                if(m_dialog.m_buttonPanel!=null)
                {
                    button = m_dialog.m_buttonPanel.getButton(getCommandKey(action));
                }
                if (button != null)
                {
                    button.doClick();
                }
            }
        }

        private String getCommandKey(Action action)
        {
            return (String)action.getValue(Action.ACTION_COMMAND_KEY);
        }

    }
}