package com.sonicsw.ma.gui.util;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JTextField;
import javax.swing.Timer;
import javax.swing.border.EmptyBorder;
import javax.swing.border.SoftBevelBorder;

public class JMAStatusBar
    extends JPanel
{
    private static final int TIMER_DELAY = 5000;
    private static final int PROGRESS_HEIGHT = 15;
    private String TEXT_READY = "Ready";

    private Timer m_timer;
    private JLabel m_statusArea;
    private JPanel panelProgress;
    private JProgressBar m_progressBar;
    private boolean m_bProgressLock = false;

    public JMAStatusBar(Action cancelAction)
    {
        super(new GridLayout(1, 2));

        init();
    }

    /**
     *  Return this StatusBar's JProgressBar
     */
    public JProgressBar getProgressBar()
    {
        JProgressBar barRes = null;

        if (m_bProgressLock == false && m_progressBar != null)
        {
            // lock it...
            m_bProgressLock = true;
            barRes = m_progressBar;
        }

        return barRes;
    }

    public void releaseProgressBar()
    {
        if (m_bProgressLock)
        {
            m_bProgressLock = false;
        }
    }

    /**
     *  Display the specified message in the main status-text field.
     */
    public void showStatusMessage(String message)
    {
        updateText(m_statusArea, (message != null) ? message : TEXT_READY);

        if (message != null)
        {
            m_timer.restart();
        }
    }

    public void setPersistentMessage(String message)
    {
        updateText(m_statusArea, (message != null) ? message : TEXT_READY);
    }

    /**
     *  Update the specified JTextField or JLabel to have the specified
     *  String value by using SwingUtilities.invokeLater, so that the
     *  modification will happen only in the Swing event thread.
     */
    protected void updateText(JComponent comp, String text)
    {
        Helper.invoke(new TextUpdater(comp, text));
    }

    /**
     *  Start or stop the progress meter, indicating work-in-progress or
     *  its completion.
     */
    public synchronized void monitorProgress(boolean start)
    {
        if (m_progressBar != null)
        {
            m_progressBar.setIndeterminate(start);
        }
    }

    /**
     *  Set up the layout and composition of this StatusBar.
     */
    private void init()
    {
        m_timer = new Timer(TIMER_DELAY,
            new ActionListener()
            {
                @Override
                public void actionPerformed(ActionEvent evt)
                {
                    showStatusMessage(TEXT_READY);
                }
            });

        setBorder(new EmptyBorder(2, 2, 0, 0));

        // add message label...
        add(m_statusArea = new JLabel(TEXT_READY));

        // create and add status panel...
        add(createStatusPanel());
    }

    private JPanel createStatusPanel()
    {
        // create status panel...
        final JPanel panel = new JPanel(new BorderLayout());

        // create combined status panel...
        panel.add(BorderLayout.CENTER, createCombinedStatusPanel());

        return panel;
    }

    private JPanel createCombinedStatusPanel()
    {
        // create combined status panel...
        final JPanel panel = new JPanel(new GridBagLayout());

        final JPanel panelProgress = createProgressPanel();
        panel.add(panelProgress);
        setAnchor(panelProgress, GridBagConstraints.WEST);
        fillWidth(panelProgress);

        final JPanel panelIndicators = createIndicatorPanel();
        panel.add(panelIndicators);
        setAnchor(panelIndicators, GridBagConstraints.EAST);
        fillWidth(panelIndicators);

        return panel;
    }

    private JPanel createProgressPanel()
    {
        // create progress panel...
        final JPanel panel = new JPanel(new BorderLayout());

        // create progress bar...
        m_progressBar = new JProgressBar(JProgressBar.HORIZONTAL);
        m_progressBar.setBorder(new SoftBevelBorder(SoftBevelBorder.LOWERED));
        m_progressBar.setVisible(false);
        m_progressBar.setPreferredSize(new Dimension(m_progressBar.getWidth(), PROGRESS_HEIGHT));

        // add progress bar...
        panel.add(m_progressBar);

        return panel;
    }

    private JPanel createIndicatorPanel()
    {
        // create indicator panel...
        return new JPanel(new FlowLayout(FlowLayout.RIGHT, 5, 1));
    }

    //-------------------------------------------------------------------------

    static class TextUpdater
        implements Runnable
    {
        private Component comp;
        private String text;

        public TextUpdater(JComponent comp, String text)
        {
            this.comp = comp;
            this.text = text;
        }

        @Override
        public void run()
        {
            if (comp instanceof JTextField)
            {
                ((JTextField)comp).setText(text);
            }
            else if (comp instanceof JLabel)
            {
                ((JLabel)comp).setText(text);
            }
        }
    }

    public GridBagConstraints setAnchor(Component c, int nAnchor)
    {
        Container parent = (Container)c.getParent();
        GridBagLayout gbl = (GridBagLayout)parent.getLayout();
        GridBagConstraints gbc = gbl.getConstraints(c);

        // If there are no constraints currently, create some.
        if (gbc == null)
        {
            gbc = new GridBagConstraints();
        }

        // Set the anchor.
        gbc.anchor = nAnchor;

        // Set it back in.
        gbl.setConstraints(c, gbc);

        return (gbc);
    }

    public GridBagConstraints fillWidth(Component c)
    {
        Container parent = (Container)c.getParent();
        GridBagLayout gbl = (GridBagLayout)parent.getLayout();
        GridBagConstraints gbc = gbl.getConstraints(c);

        // If there are no constraints currently, create some.
        if (gbc == null)
        {
            gbc = new GridBagConstraints();
        }

        // Set the fill.
        fillWidth(gbc);

        // Set the constraint back in.
        gbl.setConstraints(c, gbc);

        return (gbc);
    }

    public void fillWidth(GridBagConstraints gbc)
    {
        // Set the fill.
        switch (gbc.fill)
        {
            case GridBagConstraints.BOTH:
                break;
            case GridBagConstraints.VERTICAL:
                gbc.fill = GridBagConstraints.BOTH;
                break;
            case GridBagConstraints.HORIZONTAL:
                break;
            case GridBagConstraints.NONE:
                gbc.fill = GridBagConstraints.HORIZONTAL;
                break;
        }

        // If there is no weight, set one.
        if (gbc.weightx == 0.0)
        {
            gbc.weightx = 1.0;
        }
    }
}