/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corporation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui.util;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.event.ActionEvent;

import javax.swing.Action;
import javax.swing.JDialog;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.SwingUtilities;

import com.sonicsw.ma.gui.MgmtConsole;

/**
 * <p>Title: JPasswordDialog</p>
 * <p>Description: Generic Password/Confirm Password dialog class.  Returns
 * password in flat ascii text or hashed text formats.</p>
 * <p>Copyright: Copyright (c) 2002</p>
 * <p>Company: Sonic Software</p>
 * @author Charlie Cocchiaro
 * @version 1.0
 */
public class JPasswordDialog extends JMADialog
{
    protected JPasswordField passwordField;
    protected JPasswordField confirmField;
    private JPanel   m_contentPanel;
    
    /**
     * Constructor to create the dialog.
     *
     * @param parent The parent frame for the dialog.
     */
    public JPasswordDialog(JMAFrame parent)
    {
        super(parent,"JPasswordDialog");
        setResizableProperty();
    }

    public JPasswordDialog(JMADialog parent)
    {
        super(parent,"JPasswordDialog");
        setResizableProperty();
    }
    
    private void setResizableProperty() {
        setResizable(false);
    }

    @Override
    public Container getContentPane()
    {
        return m_contentPanel;
    }

    @Override
    protected void internalInitialize()
    {
        super.internalInitialize();

        // Setup the "content" pane...
        //
        m_contentPanel = new JPanel(new BorderLayout(2, 2));

        super.getContentPane().add(m_contentPanel);
    }
    
    /**
     * Overridden method to create the dialog pansles and components.
     */
    @Override
    protected void maInitialize()
    {
        JPartitionPanel topPanel = new JPartitionPanel(true,"p",null);
        getContentPane().add(topPanel, BorderLayout.CENTER);

        JPartitionPanel loginPanel = new JPartitionPanel(false, "p,r", "", 5, 0, 0);
        topPanel.add(loginPanel);

        passwordField = new JPasswordField(16);
        loginPanel.addRow(ResourceManager.getString(getClass(), "dialog.JPasswordDialog.Password"),passwordField);

        confirmField = new JPasswordField(16);
        loginPanel.addRow(ResourceManager.getString(getClass(), "dialog.JPasswordDialog.ConfirmPassword"),confirmField);
    }

    /**
     * Overridden method to set the actions for the standard OK and Cancel buttons
     * to this class' to provide custom behavior.
     *
     * @return Array of actions
     */
    @Override
    public Action[] getButtonActions()
    {
        Action[] OVERRIDDEN_BUTTON_ACTIONS =
        {
            new OkButtonAction("dialog.ok",this),
            new CancelButtonAction("dialog.cancel")
        };

        return (OVERRIDDEN_BUTTON_ACTIONS);
    }

    /**
     * Method to return the password as a hashed byte array.
     *
     * @return The hashed password
     */
    public byte[] getHashedPassword()
    {
        byte[] bytes = null;

        try
        {
            String password = getPassword();

            if ((password != null) && (password.length() > 0))
            {
                bytes = PasswordHash.doSHA(password);
            }
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR, "Error while getting password.", e, true);
        }

        return bytes;
    }

    /**
     * Method to return the Password as a String.
     *
     * @return The unhashed password
     */
    public String getPassword()
    {
        if (passwordField.getPassword() == null)
        {
            return null;
        }
        else
        {
            return new String(passwordField.getPassword());
        }
    }

    /**
     * Method to set the Password from a String.
     *
     * @param value String containing the password
     */
    public void setPassword(String value)
    {
        passwordField.setText(value);

        return;
    }

    /**
     * Method to return the Password as a char array.
     *
     * @return The unhashed password
     */
    public char[] getPasswordChars()
    {
        if (passwordField.getPassword() == null)
        {
            return (null);
        }
        else
        {
            return (passwordField.getPassword());
        }
    }

    /**
     * Method to set the Password from a char array.
     *
     * @param chars Array of chars containing the password
     */
    public void setPasswordChars(char[] chars)
    {
        if (chars == null)
        {
            passwordField.setText(null);
        }
        else
        {
            passwordField.setText(new String(chars));
        }

        return;
    }

    /**
     * Method to return the Password as a byte array.
     *
     * @return The unhashed password
     */
    public byte[] getPasswordBytes()
    {
        if (passwordField.getPassword() == null)
        {
            return (null);
        }
        else
        {
            String password = new String(passwordField.getPassword());
            return (password.getBytes());
        }
    }

    /**
     * Method to set the Password from a byte array.
     *
     * @param bytes Array of bytes containing the password
     */
    public void setPasswordBytes(byte[] bytes)
    {
        if (bytes == null)
        {
            passwordField.setText(null);
        }
        else
        {
            passwordField.setText(new String(bytes));
        }

        return;
    }

    /**
     * Method to return the Confirm Password as a String.
     *
     * @return The Confirm Password text
     */
    public String getConfirmPassword()
    {
        if (confirmField.getPassword() == null)
        {
            return (null);
        }
        else
        {
            return (new String(confirmField.getPassword()));
        }
    }

    /**
     * Method to validate the Password and Confirm Password fields.
     *
     * @param password The Password field value
     * @param confirmPassword The Confirm Password field value
     * @return Flag indicating whether the two fields were equal
     */
    public boolean validatePassword(String password,String confirmPassword)
    {
        boolean valid = false;

        if (password != null && confirmPassword != null)
        {
            if (password.equals(confirmPassword))
            {
                valid = true;
            }
        }

        return (valid);
    }

    /**
     * Overridden method to cleanup dialog.  Do not call dispose() in this class
     * as the calling class may want to extract the input values before destroying it.
     */
    @Override
    protected void maCleanup()
    {
        return;
    }

    /**
     *
     * <p>Title: OkButtonAction</p>
     * <p>Description: Action for the OK button.</p>
     * <p>Copyright: Copyright (c) 2002</p>
     * <p>Company: Sonic Software</p>
     * @author Charlie Cocchiaro
     * @version 1.0
     */
    protected class OkButtonAction extends BasicGuiAction
    {
        protected Component m_parent;

        /**
         * Constructor to create the action.
         *
         * @param resourceId The action resource ID for the button label
         * @param parent The parent frame for this action
         */
        public OkButtonAction(String resourceId,Component parent)
        {
            super(resourceId);

            m_parent = parent;
        }

        /**
         * Overridden method to handle the action event for the button.
         *
         * @param evt The event that caused the action
         */
        @Override
        public void actionPerformed(ActionEvent evt)
        {
            super.actionPerformed(evt);

            Component source = (Component)evt.getSource();
            JPasswordDialog dialog = (JPasswordDialog)SwingUtilities.getAncestorOfClass(
                JDialog.class,source);

            if (!dialog.validatePassword(dialog.getPassword(),dialog.getConfirmPassword()))
            {
                JOptionPane.showMessageDialog(m_parent,ResourceManager.getString(
                    getClass(),"dialog.JPasswordDialog.ValidateError"),ResourceManager.getString(
                    getClass(),"dialog.JPasswordDialog.title"),JOptionPane.ERROR_MESSAGE);
            }
            else
            {
                dialog.m_closeCommand = CLOSE_OK;
                dialog.setVisible(false);
            }
        }
    }

    /**
     *
     * <p>Title: CancelButtonAction</p>
     * <p>Description: Action for the Cancel button.</p>
     * <p>Copyright: Copyright (c) 2002</p>
     * <p>Company: Sonic Software</p>
     * @author Charlie Cocchiaro
     * @version 1.0
     */
    protected class CancelButtonAction extends BasicGuiAction
    {
        /**
         * Constructor to create the action.
         *
         * @param resourceId The action resource ID for the button label
         */
        public CancelButtonAction(String resourceId)
        {
            super(resourceId);
        }

        /**
         * Overridden method to handle the action event for the button.
         *
         * @param evt The event that caused the action
         */
        @Override
        public void actionPerformed(ActionEvent evt)
        {
            super.actionPerformed(evt);

            Component source = (Component)evt.getSource();
            JPasswordDialog dialog = (JPasswordDialog)SwingUtilities.getAncestorOfClass(
                JDialog.class,source);

            dialog.m_closeCommand = CLOSE_CANCEL;
            dialog.setVisible(false);
        }
    }
}