package com.sonicsw.ma.gui.util;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.HeadlessException;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.Action;
import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import modelobjects.layout.PartitionLayout;

public class JPathChooser
    extends JComponent
    implements ListSelectionListener,
    KeyListener,
    ItemListener
{
    // Dialog Return Values
    //
    public static final int CANCEL_OPTION = 1;
    public static final int APPROVE_OPTION = 0;
    public static final int ERROR_OPTION = -1;

    private static final int MODE_NOP = 0;
    private static final int MODE_EDIT = 1;
    private static final int MODE_NEW = 2;

    private static int PREF_WIDTH = 500;
    private static int PREF_HEIGHT = 326;
    private static Dimension PREF_SIZE = new Dimension(PREF_WIDTH, PREF_HEIGHT);

    private static int MIN_WIDTH = 500;
    private static int MIN_HEIGHT = 326;
    private static Dimension MIN_SIZE = new Dimension(MIN_WIDTH, MIN_HEIGHT);

    DefaultListModel pathModel;
    JDialog dialog;
    JList pathList;
    JLabel lblField;
    JComponent field;
    JButton btnOk;
    JButton btnCancel;
    JButton btnNew;
    JButton btnDelete;
    JButton btnApply;
    int returnValue;
    int mode;
    boolean fieldChanged;
    boolean formChanged;
    boolean selChanged;
    String[] oldPath;
    String[] choices;
    boolean pathEditable;
    JButton btnMoveUp;
    JButton btnMoveDown;
    Action applyAction;
    boolean m_readOnly;

    public JPathChooser()
    {
        super();

        dialog = null;
        mode = MODE_NOP;
        fieldChanged = false;
        formChanged = false;
        selChanged = false;
        pathEditable = true;
        choices = new String[0];
        pathModel = new DefaultListModel();
        m_readOnly = false;
        
        prepareJPathChooser();
    }
    private void prepareJPathChooser() {
        
        setLayout(new BorderLayout());

        add(createMainPanel(), BorderLayout.CENTER);
        add(createButtonPanel(), BorderLayout.SOUTH);
        add(createTableButtonPanel(), BorderLayout.EAST);

        applyAction = new ApplyAction();

        btnApply.setAction(applyAction);
        setPathChoices(null);

        formUpdate();
    }

    private JPanel createMainPanel()
    {
        JPanel panel = new JPanel();
        panel.setLayout(new PartitionLayout(true, "r,p", 5, 2, 2, "JPathChooserMainPanel"));

        pathList = new JList(pathModel);

        pathList.addListSelectionListener(this);

        panel.add(new ExtendedJScrollPane(pathList));
        panel.add(createEditPanel());

        return panel;
    }

    private JPanel createEditPanel()
    {
        JPanel panel = new JPanel();
        panel.setLayout(new PartitionLayout(false, "p,r", 5, 2, 2, "JPathChooserEditPanel"));

        lblField = new JLabel("Path: ");

        field = new JTextField();

        panel.add(lblField);
        panel.add(field);

        return panel;
    }

    public int showDialog(Component parent)
        throws HeadlessException
    {
        dialog = createDialog(parent);

        dialog.addWindowListener(new WindowAdapter()
        {
            @Override
            public void windowClosing(WindowEvent e)
            {
                returnValue = CANCEL_OPTION;
            }
        });

        returnValue = ERROR_OPTION;

        dialog.show();
        dialog.dispose();
        dialog = null;

        return returnValue;
    }

    protected JDialog createDialog(Component parent)
        throws HeadlessException
    {
        Frame frame = parent instanceof Frame ? (Frame)parent
            : (Frame)SwingUtilities.getAncestorOfClass(Frame.class, parent);

        JDialog dialog = new JDialog(frame, "Path Chooser", true);

        Container contentPane = dialog.getContentPane();
        contentPane.setLayout(new BorderLayout());
        contentPane.add(this, BorderLayout.CENTER);

        this.setPreferredSize(PREF_SIZE);
        this.setMinimumSize(MIN_SIZE);

        dialog.pack();
        dialog.setLocationRelativeTo(parent);

        return dialog;
    }

    public void setPathChoices(String[] path)
    {
        choices = (path != null) ? path : new String[0];

        JComponent theField = (JComponent)Helper.getDescendentClass(JTextField.class, this);
        Container parent = (Container)theField.getParent();

        if ((path == null) || (path.length == 0))
        {
            if (field instanceof JComboBox)
            {
                parent.remove(field);
                ((JComboBox)field).removeItemListener(this);
                ((JComboBox)field).getEditor().getEditorComponent().removeKeyListener(this);

                field = new JTextField();
                ((JTextField)field).addKeyListener(this);
                ((JTextField)field).setAction(applyAction);

                parent.add(field);

                formUpdate();
            }
        }
        else
        {
            if (field instanceof JTextField)
            {
                parent.remove(field);
                field.removeKeyListener(this);

                field = new JNoActionComboBox(choices);
                ((JComboBox)field).setSelectedItem(null);
                ((JComboBox)field).addItemListener(this);
                ((JComboBox)field).getEditor().getEditorComponent().addKeyListener(this);
                ((JComboBox)field).setAction(applyAction);

                parent.add(field);

                formUpdate();
            }
        }
    }

    public String[] getPathChoices()
    {
        return choices;
    }

    public void setSelectedPaths(String[] path)
    {
        oldPath = path;

        for (int i = 0; i < path.length; i++)
        {
            String aPath = path[i].trim();

            if (aPath.length() > 0)
            {
                pathModel.addElement(aPath);
            }
        }
    }

    public String[] getSelectedPaths()
    {
        String[] res = new String[pathModel.getSize()];

        for (int i = 0; i < res.length; i++)
        {
            res[i] = (String)pathModel.getElementAt(i);
        }

        return res;
    }

    public void setPathEditable(boolean editable)
    {
        pathEditable = editable;

        if (field instanceof JComboBox)
        {
            ((JComboBox)field).setEditable(pathEditable);
        }
    }

    public boolean isPathEditable()
    {
        return pathEditable;
    }

    public void setReadOnly(boolean readOnly)
    {
        m_readOnly = readOnly;

        formUpdate();
    }

    public boolean isReadOnly()
    {
        return m_readOnly;
    }

    private JPanel createButtonPanel()
    {
        JPanel panel = new JPanel();
        panel.setLayout(new PartitionLayout(false, "r,p,p,r", 5, 2, 2, "JPathChooserButonPanel"));

        btnOk = new JButton("OK");
        btnOk.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent evt)
            {
                dialog.setVisible(false);
                returnValue = APPROVE_OPTION;
            }
        });

        btnCancel = new JButton("Cancel");
        btnCancel.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent evt)
            {
                dialog.setVisible(false);
                returnValue = CANCEL_OPTION;
            }
        });

        panel.add(new JPanel());
        panel.add(btnOk);
        panel.add(btnCancel);
        panel.add(new JPanel());

        return panel;
    }

    private JPanel createTableButtonPanel()
    {
        JPanel panel = new JPanel();
        panel.setLayout(new PartitionLayout(true, "p,p,p,r", 5, 2, 2, "JPathChooserTableButonPanel"));

        btnNew = new JButton(new NewAction());
        btnDelete = new JButton(new DeleteAction());
        btnApply = new JButton();

        panel.add(btnNew);
        panel.add(btnDelete);
        panel.add(btnApply);

        return panel;
    }

    protected void formUpdate()
    {
        DefaultListModel model = (DefaultListModel)pathList.getModel();
        int selIndex = pathList.getSelectedIndex();
        boolean oneSel = (pathList.getSelectedIndices().length == 1);
        boolean sel = oneSel && ((selIndex >= 0) && (selIndex < model.size()));

        btnNew.setEnabled(!isReadOnly() && !fieldChanged && (mode != MODE_NEW));
        btnDelete.setEnabled(!isReadOnly() && (mode != MODE_NEW) &&
            (pathList.getSelectedIndices().length > 0));
        btnApply.setEnabled(!isReadOnly() && fieldChanged && ((mode == MODE_NEW) ||
            (mode == MODE_EDIT)));
        field.setEnabled(!isReadOnly() && (mode != MODE_NOP));
        btnOk.setEnabled(formChanged);
    }

    //--------------------------------------------------------------------------
    //
    // ListSelectionListener implementation
    //
    //--------------------------------------------------------------------------

    @Override
    public void valueChanged(ListSelectionEvent evt)
    {
        if (evt.getValueIsAdjusting())
        {
            return;
        }

        fieldChanged = false;
        mode = (pathList.getSelectedIndices().length > 0) ? MODE_EDIT : MODE_NOP;

        Helper.invoke(new Runnable()
        {
            @Override
            public void run()
            {
                DefaultListModel model = (DefaultListModel)pathList.getModel();
                int selIndex = pathList.getSelectedIndex();
                boolean oneSel = (pathList.getSelectedIndices().length == 1);
                boolean sel = oneSel && ((selIndex >= 0) && (selIndex < model.size()));
                String value = sel ? (String)model.getElementAt(selIndex) : null;

                if (field instanceof JTextField)
                {
                    ((JTextField)field).setText(value);
                }
                else
                {
                    ((JComboBox)field).setSelectedItem(value);
                }

                formUpdate();
            }
        });
    }

    //--------------------------------------------------------------------------
    //
    // ItemListener implementation
    //
    //--------------------------------------------------------------------------

    @Override
    public void itemStateChanged(ItemEvent evt)
    {
        if (evt.getStateChange() == ItemEvent.SELECTED)
        {
            fieldChanged = true;
        }

        Helper.invoke(new Runnable()
        {
            @Override
            public void run()
            {
                formUpdate();
            }
        });
    }

    //--------------------------------------------------------------------------
    //
    // KeyListener
    //
    //--------------------------------------------------------------------------

    @Override
    public void keyTyped(KeyEvent evt)
    {
        // I think this is a bit dodgy since the VK codes should ONLY be compared
        // to the events KeyCode value....and even that can't really be done in
        // the keyTyped method...but VK_ENTER has the same numeric value as
        // the new line character so...
        //
        if (evt.getKeyChar() == KeyEvent.VK_ENTER)
        {
            return;
        }

        fieldChanged = true;

        Helper.invoke(new Runnable()
        {
            @Override
            public void run()
            {
                formUpdate();
            }
        });
    }

    @Override
    public void keyPressed(KeyEvent evt)
    {
    }

    @Override
    public void keyReleased(KeyEvent evt)
    {
    }

    //--------------------------------------------------------------------------
    //
    // Actions
    //
    //--------------------------------------------------------------------------

    class NewAction
        extends BasicGuiAction
    {
        public NewAction()
        {
            super("JPathChooser.new");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            mode = MODE_NEW;

            Helper.invoke(new Runnable()
            {
                @Override
                public void run()
                {
                    formUpdate();

                    field.requestFocus();
                }
            });
        }
    }

    class DeleteAction
        extends BasicGuiAction
    {
        public DeleteAction()
        {
            super("JPathChooser.delete");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            DefaultListModel model = (DefaultListModel)pathList.getModel();
            int[] index = pathList.getSelectedIndices();

            for (int i = index.length - 1; i >= 0; i--)
            {
                model.remove(index[i]);
            }

            int newIndex = Math.max(index[index.length - 1] - 1, model.getSize() - 1);

            pathList.setSelectedIndex(newIndex);

            mode = (newIndex >= 0) ? MODE_EDIT : MODE_NOP;
            formChanged = true;
            fieldChanged = false;

            Helper.invoke(new Runnable()
            {
                @Override
                public void run()
                {
                    formUpdate();
                }
            });
        }
    }

    class ApplyAction
        extends BasicGuiAction
    {
        public ApplyAction()
        {
            super("JPathChooser.apply");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            if (evt.getSource().getClass().equals(JNoActionComboBox.class) && selChanged)
            {
                // Nasty little flag here to hack a problem: when the user selects an item
                // from the dropdown list in the combobox, it automatically fires the action
                // event so we keep this flag (which is set in the combobox list selection
                // listener) to distinguish between list selection and the internal editor
                // action.
                //
                selChanged = false;
                return;
            }

            String path = (field instanceof JTextField) ? ((JTextField)field).getText()
                : ((String)((JComboBox)field).getSelectedItem());

            if ((path == null) || (path.length() == 0))
            {
                return;
            }

            path = path.trim();

            DefaultListModel model = (DefaultListModel)pathList.getModel();

            if (mode == MODE_NEW)
            {
                if (model.indexOf(path) == -1)
                {
                    model.addElement(path);
                }

                pathList.setSelectedIndex(model.indexOf(path));

                mode = MODE_EDIT;
            }
            else
            if (mode == MODE_EDIT)
            {
                model.setElementAt(path, pathList.getSelectedIndex());
            }

            fieldChanged = false;
            formChanged = true;

            Helper.invoke(new Runnable()
            {
                @Override
                public void run()
                {
                    formUpdate();
                }
            });
        }
    }

    class JNoActionComboBox
        extends JComboBox
    {
        public JNoActionComboBox(Object[] values)
        {
            super(values);
        }

        @Override
        public void setSelectedItem(Object value)
        {
            selChanged = true;

            super.setSelectedItem(value);
        }
    }
}