package com.sonicsw.ma.gui.util;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Vector;

import javax.swing.AbstractAction;
import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.UIManager;
import javax.swing.event.AncestorEvent;
import javax.swing.event.AncestorListener;
import javax.swing.event.PopupMenuEvent;
import javax.swing.event.PopupMenuListener;
import javax.swing.event.TableColumnModelListener;
import javax.swing.event.TableModelEvent;
import javax.swing.table.DefaultTableColumnModel;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

import com.sonicsw.ma.gui.PreferenceManager;
import com.sonicsw.ma.gui.table.IModelTableModel;
import com.sonicsw.ma.gui.table.ModelListTableModel;
import com.sonicsw.ma.gui.table.PluginNameTableCellRenderer;
import com.sonicsw.ma.gui.table.RowTableColumn;
import com.sonicsw.ma.plugin.AbstractGUIPlugin;

public class JRowTable
    extends ExtendedJTable
{
    private boolean m_focusLossTemporary = false;
    private FocusListener m_focusListener = null;
    private String m_strPreferenceKey = null;
    private boolean m_bRealized = false;

    public JRowTable(TableColumn[] tableColumn, String strPreferenceKey)
    {
        this(new ModelListTableModel(tableColumn, new ArrayList()), strPreferenceKey);
    }

    public JRowTable(IModelTableModel tableModel, String strPreferenceKey)
    {
        this(tableModel, null, null, strPreferenceKey);
    }

    public JRowTable(IModelTableModel dm,
                     TableColumnModel cm,
                     ListSelectionModel sm,
                     String strPreferenceKey)
    {
        super(dm, cm, sm);
        prepareJRowTable(strPreferenceKey);
    }
    
    private void prepareJRowTable(String strPreferenceKey) {
        
        m_strPreferenceKey = strPreferenceKey;

        setAutoCreateColumnsFromModel(false);
        getTableHeader().addMouseListener(new TableSorter(this));
        getTableHeader().addMouseListener(new PopupMenuShower(this, createHeaderPopupMenu()));
        getColumnModel().addColumnModelListener((TableColumnModelListener) getModel());

        setShowHorizontalLines(false);
        setShowVerticalLines(false);

        TableCellRenderer leftR = new ExtendedJTable.ExtendedTableCellRenderer(JLabel.LEFT);
        TableCellRenderer rightR = new ExtendedJTable.ExtendedTableCellRenderer(JLabel.RIGHT);
        setDefaultRenderer(Object.class, leftR);
        setDefaultRenderer(String.class, leftR);
        setDefaultRenderer(Boolean.class, leftR);
        setDefaultRenderer(Number.class, rightR);
        setDefaultRenderer(Double.class, rightR);
        setDefaultRenderer(Long.class, rightR);
        setDefaultRenderer(Integer.class, rightR);

        setAutoResizeMode(JTable.AUTO_RESIZE_OFF);

        addAncestorListener(new AncestorListener()
        {
            @Override
            public void ancestorRemoved(AncestorEvent event){}
            @Override
            public void ancestorAdded(AncestorEvent event)
            {
                if (m_bRealized == false)
                {
                    Helper.invoke(new Runnable()
                    {
                        @Override
                        public void run()
                        {
                            loadColumnPrefs();
                        }
                    });

                    if (m_focusListener == null)
                    {
                        m_focusListener = new FocusHandler();

                    }
                    addFocusListener(m_focusListener);

                    m_bRealized = true;
                }
            };
            @Override
            public void ancestorMoved(AncestorEvent event){};
        });
    }

    public boolean getSortable() { return ((IModelTableModel) getModel()).isSortActivated(); }
    public void setSortable(boolean bSortable) { ((IModelTableModel) getModel()).activateSort(bSortable); }

    @Override
    public String getToolTipText(MouseEvent event)
    {
        java.awt.Point point = event.getPoint();
        int rowIndex = rowAtPoint(point);
        int colIndex = columnAtPoint(point);

        int realColumnIndex = convertColumnIndexToModel(colIndex);

        TableCellRenderer renderer = getColumnModel().getColumn(realColumnIndex).getCellRenderer();

        if (renderer instanceof PluginNameTableCellRenderer)
        {
            Object rowModel = ((ModelListTableModel)getModel()).getRowModel(rowIndex);
            if (rowModel instanceof AbstractGUIPlugin)
            {
                return ((AbstractGUIPlugin)rowModel).getToolTipText();
            }
        }

        return super.getToolTipText(event);
    }

    public void cleanup()
    {
        saveColumnPrefs();

        // JSP: need to find an appropriate place to clean this up. not
        // as critical now that we are caching content panes.
        //if (m_focusListener != null)
        //{
        //    removeFocusListener(m_focusListener);
        //    m_focusListener = null;
        //}
    }

    @Override
    protected TableColumnModel createDefaultColumnModel()
    {
        return new ExtendedTableColumnModel();
    }

    @Override
    public void createDefaultColumnsFromModel()
    {
        IModelTableModel tableModel = (IModelTableModel) getModel();

        if (tableModel != null)
        {
            // Remove any current columns
            ExtendedTableColumnModel cm = (ExtendedTableColumnModel) getColumnModel();
            cm.dropAllColumns();

            // Create new columns from the data model info
            for (int i = 0; i < tableModel.getColumnCount(); i++)
            {
                TableColumn column = tableModel.getColumn(i);

                column.setHeaderRenderer(new HeaderCellRenderer(tableModel, column));

                addColumn(column);
            }
        }
    }

    @Override
    public void setModel(TableModel dm)
    {
        if ((dm != null) && !(dm instanceof IModelTableModel))
        {
            throw new IllegalArgumentException("TableModel must extend IModelTableModel");
        }

        super.setModel(dm);
    }

    public void loadColumnPrefs()
    {
        boolean bChanged = false;

        ExtendedTableColumnModel modelColumn = (ExtendedTableColumnModel) JRowTable.this.getColumnModel();
        IModelTableModel modelTable = (IModelTableModel) getModel();

        if (m_strPreferenceKey != null && modelTable != null && modelColumn != null)
        {
            // get sorting preferences...
            modelTable.setSortedByColumn(PreferenceManager.getInstance().getInt("table", m_strPreferenceKey + ".sortColumn", 0),
                PreferenceManager.getInstance().getBoolean("table", m_strPreferenceKey + ".sortAscending", modelTable.isSortedInAscendingOrder()));

            int nCols = modelColumn.getWrappedColumnCount();

            // for each column, load its preferences...
            for (int nCol = 0; nCol < nCols; nCol++)
            {
                // get column width...
                int nPreferredWidth = PreferenceManager.getInstance().getInt("table", m_strPreferenceKey + "." + modelColumn.getWrappedColumnName(nCol) + ".width", -1);
                if (nPreferredWidth != -1)
                {
                    modelTable.getColumn(nCol).setPreferredWidth(nPreferredWidth);
                    bChanged = true;
                }

                WrappedColumn wCol = modelColumn.getWrappedColumn(nCol);
                TableColumn col = wCol.getTableColumn();

                // Get initial column visibility...
                boolean defVisible = (col instanceof RowTableColumn) ? ((RowTableColumn)col).isDefaultVisible() : true;
                
                // get column visibility...
                boolean bVisible = PreferenceManager.getInstance().getBoolean("table",
                        m_strPreferenceKey + "." + modelColumn.getWrappedColumnName(nCol) + ".visible", defVisible);
                
                if (!bVisible)
                {
                    modelColumn.removeColumn(col);
                }
                else
                if (!wCol.isVisible())
                {
                    modelColumn.addColumn(col);
                }
            }
        }

        if (bChanged == false)
        {
            // no preferences yet...
            int nWidthTable = getViewportWidth();

            // size to preferred, if they have been defined...
            if (nWidthTable != 0 && sizeColumnsToPreferredWidths(nWidthTable) == false)
            {
                // if no column size definitions exist, default sizes.
                sizeColumnsToTableWidth();
            }
        }
    }

    protected void saveColumnPrefs()
    {
        if (m_bRealized == false)
        {
            return;
        }

        ExtendedTableColumnModel modelColumn = (ExtendedTableColumnModel) JRowTable.this.getColumnModel();

        if (m_strPreferenceKey != null && modelColumn != null)
        {
            // save sorting preferences...
            IModelTableModel modelTable = (IModelTableModel)getModel();
            PreferenceManager.getInstance().setInt("table", m_strPreferenceKey + ".sortColumn", modelTable.getSortColumn(), false);
            PreferenceManager.getInstance().setBoolean("table", m_strPreferenceKey + ".sortAscending", modelTable.isSortedInAscendingOrder(), false);

            int nCols = modelColumn.getWrappedColumnCount();

            for (int nCol = 0; nCol < nCols; nCol++)
            {
                TableColumn col = modelColumn.getWrappedColumn(nCol).getTableColumn();

                if (col != null)
                {
                    // save column width...
                    PreferenceManager.getInstance().setInt("table", m_strPreferenceKey + "." + modelColumn.getWrappedColumnName(nCol) + ".width", col.getWidth(), false);

                    // save column visibility...
                    ExtendedTableColumnModel model = (ExtendedTableColumnModel) JRowTable.this.getColumnModel();
                    boolean bVisible = model.isColumnVisible(col);
                    PreferenceManager.getInstance().setBoolean("table", m_strPreferenceKey + "." + modelColumn.getWrappedColumnName(nCol) + ".visible", bVisible, false);
                }
            }
            PreferenceManager.getInstance().flush("table");
        }
    }

    public void sizeColumnsToTableWidth()
    {
        int nWidthTable = getViewportWidth();
        int remainder   = nWidthTable;

        if (nWidthTable != 0)
        {
            ExtendedTableColumnModel extModel = (ExtendedTableColumnModel) getColumnModel();

            for (int k = 0; k < extModel.getWrappedColumnCount(); k++)
            {
                WrappedColumn columnModel = extModel.getWrappedColumn(k);

                if (columnModel.isVisible())
                {
                    int columnWidth = Math.round(nWidthTable / extModel.getColumnCount());

                    // remove this columns width from the remaining available space
                    remainder -= columnWidth;

                    // if we are dealing with the last column then append any
                    // remaining space to it so we don't get a gap on the right
                    if (k == (extModel.getWrappedColumnCount() - 1))
                    {
                        columnWidth += remainder;
                    }

                    columnModel.getTableColumn().setPreferredWidth(columnWidth);
                }
            }
        }
    }

    public boolean sizeColumnsToPreferredWidths(int nWidthTable)
    {
        boolean bSized = false;

        if (nWidthTable > 0)
        {
            ExtendedTableColumnModel extModel = (ExtendedTableColumnModel) getColumnModel();

            for (int k = 0; k < extModel.getWrappedColumnCount(); k++)
            {
                WrappedColumn columnModel = extModel.getWrappedColumn(k);
                TableColumn col = columnModel.getTableColumn();

                if (col.getPreferredWidth() != -1)
                {
                    col.setPreferredWidth((int) Math.round(nWidthTable * (col.getPreferredWidth() * .01)));
                    bSized = true;
                }
            }
        }

        return bSized;
    }

    protected JPopupMenu createHeaderPopupMenu()
    {
        TableColumnModel model = getColumnModel();
        final JPopupMenu pm = new JPopupMenu("header-popupmenu");

        pm.add(new ColumnResetAction());
        pm.addSeparator();
        pm.addPopupMenuListener(new PopupMenuListener()
        {
            @Override
            public void popupMenuCanceled(PopupMenuEvent evt)
            {}

            @Override
            public void popupMenuWillBecomeInvisible(PopupMenuEvent evt)
            {}

            @Override
            public void popupMenuWillBecomeVisible(PopupMenuEvent evt)
            {
                for (int i = 0; i < pm.getComponentCount(); i++)
                {
                    Component item = pm.getComponent(i);

                    if (item instanceof AbstractButton)
                    {
                        TableAction action = (TableAction) ((AbstractButton) item).getAction();

                        if (action != null)
                        {
                            item.setEnabled(action.isEnabled());

                            if (item instanceof JCheckBoxMenuItem)
                            {
                                ((JCheckBoxMenuItem) item).setSelected(action.isSelected());
                            }
                        }
                    }
                }
            }
        });

        ExtendedTableColumnModel extModel = (ExtendedTableColumnModel) getColumnModel();
        for (int k = 0; k < extModel.getWrappedColumnCount(); k++)
        {
            WrappedColumn columnModel = extModel.getWrappedColumn(k);
            Action action = new ColumnKeeperAction(columnModel.getTableColumn().getIdentifier(), columnModel.getTableColumn().getHeaderValue());
            JCheckBoxMenuItem item = new JCheckBoxMenuItem(action);

            item.setSelected(columnModel.isVisible());

            pm.add(item);
        }

        pm.addSeparator();

        Action action = new ColumnFillAction();
        pm.add(new JMenuItem(action));

        return pm;
    }

    class TableAction
        extends AbstractAction
    {
        public TableAction(String name)
        {
            super(name);
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {}

        public boolean isSelected()
        {
            return false;
        }
    }

    class ColumnKeeperAction
        extends TableAction
    {
        protected Object m_identifier;

        public ColumnKeeperAction(Object identifier, Object headerValue)
        {
            super(headerValue.toString());

            m_identifier = identifier;
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            JCheckBoxMenuItem item = (JCheckBoxMenuItem) evt.getSource();
            ExtendedTableColumnModel model = (ExtendedTableColumnModel) JRowTable.this.getColumnModel();
            TableColumn column = model.getTableColumnExt(m_identifier);

            if (item.isSelected())
            {
                model.addColumn(column);
            }
            else
            {
                model.removeColumn(column);

            }
            tableChanged(new TableModelEvent(getModel(), TableModelEvent.HEADER_ROW));
            repaint();
        }

        @Override
        public boolean isSelected()
        {
            ExtendedTableColumnModel model = (ExtendedTableColumnModel) getColumnModel();
            TableColumn column = model.getTableColumnExt(m_identifier);

            return model.isColumnVisible(column);
        }

        @Override
        public boolean isEnabled()
        {
            ExtendedTableColumnModel model = (ExtendedTableColumnModel) getColumnModel();
            TableColumn column = model.getTableColumnExt(m_identifier);

            return!((model.getColumnCount() == 1) && model.isColumnVisible(column));
        }

    }

    class ColumnResetAction
        extends TableAction
    {
        public ColumnResetAction()
        {
            super("View All Columns");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            ExtendedTableColumnModel model = (ExtendedTableColumnModel) getColumnModel();

            model.resetToDefaults();
        }

        @Override
        public boolean isEnabled()
        {
            ExtendedTableColumnModel model = (ExtendedTableColumnModel) getColumnModel();

            return!model.isDefaulted();
        }
    }

    class ColumnFillAction
        extends TableAction
    {
        public ColumnFillAction()
        {
            super("Fill");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            sizeColumnsToTableWidth();
        }

        @Override
        public boolean isEnabled()
        {
            return true;
        }
    }

    class ExtendedTableColumnModel extends DefaultTableColumnModel
    {
        private Vector m_all = new Vector();

        public ExtendedTableColumnModel()
        {
            super();
        }

        public int getWrappedColumnCount()
        {
            return m_all.size();
        }

        public String getWrappedColumnName(int nCol)
        {
            String strName = null;

            if (nCol < m_all.size())
            {
                strName = (String) ((WrappedColumn) m_all.get(nCol)).getTableColumn().getHeaderValue();
            }

            return strName;
        }

        public void dropAllColumns()
        {
            while (getColumnCount() > 0)
            {
                removeColumn(getColumn(0));
            }

            m_all.clear();
        }

        public WrappedColumn getWrappedColumn(int nCol)
        {
            WrappedColumn col = null;

            if (nCol < m_all.size())
            {
                col = (WrappedColumn)m_all.get(nCol);
            }

            return col;
        }

        @Override
        public void addColumn(TableColumn column)
        {
            int index = findWrappedColumn(column);
            WrappedColumn wc = (index >= 0) ? (WrappedColumn) m_all.get(index) : null;
            boolean add = false;

            if (wc == null)
            {
                wc = new WrappedColumn(column);

                m_all.add(wc);

                wc.setInitialPosition(m_all.indexOf(wc));

                add = true;
            }
            else
            if (!wc.isVisible())
            {
                add = true;
                wc.setVisible(true);
            }

            if (add)
            {
                // The following code finds the correct insertion point for the
                // "now" visible column...
                //
                int insertIndex = 0;

                for (int i = 0; i < m_all.size(); i++)
                {
                    WrappedColumn tmpWC = (WrappedColumn)m_all.get(i);

                    if (tmpWC.getTableColumn() == column)
                    {
                        break;
                    }
                    else
                    if (tmpWC.isVisible())
                    {
                        insertIndex++;
                    }
                }

                // There is no way of inserting a column at a particular index
                // with the DefaultTableColumnModel so we simply add it and then
                // move it to the correct location (using the superclass methods
                // so as to avoid the m_all vector house keeping.
                //
                super.addColumn(column);
                super.moveColumn(getColumnCount() - 1, insertIndex);
            }
        }

        @Override
        public void removeColumn(TableColumn column)
        {
            int index = findWrappedColumn(column);
            WrappedColumn wc = (index >= 0) ? (WrappedColumn)m_all.get(index) : null;

            if (wc != null)
            {
                if (wc.isVisible())
                {
                    wc.setVisible(false);

                    super.removeColumn(column);
                }
            }
        }

        @Override
        public void moveColumn(int columnIndex, int newIndex)
        {
            if (columnIndex != newIndex)
            {
                WrappedColumn wc = (WrappedColumn)m_all.remove(columnIndex);

                if (wc != null)
                {
                    m_all.add(newIndex, wc);
                }
            }

            super.moveColumn(columnIndex, newIndex);
        }

        public void resetToDefaults()
        {
            for (int i = 0; i < m_all.size(); i++)
            {
                WrappedColumn wc = (WrappedColumn) m_all.get(i);

                addColumn(wc.getTableColumn());
            }

            WrappedColumn[] tmp = (WrappedColumn[]) m_all.toArray(new WrappedColumn[m_all.size()]);

            for (int j = 0; j < tmp.length; j++)
            {
                WrappedColumn wc = tmp[j];

                int initialPosition = wc.getInitialPosition();
                int currentPosition = getColumnIndex(wc.getTableColumn().getIdentifier());

                if (currentPosition != initialPosition)
                {
                    moveColumn(currentPosition, initialPosition);
                }
            }
        }

        private int findWrappedColumn(TableColumn column)
        {
            for (int i = 0; i < m_all.size(); i++)
            {
                WrappedColumn wc = (WrappedColumn) m_all.get(i);

                if (wc.getTableColumn() == column)
                {
                    return i;
                }
            }

            return -1;
        }

        public boolean isColumnVisible(TableColumn column)
        {
            int i = findWrappedColumn(column);
            WrappedColumn wc = (i >= 0) ? (WrappedColumn) m_all.get(i) : null;

            return (wc != null) ? wc.isVisible() : false;
        }

        public TableColumn getTableColumnExt(Object identifier)
        {
            for (int i = 0; i < m_all.size(); i++)
            {
                WrappedColumn wc = (WrappedColumn) m_all.get(i);

                if (wc.getTableColumn().getIdentifier() == identifier)
                {
                    return wc.getTableColumn();
                }
            }

            return null;
        }

        public boolean isDefaulted()
        {
            for (int i = 0; i < m_all.size(); i++)
            {
                WrappedColumn wc = (WrappedColumn) m_all.get(i);

                if (!wc.isVisible() || (wc.getInitialPosition() != i))
                {
                    return false;
                }
            }

            return true;
        }

        private String debug()
        {
            StringBuffer sb = new StringBuffer();

            for (int i = 0; i < m_all.size(); i++)
            {
                WrappedColumn wc = (WrappedColumn) m_all.get(i);

                if (i > 0)
                {
                    sb.append(", ");

                }
                if (!wc.isVisible())
                {
                    sb.append("*");

                }
                sb.append(wc.getTableColumn().getHeaderValue());
                sb.append("(").append(wc.getInitialPosition()).append(")");
            }

            return sb.toString();
        }

    }

    class WrappedColumn
    {
        private TableColumn m_column;
        private boolean m_visible;
        private int m_initialPosition = -1;
        
        public WrappedColumn(TableColumn column)
        {
            this(column, true);
        }

        public WrappedColumn(TableColumn column, boolean visible)
        {
            m_column = column;
            m_visible = visible;
        }

        public TableColumn getTableColumn()
        {
            return m_column;
        }

        public void setVisible(boolean visible)
        {
            m_visible = visible;
        }

        public boolean isVisible()
        {
            return m_visible;
        }

        public int getInitialPosition()
        {
            return m_initialPosition;
        }

        public void setInitialPosition(int initialPosition)
        {
            if (m_initialPosition == -1)
            {
                m_initialPosition = initialPosition;
            }
        }
    }

    private int limit(int i, int a, int b)
    {
        return Math.min(b, Math.max(i, a));
    }

    //-------------------------------------------------------------------------
    //
    // (Non)Focus Selection Support
    //
    // When the table looses focus any selected rows must change color to
    // reflect the loss of focus. This is standard behavior on many platforms
    // but for some reason Swing does not support it by default.
    //
    //-------------------------------------------------------------------------

    public boolean isFocusLossTemporary()
    {
        return m_focusLossTemporary;
    }

    private class FocusHandler
        implements FocusListener
    {
        /**
         * For the table we have to force the repaint of the entire
         * selected row because the table is cell-based not row-
         * based and would only repaint the selected (focused) cell
         * if we left it to its own devices.
         */
        private void updateForFocus()
        {
            JRowTable table = JRowTable.this;

            int[] sel = table.getSelectedRows();

            if ((sel != null) && (sel.length > 0))
            {
                for (int i = 0; i < sel.length; i++)
                {
                    // Defect Fix: Sonic00017156
                    // Used to call this code to get the table cells to repaint
                    // but we are actually getting the model to issue a table
                    // change event...not good.
                    //
                    //   ((IModelTableModel)table.getModel()).rowChanged(sel[i]);
                    //
                    // So we now just find the location of each selected row
                    // and ask for a repaint which is actually what we really
                    // want (no side affects).
                    // This could be optimized to issue a single repaint event
                    // but the repaints are coallesed anyway.
                    Rectangle r = table.getCellRect(sel[i], -1, true);
                    r.width = table.getWidth();
                    table.repaint(r);
                }
            }
        }

        @Override
        public void focusGained(FocusEvent evt)
        {
            updateForFocus();
        }

        @Override
        public void focusLost(FocusEvent evt)
        {
            m_focusLossTemporary = (evt.isTemporary() && (evt.getOppositeComponent() != null));

            updateForFocus();
        }
    }

    //-------------------------------------------------------------------------

    /**
     *  HeaderCellRenderer works with ModelListTableModel, adding graphical
     *  annotations to header cells to show which columns are sortable, and
     *  which column the table is sorted by (if any), and whether it's sorted
     *  in ascending or descending order.
     */
    private class HeaderCellRenderer
        extends ExtendedJTable.ExtendedTableHeaderRenderer
    {
        private Icon m_iconAscend;
        private Icon m_iconDescend;
        private IModelTableModel m_tableModel;
        private TableColumn m_tableColumn;
        private boolean m_bColumnSortable = false;

        HeaderCellRenderer(IModelTableModel tableModel,
                           TableColumn tableColumn)
        {
            m_tableModel = tableModel;
            m_tableColumn = tableColumn;
            m_iconAscend = new TableArrowIcon(true);
            m_iconDescend = new TableArrowIcon(false);
            m_bColumnSortable = tableColumn instanceof RowTableColumn &&
                ((RowTableColumn)tableColumn).isSortable();

            setProperties();
        }
        
        private void setProperties() {
            setHorizontalAlignment(JLabel.CENTER);
            setHorizontalTextPosition(JLabel.LEADING);
        }

        @Override
        public Component getTableCellRendererComponent(JTable table,
            Object value,
            boolean isSelected,
            boolean hasFocus,
            int row,
            int column)
        {
            Component c = super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);

            if (table != null)
            {
                JTableHeader header = table.getTableHeader();

                if (header != null)
                {
                    setForeground(header.getForeground());
                    setBackground(header.getBackground());
                    setFont(header.getFont());
                }
            }

            setText((value == null) ? "" : value.toString());
            setBorder(UIManager.getBorder("TableHeader.cellBorder"));

            return c;
        }

        @Override
        public void paint(Graphics g)
        {
            super.paint(g);

            g.setColor(getForeground());

            // NOTE WELL: sortability on this column is determined when this renderer is.
            if (m_bColumnSortable && m_tableModel.isSortActivated())
            {
                int sortColumn = m_tableModel.getSortColumn();

                if (sortColumn >= 0)
                {
                    TableColumn colSort = m_tableModel.getColumn(sortColumn);

                    if (colSort == m_tableColumn)
                    {
                        Dimension size = this.getSize();
                        Icon icon = m_tableModel.isSortedInAscendingOrder() ? m_iconAscend : m_iconDescend;

                        icon.paintIcon(this, g, size.width - icon.getIconWidth() - 4,
                            (size.height - icon.getIconHeight() - 2) / 2);

                        return;
                    }
                }

                setIcon(null);
            }
        }
    }

    private class TableArrowIcon implements Icon
    {
        private boolean m_isAscending;

        TableArrowIcon(boolean isAscending)
        {
            m_isAscending = isAscending;
        }

        @Override
        public int getIconHeight()
        {
            return 5;
        }

        @Override
        public int getIconWidth()
        {
            return 9;
        }

        @Override
        public void paintIcon(Component c, Graphics g, int x, int y)
        {
            if (m_isAscending)
            {
                paintIconAscending(g, x, y);
            }
            else
            {
                paintIconDescending(g, x, y);
            }
        }

        private void paintIconAscending(Graphics g, int x, int y)
        {
            //{0, 0, 0, 2, 2, 0, 0, 0, 0},
            //{0, 0, 2, 1, 1, 1, 0, 0, 0},
            //{0, 2, 1, 1, 1, 1, 1, 0, 0},
            //{2, 1, 1, 1, 1, 1, 1, 1, 0},
            //{0, 3, 3, 3, 3, 3, 3, 3, 3}
            g.setColor((Color) UIManager.getDefaults().get("infoText"));
            g.drawLine(x, y + 3, x + 3, y);
            g.drawLine(x + 4, y, x + 4, y);
            g.setColor((Color) UIManager.getDefaults().get("controlDkShadow"));
            g.drawLine(x + 3, y + 1, x + 5, y + 1);
            g.drawLine(x + 2, y + 2, x + 6, y + 2);
            g.drawLine(x + 1, y + 3, x + 7, y + 3);
            g.setColor((Color) UIManager.getDefaults().get("controlLtHighlight"));
            g.drawLine(x + 1, y + 4, x + 8, y + 4);
        }

        private void paintIconDescending(Graphics g, int x, int y)
        {
            //{2, 2, 2, 2, 2, 2, 2, 2},
            //{0, 1, 1, 1, 1, 1, 1, 3},
            //{0, 0, 1, 1, 1, 1, 3, 0},
            //{0, 0, 0, 1, 1, 3, 0, 0},
            //{0, 0, 0, 0, 3, 0, 0, 0}
            g.setColor((Color) UIManager.getDefaults().get("infoText"));
            g.drawLine(x, y, x + 7, y);
            g.setColor((Color) UIManager.getDefaults().get("controlDkShadow"));
            g.drawLine(x + 1, y + 1, x + 6, y + 1);
            g.drawLine(x + 2, y + 2, x + 5, y + 2);
            g.drawLine(x + 3, y + 3, x + 4, y + 3);
            g.setColor((Color) UIManager.getDefaults().get("controlLtHighlight"));
            g.drawLine(x + 7, y + 1, x + 4, y + 4);
        }
    }

    //-------------------------------------------------------------------------

    /**
     * TableSorter is intended to be used as a MouseListener on a JTableHeader
     * in order to interpret mouse-clicks as table sorting requests.  It can
     * only be used with IModelTableModels, and cannot be used with JTables
     * that use any other kind of TableModel.
     */
    class TableSorter
        extends MouseAdapter
    {
        private ExtendedJTable table;
        private boolean popupTrigger;

        public TableSorter(ExtendedJTable table)
        {
            this.table = table;
        }

        @Override
        public void mousePressed(MouseEvent event)
        {
            popupTrigger = event.isPopupTrigger();
        }

        @Override
        public void mouseReleased(MouseEvent event)
        {
            popupTrigger = event.isPopupTrigger();
        }

        @Override
        public void mouseClicked(MouseEvent event)
        {
            // Catch if the popupTrigger is set on a mousePressed and mouseReleased
            // event so that we can use it here...because we don't want to perform
            // sorting on a popupTrigger event...that is exclusively for the popup
            // menu.
            //
            if (popupTrigger || (event.getClickCount() != 1))
            {
                return;
            }

            IModelTableModel dm = (IModelTableModel) table.getModel();
            TableColumnModel cm = table.getColumnModel();

            int viewColumn = cm.getColumnIndexAtX(event.getX());
            int modelColumn = table.convertColumnIndexToModel(viewColumn);

            if (modelColumn != -1)
            {
                Object[] selectedObjects = getSelectedObjects(dm);

                try
                {
                    table.getSelectionModel().setValueIsAdjusting(true);
                    table.clearSelection();
                    dm.sortByColumn(modelColumn);
                    restoreObjectSelection(selectedObjects, dm);
                }
                finally
                {
                    table.getSelectionModel().setValueIsAdjusting(false);
                }

                table.getTableHeader().repaint();
            }
        }

        protected Object[] getSelectedObjects(IModelTableModel tableModel)
        {
            int[] selectedRows = table.getSelectedRows();
            int num = selectedRows.length;
            Object[] selectedObjects = new Object[num];
            for (int i = 0; i < num; i++)
            {
                selectedObjects[i] = tableModel.getRowModel(selectedRows[i]);
            }
            return (selectedObjects);
        }

        protected void restoreObjectSelection(Object[] selectedObjects,
                                              IModelTableModel tableModel)
        {
            // table selection is expected to be cleared at this point
            int num = selectedObjects.length;
            for (int i = 0; i < num; i++)
            {
                Object obj = selectedObjects[i];
                int row = ((obj == null) ? -1 : tableModel.getIndexOf(obj));
                if (row >= 0)
                {
                    table.getSelectionModel().setValueIsAdjusting(true);
                    table.addRowSelectionInterval(row, row);
                }
            }
        }
    }
}