/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui.util;

import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.ListSelectionModel;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.table.IModelTableModel;

public class JRowTableButtonPanel extends    JButtonPanel
                                  implements ListSelectionListener, DocumentListener
{
    private static final String CMD_ADD    = "JRowTableButtonPanel.add";
    private static final String CMD_EDIT   = "JRowTableButtonPanel.edit";
    private static final String CMD_REMOVE = "JRowTableButtonPanel.remove";

    private boolean m_activateTableSelection = true;

    private JRowTable m_table;
    private JButton   m_add  = null;
    private JButton   m_edit = null;
    private JButton   m_del  = null;
    private boolean   m_selDependent = true;


    public JRowTableButtonPanel(JRowTable table)
    {
        this(table, true);
    }

    public JRowTableButtonPanel(JRowTable table, boolean autoEdit)
    {
        this(table, autoEdit, JPartitionPanel.DEFAULT_GAP_SIZE, 0, 0);
    }

    public JRowTableButtonPanel(JRowTable table, int gapSize, int tbEdgeGap, int lrEdgeGap)
    {
        this(table, true, gapSize, tbEdgeGap, lrEdgeGap);
    }

    public JRowTableButtonPanel(JRowTable table, boolean autoEdit,
                                int gapSize, int tbEdgeGap, int lrEdgeGap)
    {
        super(true, gapSize, tbEdgeGap, lrEdgeGap);

        m_table = table;

        m_table.getSelectionModel().addListSelectionListener(this);

        if (autoEdit)
        {
            m_table.addMouseListener(new MouseAdapter()
            {
                @Override
                public void mouseClicked(MouseEvent evt)
                {
                    if(!m_activateTableSelection) return;

                    if (getEditAction()  == null)
                    {
                        return;
                    }

                    JRowTable table = (JRowTable)evt.getSource();

                    if ((evt.getClickCount() == 2) && (table.getSelectedRowCount() == 1))
                    {
                        getEditAction().actionPerformed(null);
                    }
                }
            });
        }
        m_table.getModel().addTableModelListener(new TableModelListener()
        {
            @Override
            public void tableChanged(TableModelEvent evt)
            {
                if(!m_activateTableSelection) return;

                updateSelectionModelState();
            }
        });

        setRemoveAction(new RemoveAction());

        updateState((table != null) ? table.getSelectionModel() : null);
    }

    private void updateSelectionModelState() {
        updateState(m_table.getSelectionModel());
    }
    
    protected JRowTable getTable()
    {
        return m_table;
    }

    public Action getAddAction()
    {
        return ((getButton(JAddButton.class) != null) ?
            getButton(JAddButton.class).getAction() : null);
    }

    public final Action getEditAction()
    {
        return ((getButton(JEditButton.class) != null) ?
            getButton(JEditButton.class).getAction() : null);
    }

    public Action getRemoveAction()
    {
        return ((getButton(JRemoveButton.class) != null) ?
            getButton(JRemoveButton.class).getAction() : null);
    }

    public void setAddAction(BasicResourceAction action)
    {
        JButton btn = getButton(JAddButton.class);

        if (btn != null)
        {
            btn.setAction(action);
        }
        else
        {
            this.add(new JAddButton(action), 0);
        }

        updateState(null);
    }

    public void setEditAction(BasicResourceAction action)
    {
        JButton btn = getButton(JEditButton.class);

        if (btn != null)
        {
            btn.setAction(action);
        }
        else
        {
            int insertIndex = (getButton(JAddButton.class) != null) ? 1 : 0;

            add(new JEditButton(action), insertIndex);
        }

        updateState(null);
    }

    public final void setRemoveAction(BasicResourceAction action)
    {
        JButton btn = getButton(JRemoveButton.class);

        if (btn != null)
        {
            btn.setAction(action);
        }
        else
        {
            add(new JRemoveButton(action));
        }

        updateState(null);
    }

    /**
     * If this method is called with "true" then it disables the updating of
     * the buttons as a result of table selection changes...this is done when
     * the JRowTableButtonPanel is registered as a DocumentListener.
     */
    public void setSelectionDependent(boolean state)
    {
        m_selDependent = state;
    }

    //---------------------------------------------------------------------
    //
    // ListSelectionListener implementation
    //
    //---------------------------------------------------------------------

    @Override
    public void valueChanged(ListSelectionEvent evt)
    {
        if (m_selDependent && m_activateTableSelection)
        {
            updateState((ListSelectionModel)evt.getSource());
        }
    }

    //---------------------------------------------------------------------
    //
    // DocumentListener implementation
    //
    //---------------------------------------------------------------------

    @Override
    public void changedUpdate(DocumentEvent evt)
    {
        updateState(m_table.getSelectionModel());
    }

    @Override
    public void insertUpdate(DocumentEvent evt)
    {
        updateState(m_table.getSelectionModel());
    }

    @Override
    public void removeUpdate(DocumentEvent evt)
    {
        updateState(m_table.getSelectionModel());
    }

    //---------------------------------------------------------------------

    public void enableComponent(boolean state)
    {
        m_activateTableSelection = state;
        m_table.setEnabled(state);
        m_table.setSortable(state);

        if(state)
        {
            updateState(m_table.getSelectionModel());
        }
        else
        {
            getAddAction().setEnabled(false);
            getEditAction().setEnabled(false);
            getRemoveAction().setEnabled(false);
        }
    }

    protected void updateState(ListSelectionModel selModel)
    {
        if(!m_activateTableSelection) return;

        int selCount = (selModel != null) ? getSelectedRowCount(selModel) : 0;

        if (getAddAction() != null)
        {
            getAddAction().setEnabled(true);
        }

        if (getEditAction() != null)
        {
            getEditAction().setEnabled(m_selDependent ? (selCount == 1) : true);
        }

        if (getRemoveAction() != null)
        {
            getRemoveAction().setEnabled((selCount > 0));
        }
    }

    protected int getSelectedRowCount(ListSelectionModel selModel)
    {
        int iMin  = selModel.getMinSelectionIndex();
        int iMax  = selModel.getMaxSelectionIndex();
        int count = 0;

        for (int i = iMin; i <= iMax; i++)
        {
            if (selModel.isSelectedIndex(i))
            {
                count++;
            }
        }

        return count;
    }

    //-------------------------------------------------------------------------
    //
    // Inner classes
    //
    //-------------------------------------------------------------------------

    class JAddButton extends JButton
    {
        public JAddButton(BasicResourceAction action)
        {
            super(action);
        }
    }

    class JEditButton extends JButton
    {
        public JEditButton(BasicResourceAction action)
        {
            super(action);
        }
    }

    class JRemoveButton extends JButton
    {
        public JRemoveButton(BasicResourceAction action)
        {
            super(action);
        }
    }

    class RemoveAction extends BasicGuiAction
    {
        RemoveAction()
        {
            super(CMD_REMOVE);
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            try
            {
                int[] selRow = m_table.getSelectedRows();

                for (int i = selRow.length - 1; i >= 0; i--)
                {
                    // We do these checks here because of a selection bug in
                    // jdk1.4.2 - the selection model doesn't always get updated
                    //
                    if (m_table.getSelectionModel().isSelectedIndex(selRow[i]) &&
                        (selRow[i] < m_table.getRowCount()))
                    {
                        ((IModelTableModel)m_table.getModel()).deleteRow(selRow[i]);
                    }
                }

                // Maintain selection if possible
                //
                int anchorIndex = m_table.getSelectionModel().getAnchorSelectionIndex();

                if (m_table.getRowCount() == 0)
                {
                    updateState(null);
                }

                if (anchorIndex < m_table.getRowCount())
                {
                    m_table.getSelectionModel().setSelectionInterval(anchorIndex, anchorIndex);
                }
                else
                if (m_table.getRowCount() > 0)
                {
                    m_table.getSelectionModel().setSelectionInterval(m_table.getRowCount() - 1, m_table.getRowCount() - 1);
                }

            }
            catch (Exception e)
            {
                MgmtConsole.displayMessage(MgmtConsole.MESSAGE_ERROR, "Error deleting row(s).", e, true);
            }
        }
    }
}
