package com.sonicsw.ma.gui.util;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import modelobjects.framework.EditRule;
import modelobjects.framework.ModelEditMediator;
import modelobjects.framework.UndoableModelAspectEdit;
import modelobjects.framework.ViewValueConversionException;
import modelobjects.framework.model.ModelAspectId;
import modelobjects.framework.model.ModelObjectAdapter;
import modelobjects.util.ObjectCopier;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.table.IModelTableModel;
import com.sonicsw.ma.gui.table.JRowTableAspectAdapter;
import com.sonicsw.ma.plugin.ConfigBeanModel;
import com.sonicsw.mx.config.ConfigFactory;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IConfigPath;
import com.sonicsw.mx.config.util.ConfigHelper;

/**
 *  JTableListAspectAdapter is a special kind of ViewAspectAdapter that
 *  is used for List-valued model-aspects that are edited in JTables.
 *  This class should be used only with care, since it must make a complete
 *  copy of the model-aspect List and of the contents of the List in order
 *  to be sure that edits made in the table (inserts, deletes, cell-edits)
 *  are both discardable in the case of reset, and undoable and redoable.
 *
 *  Note that JTableListAspectAdapter can only be used with JTables that
 *  use ModelListTableModel for their TableModels.
 *
 *  JTableListAspectAdapter supports PropertyChangeListeners that may want
 *  to track when it is editable in order to enable/disable editing actions.
 */
public class JTableAttributeMapAspectAdapter extends    JRowTableAspectAdapter
                                             implements ObjectCopier
{
    /**
     *  Construct a JTableAttributeMapAspectAdapter.
     *
     *  @param modelAspectId the ModelAspectId of the model-aspect being edited
     *  @param table the JRowTable component in which the editing occurs
     *  @param rowModelCopier the ObjectCopier used to make copies of row models
     *  @param editRule the editRule for the ViewAspectAdapter
     *  @param modelEditMediator the ModelEditMediator managing this adapter
     *  @param type the config type (AttributeDescrptionMap) for each row
     */
    public JTableAttributeMapAspectAdapter(ModelAspectId     modelAspectId,
                                           JRowTable         table,
                                           EditRule          editRule,
                                           ModelEditMediator modelEditMediator)
    {
        super(modelAspectId, table, null, editRule, modelEditMediator);

        rowModelCopier = this;
    }

    @Override
    protected void setViewAspectValue(Object viewAspectValue)
    throws IllegalArgumentException
    {
        if (viewAspectValue == null)
        {
            return;
        }

        if (!(viewAspectValue instanceof IAttributeMap))
        {
            throw new IllegalArgumentException("The value for view aspect '" + getModelAspectId() + "' must be of type IAttributeMap!");
        }

        try
        {
            IAttributeMap map      = (IAttributeMap)((IAttributeMap)viewAspectValue).clone();
            ArrayList     contents = new ArrayList(map.size());
            Iterator      i        = map.keySet().iterator();

            while (i.hasNext())
            {
                String        key    = (String)i.next();
                IAttributeMap rowMap = (IAttributeMap)map.getAttribute(key);

                if (rowMap != null)
                {
                    contents.add(new ConfigBeanModel(rowMap, rowMap.getAttributeDescription(), key));
                }
            }

            super.setViewAspectValue(contents);
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR, "setViewAspectValue failure for aspect: '" + getModelAspectId() + "'", e, false);
        }
    }

    @Override
    public Object getViewAspectValue()
    {
        IAttributeMap map = null;

        try
        {
            List contents = ((IModelTableModel)table.getModel()).getContents();

                Iterator i = contents.iterator();

                map = getParentModel();

                while (i.hasNext())
                {
                    ConfigBeanModel rowValue = (ConfigBeanModel)i.next();

                    map.setAttribute(rowValue.getViewName(), rowValue.getData());
                }
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR, "getViewAspectValue failure for aspect: '" + getModelAspectId() + "'", e, false);
        }

        return map;
    }

    @Override
    public Object copyObject(Object obj)
    {
        if (obj instanceof IAttributeMap)
        {
            return ((IAttributeMap)obj).clone();
        }

        return obj;
    }

    /**
     * Return an UndoableModelAspectEdit representing the edits made to the
     * IAttributeMap model aspect value since the last time setModelAspectValue
     * was called.
     *
     * We're over-riding this method here because the default implementation
     * in ViewAspectAdapter does a reference comparision between the new and
     * old model aspect values...and in this case the references to the
     * parent IAttributeMap's will be the same....so this method ommits the
     * reference equality check.
     */
    @Override
    public UndoableModelAspectEdit makeUndoableModelAspectEdit(ModelObjectAdapter modelObjectAdapter)
        throws ViewValueConversionException
    {
        return new UndoableModelAspectEdit("changed " + getModelAspectId(),
                                           modelObjectAdapter,
                                           getModelAspectId(),
                                           getOldModelAspectValue(),
                                           getModelAspectValue());
    }

    private IAttributeMap getParentModel()
    {
        IConfigPath     path  = ConfigFactory.createConfigPath(getModelAspectId().getName());
        ConfigBeanModel model = (ConfigBeanModel)getModelEditMediator().getModel().getModelObject();
        IAttributeMap   map   = model.getData();
        IAttributeMap   parent = (IAttributeMap)map.getAttribute(path);

        if (parent == null)
        {
            parent = (IAttributeMap)ConfigHelper.createChild(map, path);
        }
        else
        {
            parent.clear();
        }

        return parent;
    }

}
