package com.sonicsw.ma.gui.util;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.HeadlessException;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.net.URI;
import java.net.URISyntaxException;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;

import modelobjects.layout.PartitionLayout;

import com.sonicsw.ma.gui.MgmtConsole;

public class JURIEditor
    extends JComponent
    implements KeyListener
{
    // Dialog Return Values
    //
    public static final int CANCEL_OPTION = 1;
    public static final int APPROVE_OPTION = 0;
    public static final int ERROR_OPTION = -1;

    private static final int MODE_NOP = 0;
    private static final int MODE_EDIT = 1;
    private static final int MODE_NEW = 2;

    private static int PREF_WIDTH = 500;
    private static int PREF_HEIGHT = 326;
    private static Dimension PREF_SIZE = new Dimension(PREF_WIDTH, PREF_HEIGHT);

    private static int MIN_WIDTH = 500;
    private static int MIN_HEIGHT = 326;
    private static Dimension MIN_SIZE = new Dimension(MIN_WIDTH, MIN_HEIGHT);

    JDialog dialog;
    JComboBox m_scheme;
    JTextField m_specific;
    JTextField m_fragment;
    JTextField m_authority;
    JTextField m_path;
    JTextField m_query;
    JTextField m_userInfo;
    JTextField m_host;
    JTextField m_port;
    JButton btnOk;
    JButton btnCancel;
    int returnValue;
    int mode;
    URI oldURI;
    boolean formChanged;

    public JURIEditor()
    {
        super();

        dialog = null;
        mode = MODE_NOP;
        formChanged = false;

        prepareJURIEditor();
    }
    private void prepareJURIEditor() {
        
        setLayout(new BorderLayout());

        add(createURIPanel(), BorderLayout.CENTER);
        add(createButtonPanel(), BorderLayout.SOUTH);

        formUpdate();
    }

    public int showDialog(Component parent)
        throws HeadlessException
    {
        dialog = createDialog(parent);

        dialog.addWindowListener(new WindowAdapter()
        {
            @Override
            public void windowClosing(WindowEvent e)
            {
                returnValue = CANCEL_OPTION;
            }
        });

        returnValue = ERROR_OPTION;

        dialog.show();
        dialog.dispose();
        dialog = null;

        return returnValue;
    }

    protected JDialog createDialog(Component parent)
        throws HeadlessException
    {
        Frame frame = parent instanceof Frame ? (Frame)parent
            : (Frame)SwingUtilities.getAncestorOfClass(Frame.class, parent);

        JDialog dialog = new JDialog(frame, "URI Editor", true);

        Container contentPane = dialog.getContentPane();
        contentPane.setLayout(new BorderLayout());
        contentPane.add(this, BorderLayout.CENTER);

        this.setPreferredSize(PREF_SIZE);
        this.setMinimumSize(MIN_SIZE);

        dialog.pack();
        dialog.setLocationRelativeTo(parent);

        return dialog;
    }

    public void setURI(URI uri)
    {
        oldURI = uri;

        String sScheme = uri.getScheme();
        String sPart = uri.getSchemeSpecificPart();
        String sFragment = uri.getFragment();

        m_scheme.setSelectedItem((sScheme != null) ? sScheme : "");
        m_specific.setText((sPart != null) ? sPart : "");
        m_fragment.setText((sFragment != null) ? sFragment : "");
    }

    public URI getURI()
    {
        try
        {
            StringBuffer sb = new StringBuffer();

            String sScheme = ((String)m_scheme.getSelectedItem()).trim();
            String sPart = m_specific.getText().trim();
            String sFragment = m_fragment.getText().trim();

            if (sScheme.length() > 0)
            {
                sb.append(sScheme).append(':');
            }

            sb.append(sPart);

            if (sFragment.length() > 0)
            {
                sb.append('#').append(sFragment);
            }

            return new URI(sb.toString());
        }
        catch (URISyntaxException e)
        {
            MgmtConsole.displayMessage(MgmtConsole.MESSAGE_ERROR, "Error while generating URI.", e, true);
        }

        return null;
    }

    private JPanel createButtonPanel()
    {
        JPanel panel = new JPanel();
        panel.setLayout(new PartitionLayout(false, "r,p,p,r", 5, 2, 2, "JPathChooserButonPanel"));

        btnOk = new JButton("OK");
        btnOk.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent evt)
            {
                dialog.setVisible(false);
                returnValue = APPROVE_OPTION;
            }
        });

        btnCancel = new JButton("Cancel");
        btnCancel.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent evt)
            {
                dialog.setVisible(false);
                returnValue = CANCEL_OPTION;
            }
        });

        panel.add(new JPanel());
        panel.add(btnOk);
        panel.add(btnCancel);
        panel.add(new JPanel());

        return panel;
    }

    private JPanel createURIPanel()
    {
        JPanel panel = new JPanel();
        panel.setLayout(new PartitionLayout(true, "p, p", 5, 2, 2, "JURIEditPanel"));

        panel.add(createBasicURIPanel());
        panel.add(createHierarchicalURIPanel());

        return panel;
    }

    private JPanel createBasicURIPanel()
    {
        JPanel panel = new JPanel();
        panel.setLayout(new PartitionLayout(false, "p,p,p,r,p,r", 5, 2, 2, "JURIEditBasicPanel"));

        m_scheme = new JComboBox();
        m_specific = new JTextField();
        m_fragment = new JTextField();

        m_scheme.setEditable(true);
        m_scheme.addItem("file");
        m_scheme.addItem("http");
        m_scheme.addItem("ftp");

        m_scheme.addKeyListener(this);
        m_specific.addKeyListener(this);
        m_fragment.addKeyListener(this);

        panel.add(new JLabel("URI:"));
        panel.add(m_scheme);
        panel.add(new JLabel(":"));
        panel.add(m_specific);
        panel.add(new JLabel("#"));
        panel.add(m_fragment);

        return panel;
    }

    private JPanel createHierarchicalURIPanel()
    {
        JPanel panel = new JPanel();
        panel.setLayout(new PartitionLayout(false, "p,r", 5, 2, 2, "JURIEditHierarchicalPanel"));
        panel.setBorder(BorderFactory.createTitledBorder("Hierarchical"));

        panel.add(new JLabel("Part:"));
        panel.add(createPartPanel());

        panel.add(new JLabel("Authority:"));
        panel.add(createAuthorityPanel());

        return panel;
    }

    private JPanel createPartPanel()
    {
        JPanel panel = new JPanel();
        panel.setLayout(new PartitionLayout(false, "p,r,r,p,r", 5, 0, 0, "JURIEditPartPanel"));

        m_authority = new JTextField();
        m_path = new JTextField();
        m_query = new JTextField();

        panel.add(new JPanel());
        panel.add(new JLabel("Authority"));
        panel.add(new JLabel("Path"));
        panel.add(new JPanel());
        panel.add(new JLabel("Query"));

        panel.add(new JLabel("//"));
        panel.add(m_authority);
        panel.add(m_path);
        panel.add(new JLabel("?"));
        panel.add(m_query);

        return panel;
    }

    private JPanel createAuthorityPanel()
    {
        JPanel panel = new JPanel();
        panel.setLayout(new PartitionLayout(false, "r,p,r,p,r", 5, 0, 0, "JURIEditPartPanel"));

        m_userInfo = new JTextField();
        m_host = new JTextField();
        m_port = new JTextField();

        panel.add(new JLabel("User Info"));
        panel.add(new JPanel());
        panel.add(new JLabel("Host"));
        panel.add(new JPanel());
        panel.add(new JLabel("Port"));

        panel.add(m_userInfo);
        panel.add(new JLabel("@"));
        panel.add(m_host);
        panel.add(new JLabel(":"));
        panel.add(m_port);

        return panel;
    }

    protected void formUpdate()
    {
        btnOk.setEnabled(formChanged);
    }

    //--------------------------------------------------------------------------
    //
    // KeyListener
    //
    //--------------------------------------------------------------------------

    @Override
    public void keyTyped(KeyEvent evt)
    {
        // I think this is a bit dodgy since the VK codes should ONLY be compared
        // to the events KeyCode value....and even that can't really be done in
        // the keyTyped method...but VK_ENTER has the same numeric value as
        // the new line character so...
        //
        if (evt.getKeyChar() == KeyEvent.VK_ENTER)
        {
            return;
        }

        formChanged = true;

        Helper.invoke(new Runnable()
        {
            @Override
            public void run()
            {
                formUpdate();
            }
        });
    }

    @Override
    public void keyPressed(KeyEvent evt)
    {
    }

    @Override
    public void keyReleased(KeyEvent evt)
    {
    }
}