/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui.util;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.net.URI;
import java.net.URISyntaxException;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JTextField;
import javax.swing.UIManager;
import javax.swing.text.Document;

import com.sonicsw.ma.gui.MgmtConsole;

import modelobjects.layout.PartitionLayout;


public class JURLField extends JComponent
{
    public static final int EDIT_SCHEME = 1;
    public static final int EDIT_HOST   = 2;
    public static final int EDIT_PORT   = 4;

    protected JTextField m_scheme = new JInnerTextField(5);
    protected JTextField m_host   = new JInnerTextField();
    protected JTextField m_port   = new JInnerTextField(10);
    protected String     m_text   = null;


    public JURLField(String text, int editMask)
    {
        prepareJURLField(text, editMask);
    }
    
    private void prepareJURLField(String text, int editMask) {
        
        setEditMask(editMask);
        setLayout(new PartitionLayout(false, "p,p,r,p,p", JPartitionPanel.DEFAULT_GAP_SIZE, 0, 0, "JURLField"));

        add(m_scheme);
        add(new JLabel("://"));
        add(m_host);
        if (editMask > EDIT_HOST)
        {
            add(new JLabel(":"));
        }
        add(m_port);

        setText(text);
    }

    @Override
    public boolean isEnabled()
    {
        return super.isEnabled();
    }

    @Override
    public void setEnabled(boolean b)
    {
        super.setEnabled(b);
    }

    public boolean isEditable()
    {
        return (getEditMask() > 0);
    }

    public void setEditable(boolean b)
    {
        setEditMask(0);
    }

    public int getEditMask()
    {
        int res = 0;

        if (m_scheme.isEditable() == true)
        {
            res += EDIT_SCHEME;
        }

        if (m_host.isEditable() == true)
        {
            res += EDIT_HOST;
        }

        if (m_port.isEditable() == true)
        {
            res += EDIT_PORT;
        }

        return res;
    }

    public void setEditMask(int editMask)
    {
        setFieldEditable(m_scheme, ((editMask & EDIT_SCHEME) == EDIT_SCHEME), 6);
        setFieldEditableWithBorder(m_host,   ((editMask & EDIT_HOST)   == EDIT_HOST), 16);
        setFieldEditable(m_port,   ((editMask & EDIT_PORT)   == EDIT_PORT), 10);
    }

    public String getText()
    {
        StringBuffer sb     = new StringBuffer();
        String       scheme = m_scheme.getText().trim();
        String       host   = m_host.getText().trim();
        String       port   = m_port.getText().trim();

        if (scheme.length() > 0 && host.length() == 0 && port.length() == 0)
        {
            return "";
        }
        if (scheme.length() > 0)
        {
            sb.append(scheme).append("://");
        }

        if (host.length() > 0)
        {
            sb.append(host);
        }

        if (port.length() > 0)
        {
            sb.append(":").append(port);
        }

        return sb.toString();
    }

    public void setText(String text)
    {
        m_text = text;

        try
        {
            if(m_port.isEditable())
            {

                URLUtility url = new URLUtility(m_text,m_host.isEditable());

                setSchemeText(url.getBrokerProtocol());
                setHostText(url.getBrokerHostName());

                int port = url.getBrokerPort();

                String rawFragment = null;
                try {
                    URI uri = new URI(m_text);
                    rawFragment = uri.getRawFragment();
                } catch (URISyntaxException e) {
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
                }

                if (port != -1)
                {
                    String portNum = Integer.toString(port);
                    if(rawFragment!=null)
                    {
                        portNum+="#"+rawFragment;
                    }
                    m_port.setText(portNum);
                }
                else
                {
                    if(rawFragment!=null)
                    {
                        m_port.setText("#"+rawFragment);
                    }
                }
            }
            else
            {
                int index = m_text.indexOf("://");
                String scheme = m_text.substring(0, index);
                setSchemeText(scheme);
                setHostText(m_text.substring(index + 3));
            }

        } catch (Exception e) {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
        }
    }

    public void setSchemeText(String text)
    {
        if (text != null)
        {
            m_scheme.setText(text);
            m_scheme.setCaretPosition(0);

            revalidate();
        }
    }

    public void setHostText(String text)
    {
        if (text != null)
        {
            m_host.setText(text);
        }
    }

    @Override
    public void setBackground(Color color)
    {
        m_scheme.setBackground(color);
        m_host.setBackground(color);
        m_port.setBackground(color);
    }

    @Override
    public Color getBackground()
    {
        return m_scheme.getBackground();
    }

    public Document getSchemeDocument()
    {
        return m_scheme.getDocument();
    }

    public Document getHostDocument()
    {
        return m_host.getDocument();
    }

    public Document getPortDocument()
    {
        return m_port.getDocument();
    }

    private void setFieldEditable(JTextField field, boolean editable, int columns)
    {
        field.setEditable(editable);
        field.setBorder(editable ? UIManager.getBorder("TextField.border") : null);
        field.setColumns(editable ? columns : 0);
        field.setFocusable(editable);
    }

    private void setFieldEditableWithBorder(JTextField field, boolean editable, int columns)
    {
        field.setEditable(editable);
//        field.setBorder(editable ? UIManager.getBorder("TextField.border") : null);
        field.setColumns(editable ? columns : 0);
        field.setFocusable(editable);
    }

    //-------------------------------------------------------------------------

    class JInnerTextField extends JTextField
    {
        public JInnerTextField()
        {
            this(0);
        }

        public JInnerTextField(int columns)
        {
            super(columns);
        }

        @Override
        public void setText(String text)
        {
            super.setText(text);

            setMinimumSize(getPreferredSize());
        }

    }

    //-------------------------------------------------------------------------

    static int schemeIndex = 0;
    static String[] SCHEME = { "tcp", "ssl", "http", "https", };

    public static void main(String[] arg)
    {
        final JFrame frame = new JFrame("JURLField Text");
        final JURLField field  = new JURLField("http://localhost", EDIT_PORT);
        JPartitionPanel panel = new JPartitionPanel(true, "p,r", null);
        JButton   scheme = new JButton("Scheme Toggle");
        JButton   get    = new JButton("getText");

        panel.add(field);

        scheme.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent evt)
            {
                schemeIndex = ++schemeIndex % SCHEME.length;
                field.setSchemeText(SCHEME[schemeIndex]);
            }
        });
        get.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent evt)
            {
                JOptionPane.showMessageDialog(frame, field.getText());
            }
        });

        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        frame.getContentPane().add(panel,  BorderLayout.CENTER);
        frame.getContentPane().add(scheme, BorderLayout.NORTH);
        frame.getContentPane().add(get,    BorderLayout.SOUTH);
        frame.setVisible(true);
    }

}
