package com.sonicsw.ma.gui.util;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.event.AWTEventListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;

import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;

/**
 * JWaitCursor
 * <p>Title: JWaitCursor</p>
 * <p>Description: Adds wait cursor to component. </p>
 * @author Jeffrey S. Pace
 * @version 1.0
 */
public class JWaitCursor
{
    protected Cursor m_cursor;
    protected Component m_component;
    protected boolean m_bOwnCursor = false;

    private static GlassOverlay m_glass = new GlassOverlay();

    /**
     * JWaitCursor(Component c)
     * @param c component to add JWaitCursor to.  Consumes all keystrokes and mouse events on c
     * while wait cursor is displayed.
     */
    public JWaitCursor(Component c)
    {
        if (c != null)
        {
            GlassOverlay overLay = createOverLay();
            synchronized (overLay)
            {
                // make sure wait cursor not already displayed...
                if (!c.getCursor().equals(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR)))
                {
                    m_component = c;

                    // can only set glass panes on JDialog or JFrame...
                    if (m_component instanceof JFrame)
                    {
                        ((JFrame) m_component).setGlassPane(overLay);
                    }
                    else if (m_component instanceof JDialog)
                    {
                        ((JDialog) m_component).setGlassPane(overLay);
                    }
                    else
                    {
                        // find parent JDialog or JFrame...
                        JFrame parentFrame = Helper.getJFrame(m_component);
                        if (parentFrame != null)
                        {
                            parentFrame.setGlassPane(overLay);
                        }
                        else
                        {
                            JDialog parentDialog = Helper.getJDialog(m_component);
                            if (parentDialog != null)
                            {
                                parentDialog.setGlassPane(overLay);
                            }
                        }
                    }

                    // set the cursor...
                    overLay.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
                    overLay.setVisible(true);

                    // store off the current cursor...
                    m_cursor = c.getCursor();
                    c.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
                    m_bOwnCursor = true;
                }
            }
        }
    }

    private GlassOverlay createOverLay() {
        return getOverLay();
    }
    
    protected GlassOverlay getOverLay()
    {
        return m_glass;
    }

    /**
     * release()
     * releases the wait cursor.
     */
    public void release()
    {
        if (m_component != null)
        {
            GlassOverlay overLay = getOverLay();
            synchronized (overLay)
            {
                if (m_bOwnCursor)
                {
                    // restore the old cursor, remove the glass...
                    m_component.setCursor(m_cursor);
                    overLay.setCursor(m_cursor);
                    overLay.setVisible(false);
                    m_component = null;
                    m_cursor = null;
                    m_bOwnCursor = false;
                }
            }
        }
    }

    public static class GlassOverlay
        extends JPanel
        implements AWTEventListener
    {
        Window myParentWindow = null;

        public GlassOverlay()
        {
            prepareGlassOverlay();
        }
        private void prepareGlassOverlay() {
            
            setOpaque(false);
            addMouseListener(new MouseAdapter()
            {});
        }

        @Override
        public void setVisible(boolean aVisibleBoolean)
        {
            if (aVisibleBoolean)
            {
                if (this.myParentWindow == null)
                {
                    this.myParentWindow = SwingUtilities.windowForComponent(this);
                }

                Toolkit.getDefaultToolkit().addAWTEventListener(this, AWTEvent.KEY_EVENT_MASK);
            }
            else
            {
                Toolkit.getDefaultToolkit().removeAWTEventListener(this);
            }
            super.setVisible(aVisibleBoolean);
        }

        @Override
        public void eventDispatched(AWTEvent anEvent)
        {
            if ((anEvent instanceof KeyEvent) && (anEvent.getSource()instanceof Component))
            {
                if (SwingUtilities.windowForComponent((Component) anEvent.getSource()) == this.myParentWindow)
                {
                    ((KeyEvent) anEvent).consume();
                }
            }
        }
    }
}