package com.sonicsw.ma.gui.util;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.JPopupMenu;
import javax.swing.JTable;
import javax.swing.JTree;
import javax.swing.JViewport;
import javax.swing.SwingUtilities;

/**
 * PopupMenuShower looks at mouse events to decide when it's time to show
 * a JPopupMenu, and then adjusts the position of the popup if it wasn't
 * completely on screen.
 */
public class PopupMenuShower extends MouseAdapter implements ActionListener
{
    private JTree      m_tree;
    private JTable     m_table;
    private JPopupMenu m_popup;

    public PopupMenuShower(JPopupMenu popup)
    {
        this((JTable)null, popup);
    }

    public PopupMenuShower(JTable table, JPopupMenu popup)
    {
        m_table = table;
        m_popup = popup;

        JViewport vp = (JViewport)table.getParent();

        if (vp != null)
        {
            vp.addMouseListener(this);
        }
    }

    public PopupMenuShower(JTree tree, JPopupMenu popup)
    {
        m_tree  = tree;
        m_popup = popup;
    }

    protected void showMenuIfPopupTrigger(MouseEvent evt)
    {
//        Helper.logDebugMessage("showMenuIfPopupTrigger: " + evt.getComponent());

        if ((m_popup != null) && m_popup.isPopupTrigger(evt))
        {
            // if fewer than two rows are selected in the table or tree,
            // first select the row under the mouse.

            if ((m_table != null) && (m_table.getSelectedRowCount() < 2))
            {
                int row = m_table.rowAtPoint(new Point(evt.getX(), evt.getY()));

                if (row >= 0)
                {
                    m_table.setRowSelectionInterval(row, row);
                }
            }
            else if ((m_tree != null) && (m_tree.getSelectionCount() < 2))
            {
                int row = m_tree.getRowForLocation(evt.getX(), evt.getY());

                if (row >= 0)
                {
                    m_tree.setSelectionRow(row);
                }
            }

            if (m_popup.getComponentCount() > 0)
            {
                Point loc = ensurePopupFullyOnScreen(evt);

                m_popup.show(evt.getComponent(), loc.x, loc.y);
            }
        }
    }

    protected Point ensurePopupFullyOnScreen(MouseEvent evt)
    {
        Component comp = evt.getComponent();
        Dimension ssz  = comp.getToolkit().getScreenSize();
        Dimension psz  = m_popup.getSize();
        Point     loc  = new Point(evt.getX(), evt.getY());

        SwingUtilities.convertPointToScreen(loc, comp);

        if ((loc.y + psz.height) > ssz.height)
        {
            loc.y -= psz.height;
        }

        if ((loc.x + psz.width) > ssz.width)
        {
            loc.x -= psz.width;
        }

        SwingUtilities.convertPointFromScreen(loc, comp);

        return loc;
    }

    @Override
    public void mousePressed(MouseEvent evt)
    {
        showMenuIfPopupTrigger(evt);
    }

    @Override
    public void mouseReleased(MouseEvent evt)
    {
        showMenuIfPopupTrigger(evt);
    }

    @Override
    public void actionPerformed(ActionEvent evt)
    {
        Component comp = (Component)evt.getSource();

        MouseEvent mouseEvent = new MouseEvent(comp, 0, 0, 0, 0, comp.getHeight() - 1, 0, true);

        if (m_popup.getComponentCount() > 0)
        {
            Point loc = ensurePopupFullyOnScreen(mouseEvent);

            m_popup.show(comp, loc.x, loc.y);
        }
    }

    public void setPopupMenu(JPopupMenu menu)
    {
        m_popup = menu;
    }

    public JPopupMenu getPopupMenu()
    {
        return m_popup;
    }

    public JTree getTree()
    {
        return m_tree;
    }

    public JTable getTable()
    {
        return m_table;
    }
}
