/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corporation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.ma.gui.util;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.text.Collator;
import java.util.Collections;
import java.util.Comparator;
import java.util.Vector;

import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.DefaultListModel;
import javax.swing.Icon;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.ListCellRenderer;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.PreferenceManager;
import com.sonicsw.mx.config.ConfigFactory;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigPath;


/**
 * Dialog box with a list of users to let the user pick a principal.
 */
public class PricipalChooserDialog extends JMADialog
                                              implements ListSelectionListener
{
    // moved here to remove dependancy on mq
    public static final IConfigPath USER_NAME        = ConfigFactory.createConfigPath("USER_NAME");
    public static final IConfigPath GROUP_NAME       = ConfigFactory.createConfigPath("GROUP_NAME");

    private JList     m_aclList;
    private JCheckBox m_showUsers;
    private JCheckBox m_showSystem;

    private Object[]  m_userNames;
    private Object[]  m_groupNames;
    private Object    m_selectedItem;
    private Action    m_okAction;
    private boolean   m_routingPrincipals = false;

    public final static String GROUP_PREFIX = "#^_";
    public final static String USER_PREFIX  = "^@_";

    // Note: Don't initialize the panel and component instance variables in this
    // class as that will reset the actual object references created when the base
    // class calls this class' initPanels() and initViewAdapters() methods, during
    // construction of this class.
    public PricipalChooserDialog(JMAFrame parent, Object[] users, Object[] groups, boolean routingPrincipals)
    {
        super(parent, "policy-acl-list");

        m_groupNames = getUserGroupNames(groups, false);
        m_userNames  = getUserGroupNames(users, true);
        m_routingPrincipals = routingPrincipals;

        myInit();
        setModalProperty();
    }
    
    private void setModalProperty() { 
        setModal(true);
    }


    private Object[] getUserGroupNames(Object[] objects, boolean bForUser)
    {
        // Fetch users or groups under the domain specified for the policy
        Vector vUserName = new Vector();

        try
        {
            String pefix = bForUser ? USER_PREFIX : GROUP_PREFIX;
            IConfigPath attributeName = bForUser ? USER_NAME : GROUP_NAME;

            for (int i=0; i< objects.length; i++)
            {
                IConfigBean bean = (IConfigBean)objects[i];
                String name = (String)bean.getAttribute(attributeName);
                vUserName.addElement(pefix + name);
            }
        }
        catch (Exception e)
        {
            // This exception can be none-error if there is no user at all.
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
        }

        return vUserName.toArray();  // if no user, it will return an array of object[0]
    }

    /**
     * userName and groupName have different prefix so that ListCellrenderer can tell user apart from group and use
     * appropreicate icon for each. Here we have to get rid of the prefix.
     * @param name
     */
    private static String getRidPrefix(String name)
    {
        return name.substring(GROUP_PREFIX.length());
    }

    private static String getRidPrefix(Object obj)
    {
        if (obj == null)
        {
            return "";
        }

        return getRidPrefix(obj.toString());
    }

    /**
     * abstract mehtods we have to provide implementation
     */
    @Override
    protected void maInitialize() { }

    /**
     * abstract method we have to provide implementation
     */
    @Override
    protected void maCleanup()
    {
        PreferenceManager pm = PreferenceManager.getInstance();
        pm.setBoolean("view", "userPrincipals", m_showUsers.isSelected(), false);
        pm.setBoolean("view", "systemPrincipals", m_showSystem.isSelected(), false);
    }

    /**
     *  The default initializer internalInitialize() is called from super class' constructor
     *  and it is before this class has the chance to set the m_principals. So we have to do some
     *  init work here.
     */
    private void myInit()
    {
        m_showUsers  = new JCheckBox("Include Users",          PreferenceManager.getInstance().getBoolean("view", "userPrincipals", false));
        m_showSystem = new JCheckBox("Show System Principals", PreferenceManager.getInstance().getBoolean("view", "systemPrincipals", false));
        m_aclList    = new JList();
        m_aclList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

        updateList();

        JPanel panel = new JPanel(new BorderLayout());
        panel.setBorder(BorderFactory.createTitledBorder(""));

        ExtendedJScrollPane scroll = new ExtendedJScrollPane(m_aclList);
        panel.add(scroll, BorderLayout.CENTER);

        JPartitionPanel checkBoxPanel = new JPartitionPanel(true, "p,p", null);
        checkBoxPanel.add(m_showUsers);
        checkBoxPanel.add(m_showSystem);
        panel.add(checkBoxPanel, BorderLayout.SOUTH);

        getContentPane().add(BorderLayout.CENTER, panel);

        m_aclList.setCellRenderer(new ACLCellRenderer());

        m_aclList.addListSelectionListener(this);

        FilterActionListener fal = new FilterActionListener();
        m_showUsers.addActionListener(fal);
        m_showSystem.addActionListener(fal);

        m_aclList.addMouseListener(new MouseAdapter()
        {
            @Override
            public void mouseClicked(MouseEvent evt)
            {
                if (evt.getClickCount() == 2)
                {
                    ((JButtonPanel)getButtonPanel()).getButton((String)m_okAction.getValue(Action.ACTION_COMMAND_KEY)).doClick();
                }
            }
        });
    }

    /**
     * @return the item user selected from the list of users
     */
    public String getSelectedItem()
    {
        return getRidPrefix((String)m_selectedItem);
    }

    // return the selection with the prefix so it can be identified.

    public String getSelectedItemWithPrefix()
    {
        return (String)m_selectedItem;
    }

    private void updateList()
    {
        DefaultListModel model = new DefaultListModel();

        if (!m_routingPrincipals)
        {
            for (int i = 0; i < m_groupNames.length; i++)
            {
                if (m_showSystem.isSelected())
                {
                    model.addElement(m_groupNames[i]);
                }
                else
                {
                    String name = getRidPrefix((String)m_groupNames[i]);

                    if (!name.startsWith("$"))
                    {
                        model.addElement(m_groupNames[i]);
                    }
                }
            }
        }

        if (m_showUsers.isSelected() == true)
        {
            for(int i = 0; i < m_userNames.length; i++)
            {
                if (m_showSystem.isSelected())
                {
                    model.addElement(m_userNames[i]);
                }
                else
                {
                    String name = getRidPrefix((String)m_userNames[i]);

                    if (!name.startsWith("$"))
                    {
                        model.addElement(m_userNames[i]);
                    }
                }
            }
        }

        m_aclList.setModel(sort(model));
    }

    private DefaultListModel sort(DefaultListModel model)
    {
        // We need a List (a Vector) for sorting
        Vector list = new Vector();
        for (int x = 0; x < model.getSize(); ++x)
        {
            list.addElement(model.get(x));
        }

        // sort the List
        Collections.sort(list, new PrincipalComparator());

        // update the model with a sorted List
        for (int x = 0; x < model.getSize(); x++)
        {
            if (model.getElementAt(x) != list.elementAt(x))
            {
                model.set(x, list.elementAt(x));
            }
        }

        return model;
    }

    //-------------------------------------------------------------------------
    //
    // ItemListener implementation
    //
    //-------------------------------------------------------------------------

    @Override
    public void valueChanged(ListSelectionEvent evt)
    {
        if (evt.getSource() == m_aclList)
        {
            m_okAction.setEnabled(true);
        }
    }

    //-------------------------------------------------------------------------

    @Override
    public Action[] getButtonActions()
    {
        m_okAction = new OKAction(getDefaultOKAction());
        m_okAction.setEnabled(false);

        return new Action[] { m_okAction, getDefaultCancelAction() };
    }

    /**
     * We piggyback the default comparator for a collator in order to strip
     * off the principal prefix so that we get a 'good' sort.
     */
    class PrincipalComparator implements Comparator
    {
        @Override
        public int compare(Object o1, Object o2)
        {
            return Collator.getInstance().compare(getRidPrefix(o1), getRidPrefix(o2));
        }

        @Override
        public boolean equals(Object obj)
        {
            return Collator.getInstance().equals(obj);
        }
        
        @Override
        public int hashCode() {
            return Collator.getInstance().hashCode();
        }
    }

    /**
     * Inner class to provide our own rendering of cells in ACL list.
     *
     */
    class ACLCellRenderer extends JLabel implements ListCellRenderer
    {
        public ACLCellRenderer()
        {
            setOpaqueProperty();
        }
        
        private void setOpaqueProperty() {
            setOpaque(true);
        }

        @Override
        public Component getListCellRendererComponent (JList list,
                                                       Object value,            // value to display
                                                       int index,               // cell index
                                                       boolean isSelected,      // is the cell selected
                                                       boolean cellHasFocus)    // the list and the cell have the focus
        {
            String text = value.toString();

            String iconName;

            if(text.startsWith(GROUP_PREFIX))
            {
                iconName = "MF_AUTHENTICATION_GROUP";
            }
            else
            {
                iconName = "MF_AUTHENTICATION_USER";
            }

            Icon icon = ResourceManager.getTypeIcon(getClass(), iconName);
            setIcon(icon);

            if (isSelected)
            {
                setBackground(list.getSelectionBackground());
                setForeground(list.getSelectionForeground());
            }
            else
            {
                setBackground(list.getBackground());
                setForeground(list.getForeground());
            }

            // get rid of the prefix part
            text = getRidPrefix(text);
            setText(text);

            return this;
        }
    }

    /**
     * Inner class to handle the Include user check box user action.
     */
    class FilterActionListener implements ActionListener
    {
        @Override
        public void actionPerformed(ActionEvent evt)
        {
            updateList();

            // selection on the list is gone, disable the OK button
            m_okAction.setEnabled(false);
        }
    }

    /**
     * Inner class for handling OK button action.
     *
     */
    class OKAction extends BasicGuiAction
    {
        Action m_target;

        public OKAction(Action target)
        {
            super("AuthorizationPolicyACLListDialog.ok");

            m_target = target;
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            int sel = m_aclList.getSelectedIndex();

            if (sel >= 0 )
            {
                m_selectedItem = m_aclList.getSelectedValue();

                m_target.actionPerformed(evt);
            }
        }
    }

}
