package com.sonicsw.ma.gui.util;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;

import modelobjects.framework.EditRule;
import modelobjects.framework.ModelEditMediator;
import modelobjects.framework.UndoableModelAspectEdit;
import modelobjects.framework.ViewValueConversionException;
import modelobjects.framework.model.ModelAspectId;
import modelobjects.framework.model.ModelObjectAdapter;
import modelobjects.util.ObjectCopier;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.table.IModelTableModel;
import com.sonicsw.ma.gui.table.JRowTableAspectAdapter;
import com.sonicsw.ma.plugin.ConfigBeanModel;
import com.sonicsw.mx.config.ConfigFactory;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IConfigPath;
import com.sonicsw.mx.config.util.ConfigHelper;

/**
 *  JTableMapAspectAdapter is a special kind of ViewAspectAdapter that
 *  is used for Map-valued model-aspects that are edited in JTables.
 *  This class should be used only with care, since it must make a complete
 *  copy of the model-aspect Map and of the contents of the Map in order
 *  to be sure that edits made in the table (inserts, deletes, cell-edits)
 *  are both discardable in the case of reset, and undoable and redoable.
 *
 *  Note that JTableMapAspectAdapter can only be used with JTables that
 *  use ModelListTableModel for their TableModels.
 *
 *  JTableMapAspectAdapter supports PropertyChangeListeners that may want
 *  to track when it is editable in order to enable/disable editing actions.
 */
public class PrimitiveMapTableAspectAdapter extends    JRowTableAspectAdapter
                                            implements ObjectCopier
{
    /**
     *  Construct a JTableAttributeMapAspectAdapter.
     *
     *  @param modelAspectId the ModelAspectId of the model-aspect being edited
     *  @param table the JRowTable component in which the editing occurs
     *  @param rowModelCopier the ObjectCopier used to make copies of row models
     *  @param editRule the editRule for the ViewAspectAdapter
     *  @param modelEditMediator the ModelEditMediator managing this adapter
     *  @param type the config type (AttributeDescrptionMap) for each row
     */
    public PrimitiveMapTableAspectAdapter(ModelAspectId     modelAspectId,
                                          JRowTable         table,
                                          EditRule          editRule,
                                          ModelEditMediator modelEditMediator)
    {
        super(modelAspectId, table, null, editRule, modelEditMediator);

        rowModelCopier = this;
    }

    @Override
    protected void setViewAspectValue(Object viewAspectValue)
    throws IllegalArgumentException
    {
        if (viewAspectValue == null)
        {
            return;
        }

        if (!(viewAspectValue instanceof IAttributeMap))
        {
            throw new IllegalArgumentException("The view aspect value must be of type IAttributeMap!");
        }

        try
        {
            IAttributeMap map      = (IAttributeMap)((IAttributeMap)viewAspectValue).clone();
            ArrayList     contents = new ArrayList(map.size());
            Iterator      i        = map.keySet().iterator();

            while (i.hasNext())
            {
                String key   = (String)i.next();
                Object value = map.get(key);

                if (value != null)
                {
                    contents.add(new PrimitiveModel(key, value));
                }
            }

            super.setViewAspectValue(contents);
        }
        catch (Exception e)
        {
            MgmtConsole.displayMessage(MgmtConsole.MESSAGE_ERROR, e.toString(), e, true);
        }
    }

    // the following is provided so subclasses can access the method on JTableListAspectAdapter

    protected void setViewAspectValue(List value)
    {
        super.setViewAspectValue(value);
    }

    @Override
    public Object getViewAspectValue()
    {
        IAttributeMap map = null;

        try
        {
            List     list = ((IModelTableModel)table.getModel()).getContents();
            Iterator i    = list.iterator();

            map = getParentModel();

            while (i.hasNext())
            {
                PrimitiveModel rowModel = (PrimitiveModel)i.next();
                String         rowName  = (String)rowModel.getKey();

                map.setAttribute(rowName, rowModel.getValue());
            }
        }
        catch (ConfigServiceException e)
        {
            MgmtConsole.displayMessage(MgmtConsole.MESSAGE_ERROR, e.toString(), e, true);
        }

        return map;
    }

    /**
     * We do nothing here since the attribute map was already (deep) cloned
     * on the set.
     */
    @Override
    public Object copyObject(Object obj)
    {
        return obj;
    }

    public static Object createRowModelInstance(String key, Object value)
    {
        return new PrimitiveModel(key, value);
    }

    /**
     * Return an UndoableModelAspectEdit representing the edits made to the
     * IAttributeMap model aspect value since the last time setModelAspectValue
     * was called.
     *
     * We're over-riding this method here because the default implementation
     * in ViewAspectAdapter does a reference comparision between the new and
     * old model aspect values...and in this case the references to the
     * parent IAttributeMap's will be the same....so this method ommits the
     * reference equality check.
     */
    @Override
    public UndoableModelAspectEdit makeUndoableModelAspectEdit(ModelObjectAdapter modelObjectAdapter)
        throws ViewValueConversionException
    {
        return new UndoableModelAspectEdit("changed " + getModelAspectId(),
                                           modelObjectAdapter,
                                           getModelAspectId(),
                                           getOldModelAspectValue(),
                                           getModelAspectValue());
    }

    protected IAttributeMap getParentModel()
    {
        IConfigPath     path  = ConfigFactory.createConfigPath(getModelAspectId().getName());
        ConfigBeanModel model = (ConfigBeanModel)getModelEditMediator().getModel().getModelObject();
        IAttributeMap   map   = model.getData();
        IAttributeMap   parent = (IAttributeMap)map.getAttribute(path);

        if (parent == null)
        {
            parent = (IAttributeMap)ConfigHelper.createChild(map, path);
        }
        else
        {
            parent.clear();
        }

        return parent;
    }

    //-------------------------------------------------------------------------

    public static class PrimitiveModel
    {
        protected String m_key;
        protected Object m_value;

        public PrimitiveModel(String key, Object value)
        {
            m_key   = key;
            m_value = value;
        }

        public String getKey()
        {
            return m_key;
        }

        public Object getValue()
        {
            return m_value;
        }

        @Override
        public boolean equals(Object value)
        {
            if (!(value instanceof PrimitiveModel))
            {
                return false;
            }

            return getKey().equals(((PrimitiveModel)value).getKey());
        }
        
        @Override
        public int hashCode() {
            return Objects.hashCode(getKey());
        }

    }

}
