package com.sonicsw.ma.gui.util;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.Properties;
import java.util.ResourceBundle;

import javax.swing.Icon;
import javax.swing.ImageIcon;

import com.sonicsw.ma.gui.MgmtConsole;

import progress.message.zclient.Version;


/**
 * <p>Title: ResourceManager</p>
 * <p>Description: Provides static methods that expose typical ResourceBundle
 * functionality where the base name is calculated from the supplied class.
 * The methods always attempt to unwind the class - recursive calls to
 * Class.getSuperclass() - allows for a rudimentary form of resource bundle
 * chaining via the class hierarchy.</p>
 * <p>Copyright: Copyright (c) 2002</p>
 * <p>Company: Sonic Software</p>
 * @author Huw Jones
 * @version 1.0
 */
public class ResourceManager
{
    private static final String APPLICATION_LOGO = "application.logo";
	private static final String APPLICATION_ICON_36 = "application36.icon";
    private static final String APPLICATION_ICON_18 = "application18.icon";
    private static final String BRANDING_PROPERTIES = "branding.properties";
    public static final String SONICSW_HOME_PROPERTY = "sonicsw.home";
    private static final String RES_BASE = "com.sonicsw.";
    private static final String RES_NAME = ".resrc";

    public static final char SEPARATOR = '/';

    public static final String KEY_CONFIG = "config";
    public static final String KEY_RUNTIME = "runtime";
    public static final String KEY_LABEL = "label";
    public static final String KEY_UNIT = "unit";
    public static final String KEY_LIST = "list";

    private static Class m_classThis = ResourceManager.class;
    private static volatile Properties applicationProperties;

	private static Map<String, Icon> APPLICATION_LOGOS_MAP = new HashMap<String, Icon>();

    private static Object LOCK_OBJ = new Object();
    public static Object getObject(Class resClass, Object[] key)
    {
        return getObject(resClass, buildKey(key));
    }

    public static Object getObject(Class resClass, Object key)
    {
        Object res = null;
        String[] baseName = getBaseNames(resClass);
        int i = 0;
        Exception exception = null;

        while ((res == null) && (i < baseName.length))
        {
            try
            {
                ResourceBundle bundle = ResourceBundle.getBundle(baseName[i] +
                    RES_NAME, Locale.getDefault(), resClass.getClassLoader());

                res = bundle.getObject((key != null) ? key.toString() : null);
            }
            catch (MissingResourceException e)
            {
                if (exception == null)
                {
                    exception = e;
                }
            }

            i++;
        }

        if ((res == null) && (exception != null))
        {
            debugErrorOutput(exception.getMessage());
        }

        return res;
    }

    public static String getString(Class resClass, Object[] key)
    {
        return getString(resClass, buildKey(key));
    }

    public static String getString(Class resClass, Object key)
    {
        String res = null;
        String[] baseName = getBaseNames(resClass);
        int i = 0;
        Exception exception = null;

        while ((res == null) && (i < baseName.length))
        {
            try
            {
                ResourceBundle bundle = ResourceBundle.getBundle(baseName[i] +
                    RES_NAME, Locale.getDefault(), resClass.getClassLoader());

                res = bundle.getString((key != null) ? key.toString() : null);
            }
            catch (MissingResourceException e)
            {
                if (exception == null)
                {
                    exception = e;
                }
            }

            i++;
        }

        if ((res == null) && (exception != null))
        {
            debugErrorOutput(exception.getMessage());
        }

        return res;
    }

    public static String[] getStringArray(Class resClass, Object[] key)
    {
        return getStringArray(resClass, buildKey(key));
    }

    public static String[] getStringArray(Class resClass, String key)
    {
        String[] res = new String[0];
        String[] baseName = getBaseNames(resClass);
        int i = 0;
        Exception exception = null;

        while ((res == null) && (i < baseName.length))
        {
            try
            {
                ResourceBundle bundle = ResourceBundle.getBundle(baseName[i] +
                    RES_NAME, Locale.getDefault(), resClass.getClassLoader());

                res = bundle.getStringArray(key);
            }
            catch (MissingResourceException e)
            {
                if (exception == null)
                {
                    exception = e;
                }
            }

            i++;
        }

        if ((res == null) && (exception != null))
        {
            debugErrorOutput(exception.getMessage());
        }

        return res;
    }

    public static Object getMessage(Class resClass, Object[] key, Object[] args)
    {
        return getMessage(resClass, buildKey(key), args);
    }

    public static String getMessage(Class resClass, Object key, Object[] args)
    {
        String res = (String)getObject(resClass, key);

        return (res != null) ? new MessageFormat(res).format(args) : null;
    }

    public static Object getIcon(Class resClass, Object[] key)
    {
        return getIcon(resClass, buildKey(key));
    }

    public static Icon getIcon(Class resClass, Object key)
    {
        return getIcon(resClass, key, getBaseNames(resClass), "image");
    }

    public static Icon getTypeIcon(Object key)
    {
        return getTypeIcon(m_classThis, key);
    }

    public static Icon getTypeIcon(Class resClass, Object key)
    {
        Icon icon = getIcon(resClass, key, new String[]
                            {"com.sonicsw.ma.gui"}
                            ,
                            "image" + SEPARATOR + "type");

        // If the icon was not found, force the "Unknown" icon to be used.  This
        // will typically never fail, the unknown icon will always be present.
        if (icon == null)
        {
            icon = getIcon(resClass, null, new String[]
                           {"com.sonicsw.ma.gui"}
                           ,
                           "image" + SEPARATOR + "type");
        }

        return (icon);
    }

    protected static Icon getIcon(Object key, String[] pathBase,
                                  String pathPart)
    {
        return getIcon(m_classThis, key, pathBase, pathPart);
    }

    protected static Icon getIcon(Class resClass, Object key, String[] pathBase,
                                  String pathPart)
    {
        Icon res = null;
        int i = 0;

        while ((res == null) && (i < pathBase.length))
        {
            String keyValue = (key != null) ? key.toString() : "Unknown";

                StringBuffer sbImage = new StringBuffer();

                sbImage.append(pathBase[i].replace('.', SEPARATOR));
                sbImage.append(SEPARATOR);
                sbImage.append(pathPart);
                sbImage.append(SEPARATOR);
                sbImage.append(keyValue);
                sbImage.append(".gif");

                URL imgUrl = resClass.getClassLoader().getResource(sbImage.
                    toString());

                if (imgUrl != null) {
                	res = new ImageIcon(imgUrl);
                	break;
                }
                i++;
            }

        if ((res == null))
        {
			String message = "Resource " + key + ".gif could not be found in " + Arrays.toString(pathBase);
			Helper.logDebugMessage(message);
        }

        return res;
    }

    public static Object[] prefixArray(Object[] list, Object item)
    {
        Object[] newList = new Object[list.length + 1];

        newList[0] = item;

        for (int i = 0; i < list.length; i++)
        {
            newList[i + 1] = list[i];
        }

        return newList;
    }

    private static void debugErrorOutput(String value)
    {
        if (MgmtConsole.DEVELOPER_MODE)
        {
            Helper.logDebugMessage(value);
        }
    }

    //-------------------------------------------------------------------------

    protected static String buildKey(Object[] key)
    {
        StringBuffer res = new StringBuffer();

        for (int i = 0; i < key.length; i++)
        {
            if (key[i] == null)
            {
                continue;
            }

            String keyValue = key[i].toString().trim();

            if ((keyValue != null) && (keyValue.length() > 0))
            {
                if ((res.length() > 0) && (res.charAt(res.length() - 1) != '.'))
                {
                    res.append('.');
                }

                res.append(keyValue);
            }
        }

        return res.toString();
    }

    protected static String getBaseName(Class resClass)
    {
        String name = resClass.getPackage().getName();
        String base = name;

        if (name.startsWith(RES_BASE))
        {
            base = name.substring(0, name.indexOf('.', RES_BASE.length() + 1));
        }

        return base + ".gui";
    }

    protected static String[] getBaseNames(Class resClass)
    {
        ArrayList name = new ArrayList();
        Class aClass = resClass;
        Class lastClass = null;

        while (aClass != null)
        {
            String aName = getBaseName(aClass);

            if (!name.contains(aName))
            {
                name.add(aName);
            }

            lastClass = aClass;
            aClass = aClass.getSuperclass();

            // Catch the situation where we transition from a sonic class
            // to a system class...we don't want to load resources for
            // system classes...
            //
            if (lastClass.getName().startsWith(RES_BASE) &&
                (aClass != null) && !aClass.getName().startsWith(RES_BASE))
            {
                aClass = null;
            }
        }

        return (String[])name.toArray(new String[name.size()]);
    }

    public static Properties getApplicationProperties() {
        if( applicationProperties != null)
        {
            return applicationProperties ;
        }
        synchronized (LOCK_OBJ) {
        	if( applicationProperties == null) {
	            applicationProperties = new Properties();
	            try {
	                String propsPath = getMQBinFolderPath() + BRANDING_PROPERTIES;
	                File file = new File(propsPath);
	                if(!file.exists())
	                {
	                    return null;
	                }
	                URL propsURL = file.toURL();
	                applicationProperties.load(propsURL.openStream());
	                return applicationProperties;
	            } catch (MalformedURLException e) {
	                e.printStackTrace();
	                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
	            } catch (IOException e) {
	                e.printStackTrace();
	                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
	            }
	            return null;
        	} else {
        		return applicationProperties ;
        	}
		}
    }

    public static String getApplicationPropertyString(String key){
        Properties applicationProperties = getApplicationProperties();
        if(applicationProperties == null)
        {
            return null;
        }
        return applicationProperties.getProperty(key);
    }

    public final static Icon getApplicationIcon(Class className, String key, int size){
	 	if(size == 18)
        {
            return getIconOfSize(className, key, APPLICATION_ICON_18);
        }
        else if(size == 36)
        {
            return getIconOfSize(className, key, APPLICATION_ICON_36);
        }
	 	return null;
	}

	private static Icon getIconOfSize(Class className, String key, String brandingIconKey) {
		String imgUrl = getMQBinFolderPath() + getApplicationPropertyString(brandingIconKey);
		return getOrCreateIcon(imgUrl, className, key);
	}
	
	public static Icon getApplicationLogo(Class className, String key){
	    String imgUrl = getMQBinFolderPath() + getApplicationPropertyString(APPLICATION_LOGO);
        return getOrCreateIcon(imgUrl, className, key);
    }
	
	private static Icon getOrCreateIcon(String imgUrl, Class className, String key)
	{
	    if (APPLICATION_LOGOS_MAP.containsKey(imgUrl))
        {
            return APPLICATION_LOGOS_MAP.get(imgUrl);
        }
        else if (APPLICATION_LOGOS_MAP.containsKey(className+key))
        {
            return APPLICATION_LOGOS_MAP.get(className+key);
        }
        return createIconImage(imgUrl, className, key);
	}

    private static Icon createIconImage(String imgUrl, Class className, String key) {
    	Icon icon;
        if( new File(imgUrl).exists())
        {
            icon = new ImageIcon(imgUrl);
            APPLICATION_LOGOS_MAP.put(imgUrl, icon);
        }
        else
        {
            icon = getIcon(className, key);
            APPLICATION_LOGOS_MAP.put(className+key, icon);
        }
        return icon;
    }

    public static final String getMQFolderPath(){
        String SONICSW_HOME = System.getProperty(SONICSW_HOME_PROPERTY);
        return SONICSW_HOME + File.separator + "MQ" + getMajorMinorVersion() + File.separator;
    }

    private static String getMajorMinorVersion() {
        return Version.getMajorVersion() + "." + Version.getMinorVersion();
    }

    public static final String getMQBinFolderPath(){
        return getMQFolderPath() + "bin" + File.separator;
    }
}