/**
 * Copyright (c) 2003 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.mgmtapi.config.impl;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.sonicsw.ma.mgmtapi.config.IMgmtBeanBase;
import com.sonicsw.ma.mgmtapi.config.MgmtBeanFactory;
import com.sonicsw.ma.mgmtapi.config.MgmtException;
import com.sonicsw.mx.config.ConfigFactory;
import com.sonicsw.mx.config.ConfigServerUtility;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigElement;
import com.sonicsw.mx.config.IConfigServer;
import com.sonicsw.mx.config.TxnConfigServerUtility;

import com.sonicsw.mf.common.IDirectoryFileSystemService;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.query.FromElementType;
import com.sonicsw.mf.common.config.query.Query;

public abstract class AbstractMgmtBeanFactory
{
    private HashMap                m_registered = new HashMap();
    private TxnConfigServerUtility m_csu        = new TxnConfigServerUtility();

    //-------------------------------------------------------------------------
    //
    // Connection/Disonnection
    //
    //-------------------------------------------------------------------------

    /**
     * Connect to the domain using the specfied connection parameters.
     * @param domain
     * @param url
     * @param user
     * @param password
     * @throws MgmtException
     */
    public void connect(String domain, String url, String user, String password)
    throws MgmtException
    {
        try
        {
            m_csu.connect(domain, url, user, password, true);
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to connect to domain : " + e.getMessage(), e);
        }
    }

    /**
     * Connect to the domain using the specified directory service interface
     * @param dsProxy
     * @throws MgmtException
     */
    public void connect(IDirectoryFileSystemService dsProxy)
    throws MgmtException
    {
        try
        {
            m_csu.connect(dsProxy, true);
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to connect to domain : " + e.getMessage(), e);
        }
    }

    public void connect(TxnConfigServerUtility csu)
    {
        m_csu =csu;
    }
    
    
    
    /**
     * Disconnect from the domain
     * @throws ConfigServiceException
     */
    public void disconnect() throws ConfigServiceException
    {
        m_csu.disconnect();
    }

    /**
     * Get the domain name for the underlying connection
     * @return The domain name
     */
    public String getDomain()
    {
        return m_csu.getDomain();
    }

    /**
     * This method fetches the Config Server reference from the wrapped
     * Config Server utility.
     *
     * @return  the server reference.
     */
    public IConfigServer getConfigServer()
    {
        IConfigServer ret = m_csu.getConfigServer();

        if(ret == null)
        {
            throw new RuntimeException("ConfigServer is not connected");
        }

        return ret;
    }

    /**
     * Get the ConfigServerUtility which is a wrapper around the Config Server
     * to provide additional functionality such as import / export.
     *
     * @return The ConfiguServerUtility
     */
    public ConfigServerUtility getConfigServerUtility()
    {
        return m_csu;
    }

    //-------------------------------------------------------------------------

    /**
     * Commit all changes that have been made to the beans and saved. This takes
     * these changes and sends them to the DS.
     * @throws MgmtException
     */
    public void commit()
    throws MgmtException
    {
        try
        {
            getConfigServer().commit();
        }
        catch (Throwable e)
        {
            throw new MgmtException("Failed to commit changes - " + e.getMessage(), e);
        }
    }

    /**
     * Rollback changes made to the underlying config beans. All modified beans
     * are reverted to their unmodified state.
     * @throws MgmtException
     */
    public void rollback()
    throws MgmtException
    {
        try
        {
            getConfigServer().rollback();
        }
        catch (Throwable e)
        {
            throw new MgmtException("Failed to rollback changes - " + e.getMessage(), e);
        }
    }

    /**
     * Flush the config server of all modified and unmodified beans. This empties
     * the config server and frees up any memory used by loaded config beans.
     * Any references to beans are invalid following a flush - you must re-get
     * your beans from the DS.
     * @throws MgmtException
     */
    public void flush()
    throws MgmtException
    {
        try
        {
            getConfigServer().flush();
        }
        catch (Throwable e)
        {
            throw new MgmtException("Failed to flush changes - " + e.getMessage(), e);
        }
    }

    /**
     * Get a list of all the folders under the specified parent folder.
     *
     * @param parent          The parent folder under all available child
     *                        folders will be returned.
     * @return                A list of folders (java.lang.String)
     * @throws MgmtException  If a failure occurs
     */
    public List getFolderNames(String parent)
    throws MgmtException
    {
        try
        {
            return new ArrayList(getConfigServer().listFolders(parent));
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to getFolderNames - " + e.getMessage(), e);
        }
    }

    /**
     * Get a list of all the configurations elements under the specified parent folder.
     *
     * @param parent          The parent folder under which to find all
     *                        available child configurations.
     * @return                A list of folders (java.lang.String)
     * @throws MgmtException  When an error occurs building the list of
     *                        configurations.
     */
    public List getConfigurationNames(String parentFolder)
    throws MgmtException
    {
        try
        {
            Set names = getConfigServer().listConfigElements(parentFolder);

            ArrayList list = new ArrayList(0);

            for(Iterator i = names.iterator(); i.hasNext(); )
            {
               String name = (String)i.next();
               list.add(name.substring(name.lastIndexOf("/") + 1));
            }
            return list;
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to getConfigurationNames - " + e.getMessage(), e);
        }
    }

    /**
     * Generates a java.lang.String array of all the folders under the
     * specified parent folder.
     *
     * @param parent          The parent folder under all available child
     *                        folders will be found.
     * @return                A list of folders (java.lang.String). This method
     *                        will never return 'null' - instead a zero-length
     *                        array is used to simplify calling code.
     * @throws MgmtException  If a failure occurs
     */
    public String[] getFolders(String parentFolder)
    throws MgmtException
    {
        try
        {
            Set dirs = getConfigServer().listFolders(parentFolder);
            return (String[])dirs.toArray(new String[0]);
        }
        catch (Throwable e)
        {
            throw new MgmtException("Failed to get folders" + e.getMessage(), e);
        }
    }

    /**
     * Check if a specified path exists.
     * @param path
     * @return
     */
    public boolean pathExists(String path)
    {
        return getConfigServer().pathExists(path);
    }

    /**
     * List all configurations of a specified type e.g. MQ_BROKER.
     * @param type The type e.g. MQ_BROKER
     * @return A list of names of configations
     * @throws MgmtException
     */
    public List listConfigElements(String type)
    throws MgmtException
    {
        try
        {
            Query query = new Query().setFrom(new FromElementType(type));
            Set elements = getConfigServer().listConfigElements(query);

            ArrayList names    = new ArrayList(elements.size());
            if(!elements.isEmpty()){
                Iterator i = elements.iterator();
                String name = "";
                while (i.hasNext()) {
                    name = (String)i.next();
                    if(name.endsWith(ConfigServerUtility.DEFAULT_SUFFIX) ||
                            name.endsWith(ConfigServerUtility.DOMAIN_DESCRIPTOR_SUFFIX) ||
                            name.endsWith(ConfigServerUtility.POLICY_DESCRIPTOR_SUFFIX)){
                        name = name.substring(0, name.lastIndexOf("/"));
                    }
                    names.add(name);
                }
            }
            return names;
        }
        catch (Throwable ce){
            throw new MgmtException("Can't get config elements. Type: " + type, ce);
        }
    }

    /**
     * Lists all the folders and elements names under the specified path.
     */
    public List list(String path)
    throws MgmtException
    {
        try
        {
			Set names = getConfigServer().list(path);
			ArrayList list = new ArrayList(0);
			for (Iterator i = names.iterator(); i.hasNext();) {
				HashMap entry = (HashMap) i.next();
				
				String name = null;
				if (entry.containsKey(IConfigServer.FOLDER_NAME)) {
					name = (String) entry.get(IConfigServer.FOLDER_NAME);
				} else if (entry.containsKey(IConfigServer.ELEMENT_IDENTITY)) {
					name = ((IElementIdentity) entry
							.get(IConfigServer.ELEMENT_IDENTITY)).getName();
				}
				if (name != null) {
					list.add(name.substring(name.lastIndexOf("/") + 1));
				}
			}
			return list;
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to list  - " + e.getMessage(), e);
        }
    }

    /**
     * Create a folder at the specified path and assocatiate the attributes with
     * the folder. The parent folder must exist for the specified path.
     * @param path The path to the new folder.
     * @param attributes Meta attributes to associate with this folder.
     * @throws MgmtException
     */
    public void createFolder(String path, Map attributes)
    throws MgmtException
    {
        try
        {
            if (!path.endsWith("/"))
            {
                path = path + "/";
            }

            getConfigServer().createFolder(path, attributes, false);
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to create folder with meta-attributes - " + e.getMessage(), e);
        }
    }

    /**
     * Create a folder at the specified path. The parent folder must exist.
     * @param path
     * @throws MgmtException
     */
    public void createFolder(String path)
    throws MgmtException
    {
        try
        {
            if (!path.endsWith("/"))
            {
                path = path + "/";
            }

            getConfigServer().createFolder(path);
        }
        catch (Throwable e)
        {
            throw new MgmtException("Failed to create folder -" + e.getMessage(), e);
        }
    }

    /**
     * Delete the specified folder. Throws and Exception if the folder isn't
     * empty.
     * @param path
     * @throws MgmtException
     */
    public void deleteFolder(String path)
    throws MgmtException
    {
        try
        {
            getConfigServer().deleteFolder(path);
        }
        catch (Throwable e)
        {
            throw new MgmtException("Failed to create folder -" + e.getMessage(), e);
        }
    }

    /**
     * Recursively delete all folders and configurations under the specified
     * path, including the path folder itself.
     * @param path
     * @throws MgmtException
     */
    public void deleteFolderPath(String path)
    throws MgmtException
    {
        try
        {
            m_csu.deleteBeanPath(path);
        }
        catch (Throwable e)
        {
            throw new MgmtException("Failed to create folder -" + e.getMessage(), e);
        }
    }

    /**
     * Generates a java.lang.String array of all the configurations under the
     * specified parent folder.
     *
     * @param parent          The parent folder under all available child
     *                        configurations will be found.
     * @return                A list of configurations (java.lang.String).
     *                        This method will never return 'null' - instead a
     *                        zero-length array is used to simplify calling code.
     * @throws MgmtException  If a failure occurs
     */
    public String[] getConfigurations(String parentFolder)
    throws MgmtException
    {
        List list = getConfigurationNames(parentFolder);
        return (String[])list.toArray(new String[list.size()]);
    }

    /**
     * Check if a configuration with the specified name already exists in
     * the specified folder
     * @param name
     * @param parentFolder
     * @return
     */
    public boolean doesConfigurationNameExist(String name, String folder)
    {
        try
        {
            List names = getConfigurationNames(folder);

            return names.contains(name);
        }
        catch (Throwable  e)
        {
        }
        return false;
    }

    //-------------------------------------------------------------------------
    //
    // Global Configuration Property Lookup
    //
    //
    // Methods that let static Configuration properties be found.
    //
    // eg. a way to find out where CONTAINER configurations should be stored
    //     in the Directory Service.
    //
    //-------------------------------------------------------------------------

    public Object lookupValue(IConfigBean bean, Object key)
    {
        String type    = bean.getConfigType().getName();
        String version = bean.getConfigType().getVersion();

        return lookupValue(type, version, key);
    }

    public Class lookupReference(String type, String version)
    {
        return (Class)lookupValue(type, version, IMgmtBeanBase.CLASS_PROPERTY);
    }

    public Object lookupValue(String type, String version, Object key)
    {
        Map map = (Map)m_registered.get(type);

        return (map != null) ? map.get(key) : null;
    }

    public List listFolderElements(String type) throws MgmtException
    {
        return listConfigElements(type);
    }

    /**
     * Return list of folders names under specified path
     */
    public List listFolders(String parentFolder) throws MgmtException
    {
        try
        {
            Set      dirs   = getConfigServer().listFolders(parentFolder);
            List     result = new ArrayList(dirs.size());
            Iterator i      = dirs.iterator();

            while (i.hasNext())
            {
                result.add(i.next());
            }

            return result;
        }
        catch (Throwable e)
        {
             throw new MgmtException("Failed listFolders - " + e.getMessage(), e);
        }
    }

    /**
     * Delete folder and all his children's folders under specified path.
     * All folders must be empty.
     */
    protected void deleteFolders(String path) throws MgmtException
    {
        List folders = getFolders(new ArrayList(), path);

        folders.add(path); // add parent folder to the list

        for(Iterator iter = folders.iterator(); iter.hasNext(); )
        {
            deleteFolder((String)iter.next());
        }
    }

    /*
    * Recurcevily build list of all folders under a specified path
    */
    protected List getFolders(List list, String path) throws MgmtException
    {
        String[] folders = getFolders(path);

        for( int i = 0; i < folders.length; i++ )
        {
            String folder = folders[i];

            if(!list.contains(folder))
             {
                list.add(0, folder); // to keep required order
            }

            getFolders(list, folder);
        }
        return list;
    }

    public void setFolderMetaAttributes(String path, Map attributes) throws MgmtException
    {
        try
        {
            getConfigServer().setMetaAttributes(path, attributes);
        }
        catch(Throwable e)
        {
            throw new MgmtException("Can't set folder meta-attributes - " + e.getMessage(), e);
        }
    }

    //-------------------------------------------------------------------------

    /**
     * Loads the beans found under a DS path.
     *
     * @param dsPath  The DS path to where in the DS the config elements are kept
     * @return        A list of IMgmtBeanBase-derived beans
     * @throws MgmtException  If an error/load failure occurs
     */
    public List loadBeansInDirectory(String dsPath) throws MgmtException
    {
        ArrayList list = new ArrayList();
        
        try
        {
            Set set = getConfigServer().loadConfigElements(dsPath);
            
            for(Iterator iter = set.iterator(); iter.hasNext(); )
            {
                try
                {
                    list.add(createBeanObject((IConfigBean)iter.next()));
                }
                catch(Throwable e)
                {
                }
            }
        }
        catch (Throwable e)
        {
            throw new MgmtException("Failed to load beans - " + e.getMessage(), e);
        }
        return list;
    }

    /**
     * Save the specified bean to the DS.
     * @param  bean           The bean to be stored in the DS
     * @throws MgmtException  If an error/save failure occurs
     */
    public void saveBean(IMgmtBeanBase bean)
    throws MgmtException
    {
        if(bean == null)
        {
            throw new MgmtException("Attempt to save a null bean");
        }

        try
        {
            ((MgmtBeanBase)bean).saveBean();
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to save bean - " + e.getMessage(), e);
        }
    }

    /**
     * Save the specified beans to the DS.
     * @param  bean           The bean(s) to be stored in the DS
     * @throws MgmtException  If an error/save failure occurs
     */
    public void saveBeans(IMgmtBeanBase[] beans)
    throws MgmtException
    {
        try
        {
            for (int i = 0; i < beans.length; i++)
            {
                saveBean(beans[i]);
            }
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to save beans - " + e.getMessage(), e);
        }
    }

    /**
     * Generate the container boot configuration for the specified container bean.
     */
    public String exportContainerBootConfiguration(IMgmtBeanBase bean, String domainName)
    throws MgmtException
    {
        if(bean == null)
        {
            throw new MgmtException("Attempt to export container boot configuration of a null bean");
        }

        try
        {
            return m_csu.exportContainerBootConfiguration(bean.getConfigBean(), domainName);
        }
        catch (Throwable e)
        {
            throw new MgmtException(e.getMessage(), e);
        }
    }
    /**
     * Generate the ds boot configuration for the specified bean.
     */
    public String exportDSBootConfiguration(IMgmtBeanBase bean)
    throws MgmtException
    {
        try
        {
            return m_csu.exportDSBootConfiguration(bean.getConfigBean());
        }
        catch (Throwable e)
        {
            throw new MgmtException(e.getMessage(), e);
        }
    }

    /**
     * Get the specified bean from the DS.
     */
    public IMgmtBeanBase getBean(String viewPath)
    throws MgmtException
    {
        IMgmtBeanBase bean = null;

        try
        {
            IConfigElement cfgBean = getConfigServer().loadConfigElement(viewPath);
            bean = createBeanObject((IConfigBean)cfgBean);
        }
        catch (Throwable e)
        {
            throw new MgmtException("Failed to get bean - " + viewPath + "." + e.getMessage(), e);
        }
        return bean;
    }

    /**
     * Delete the specified bean from the DS.
     */
    public void deleteBean(IMgmtBeanBase bean)
    throws MgmtException
    {
        if(bean == null)
        {
            throw new MgmtException("Attempt to delete a null bean");
        }

        try
        {
            ((MgmtBeanBase)bean).deleteBean();
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to delete bean - " + e.getMessage(), e);
        }
    }

    /**
     * Delete the specified beans from the DS.
     *
     * @param beans
     * @throws MgmtException
     */
    public void deleteBeans(IMgmtBeanBase[] beans)
    throws MgmtException
    {
        try
        {
            for (int i = 0; i < beans.length; i++)
            {
                deleteBean(beans[i]);
            }
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to delete beans - " + e.getMessage(), e);
        }
    }

    /**
     * Return a name that is unique across all beans.
     * @return
     */
    public String createUniqueName()
    {
        return ConfigFactory.createGUID();
    }

    /*
     * Creates copy of the specified bean.
     *
     * @param bean
     * @return a copy of the specified bean.
     * @throws MgmtException
     */
    /*
    public IMgmtBeanBase createCopy(IMgmtBeanBase bean)
    throws MgmtException
    {
        if(bean == null)
            throw new MgmtException("Attempt to create a copy of a null bean");

        return ((MgmtBeanBase)bean).clone(false);
    }
    */

    /*
     * Creates template of the specified bean.
     *
     * @param bean
     * @throws MgmtException
     * @return a template(or instance) of the specified bean.
     */
    /*
    public IMgmtBeanBase createTemplate(IMgmtBeanBase bean)
    throws MgmtException
    {
        if(bean == null)
            throw new MgmtException("Attempt to create a template of a null bean");

        if (bean.isTemplate())
            throw new MgmtException("Failed to create template - supplied bean can't be a template bean");

        if (bean.isInstanceOfTemplate())
            throw new MgmtException("Failed to create template - supplied bean can't be an instance bean");

        return ((MgmtBeanBase)bean).clone(true);
    }
    */

    /*
     * Creates instance of the specified template bean.
     *
     * @param bean
     * @throws MgmtException
     * @return a template(or instance) of the specified bean.
     */
    /*
    public IMgmtBeanBase createInstance(IMgmtBeanBase bean)
    throws MgmtException
    {
        if(bean == null)
            throw new MgmtException("Attempt to create an instance of a null bean");

        if (!bean.isTemplate())
            throw new MgmtException("Failed to create template - supplied bean isn't a template bean");

        return ((MgmtBeanBase)bean).clone(true);
    }
    */

    //-------------------------------------------------------------------------
    //
    // Non-Public methods
    //
    //-------------------------------------------------------------------------

    /**
     * Register a Config Java Bean in the cache based on its type and version.
     * This cache will be used to dynamically create the correct config
     * bean based on its type and version
     */
    protected void registerBean(Class cls, String type, String cVersion, String pVersion)
    {
        String key = type;

        synchronized (m_registered)
        {
            if (!m_registered.containsKey(key))
            {
                HashMap map = new HashMap();

                map.put(IMgmtBeanBase.CLASS_PROPERTY,     cls);
                map.put(IMgmtBeanBase.TYPE_PROPERTY,      type);
                map.put(IMgmtBeanBase.C_VERSION_PROPERTY, cVersion);
                map.put(IMgmtBeanBase.P_VERSION_PROPERTY, pVersion);

                m_registered.put(key, map);
            }
        }
    }

    protected void registerBean(String type, String cVersion, String pVersion, Map register)
    {
        String key = type;

        synchronized (m_registered)
        {
            if (!m_registered.containsKey(key))
            {
                m_registered.put(key, register);
            }
        }
    }

    protected void registerBean(Map register)
    {
        String type     = (String)register.get(IMgmtBeanBase.TYPE_PROPERTY);
        String cVersion = (String)register.get(IMgmtBeanBase.C_VERSION_PROPERTY);
        String pVersion = (String)register.get(IMgmtBeanBase.P_VERSION_PROPERTY);
        Class  cls      = (Class)register.get(IMgmtBeanBase.CLASS_PROPERTY);

        if ((type != null) && (cVersion != null) && (cls != null))
        {
            registerBean(type, cVersion, pVersion, register);
        }
    }

    /*
    IMgmtBeanBase[] getTemplateInstances(IMgmtBeanBase templateBean)
    throws MgmtException
    {
        if (!(templateBean.getConfigBean().isPrototypeInstance()))
            throw new MgmtException("Failed to get template instances - supplied bean is not a template");

        IConfigPrototype prototype = (IConfigPrototype)templateBean.getConfigBean();
        ArrayList        list      = new ArrayList();

        try
        {
            Iterator i = prototype.getPrototypeInstances().iterator();

            while (i.hasNext())
            {
                IConfigBean instance = (IConfigBean)i.next();

                list.add(createBeanObject(instance));
            }
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to get template instances - " + e.getMessage(), e);
        }
        return (IMgmtBeanBase[])list.toArray(new IMgmtBeanBase[list.size()]);
    }
    */

    /*
    IMgmtBeanBase getTemplate(IMgmtBeanBase instanceBean)
    throws MgmtException
    {
        if (!instanceBean.getConfigBean().isPrototypeInstance())
            throw new MgmtException("Failed to get template - supplied bean is not a template instance");

        try
        {
            return createBeanObject((IConfigBean)instanceBean.getConfigBean().getPrototype());
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to get template - " + e.getMessage(), e);
        }
    }
    */

    public IMgmtBeanBase createBeanObject(IConfigBean bean) throws MgmtException
    {
        String name    = bean.getConfigType().getName();
        String version = bean.getConfigType().getVersion();
        
        // version will get ignored; we'll handle version <= current version
        Class cls = lookupReference(name, version);

        if(cls == null)
         {
            throw new MgmtException("Factory '" + getClass().getName() + "' can't create beans of type " +
                                    name + " and version " + version);
        // throw an error if we're trying to create a bean of a newer
        // version
        }
        
        try
        {
        	checkBeanVersion(cls, version);
            Constructor constr = cls.getConstructor(new Class[] { MgmtBeanFactory.class });

            IMgmtBeanBase base = (IMgmtBeanBase)constr.newInstance(new Object[] { this });

            ((MgmtBeanBase)base).setBean(bean);

            return base;
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to create bean for " + name + " - " + e.getMessage(), e);
        }
    }

    //-------------------------------------------------------------------------
    //
    // Recursive routines for automatically cloning all the subbeans under
    // a particular bean
    //
    //-------------------------------------------------------------------------

    /*
    public void cloneBean(HashMap       handled,
                          boolean       convert,
                          String[]      patterns,
                          IAttributeMap oData,
                          IAttributeMap cData)
    {
        for(Iterator iter = oData.keySet().iterator(); iter.hasNext(); )
        {
            String attributeName = (String)iter.next();

            Object oObj = null;
            Object cObj = null;

            // Can't assume that get (IAttributeMap) is going to return null if
            // it can't find anything...the ConfigLayer throws exceptions if
            // referenced entries can't be found...it shouldn't do this really!
            //
            try { oObj = oData.get(attributeName); } catch (ConfigServiceRuntimeException e) {}
            try { cObj = cData.get(attributeName); } catch (ConfigServiceRuntimeException e) {}

            if(oObj == null || cObj == null)
                continue;

            if(oObj instanceof IConfigBean)
            {
                try
                {
                    String      beanName  = ((IConfigBean)oObj).getName();
                    IConfigBean beanValue = null;

                    if(!handled.containsKey(beanName))
                    {
                        String newName = generateNameFromOldName(patterns, beanName);

                        if(convert)
                        {
                            if(oObj instanceof IConfigPrototype)
                                beanValue = (IConfigBean)((IConfigPrototype)oObj).newInstance(newName);
                            else
                                beanValue = (IConfigBean)((IConfigBean)oObj).createPrototype(newName);
                        }
                        else
                        {
                            if (((IConfigBean)oObj).isPrototypeInstance())
                            {
                                beanValue = (IConfigBean)((IConfigBean)oObj).clonePrototypeInstance(newName);
                            }
                            else
                            {
                                beanValue = (IConfigBean)((IConfigBean)oObj).clone();
                                beanValue.setName(newName);
                            }
                        }

                        handled.put(beanName, beanValue);
                    }
                    else
                        beanValue = (IConfigBean)handled.get(beanName);

                    cData.setAttribute(attributeName, beanValue);
                }
                catch(Throwable e)
                {
                    System.err.println("Failed to create clone for sub bean - " + e.getMessage());
                }
            }

            if(oObj instanceof IAttributeMap)
                cloneBean(handled, convert, patterns, (IAttributeMap)oObj, (IAttributeMap)cObj);
            else if(oObj instanceof IAttributeList)
                cloneBean(handled, convert, patterns, (IAttributeList)oObj, (IAttributeList)cObj);
        }
    }
    */

    /*
    private void cloneBean(HashMap        handled,
                           boolean        convert,
                           String[]       patterns,
                           IAttributeList oData,
                           IAttributeList cData)
    {
        for(int i = 0; i < oData.size(); i++)
        {
            Object oObj = null;
            Object cObj = null;

            // Can't assume that get (IAttributeMap) is going to return null if
            // it can't find anything...the ConfigLayer throws exceptions if
            // referenced entries can't be found...it shouldn't do this really!
            //
            try { oObj = oData.get(i); } catch (ConfigServiceRuntimeException e) {}
            try { cObj = cData.get(i); } catch (ConfigServiceRuntimeException e) {}

            if(oObj == null || cObj == null)
                continue;


            if(oObj instanceof IConfigBean)
            {
                try
                {
                    String      beanName  = ((IConfigBean)oObj).getName();
                    IConfigBean beanValue = null;

                    if(!handled.containsKey(beanName))
                    {
                        String newName = generateNameFromOldName(patterns, beanName);

                        if(convert)
                        {
                            if(oObj instanceof IConfigPrototype)
                                beanValue = (IConfigBean)((IConfigPrototype)oObj).newInstance(newName);
                            else
                                beanValue = (IConfigBean)((IConfigBean)oObj).createPrototype(newName);
                        }
                        else
                        {
                            if (((IConfigBean)oObj).isPrototypeInstance())
                            {
                                beanValue = (IConfigBean)((IConfigBean)oObj).clonePrototypeInstance(newName);
                            }
                            else
                            {
                                beanValue = (IConfigBean)((IConfigBean)oObj).clone();
                                beanValue.setName(newName);
                            }
                        }
                        handled.put(beanName, beanValue);
                    }
                    else
                        beanValue = (IConfigBean)handled.get(beanName);

                    cData.set(i, beanValue);
                }
                catch(Throwable e)
                {
                    System.err.println("Failed to create clone for sub bean - " + e.getMessage());
                }
            }

            if(oObj instanceof IAttributeMap)
                cloneBean(handled, convert, patterns, (IAttributeMap)oObj, (IAttributeMap)cObj);
            else if(oObj instanceof IAttributeList)
                cloneBean(handled, convert, patterns, (IAttributeList)oObj, (IAttributeList)cObj);
        }
    }
    */

    protected String generateNameFromOldName(String[] patterns, String name)
    {
        String namePath = name.substring(0, name.lastIndexOf('/'));

        for(int i = 0; i < patterns.length; i++)
        {
            String patternPath = patterns[i].substring(0, patterns[i].lastIndexOf('/'));

            if(patternPath.equals(namePath))
            {
                return ConfigFactory.generateDSName(patterns[i]);
            }
        }
        return null;
    }
    
    protected void checkBeanVersion(Class cls, String elVersionStr)
        throws Exception
    {
        //every bean interface defines a DS_C_VERSION. It should
    	// never be null
    	Field beanAPIVersionField = cls.getField("DS_C_VERSION");
    	String beanAPIVersionStr = (String) beanAPIVersionField.get(cls);
    	if (elVersionStr.compareTo(beanAPIVersionStr) > 0)
        {
            throw new MgmtException("Current API release version " + beanAPIVersionStr + " cannot handle greater bean release version " + elVersionStr);
        }
    	
    }
}
