package com.sonicsw.ma.mgmtapi.config.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import com.sonicsw.ma.mgmtapi.config.AttributeNotFoundException;
import com.sonicsw.ma.mgmtapi.config.IMgmtAttributeMetaData;
import com.sonicsw.ma.mgmtapi.config.IMgmtBase;
import com.sonicsw.ma.mgmtapi.config.IMgmtBeanBase;
import com.sonicsw.ma.mgmtapi.config.IMgmtListBase;
import com.sonicsw.ma.mgmtapi.config.IMgmtSubBeanBase;
import com.sonicsw.ma.mgmtapi.config.MgmtBeanFactory;
import com.sonicsw.ma.mgmtapi.config.MgmtException;
import com.sonicsw.mx.config.ConfigFactory;
import com.sonicsw.mx.config.ConfigServerUtility;
import com.sonicsw.mx.config.IAttributeList;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IAttributeMetaData;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigPath;

public abstract class MgmtBeanBase implements IMgmtBeanBase, Cloneable
{
    private MgmtBeanFactory m_factory;
    private IConfigBean     m_bean;

    public MgmtBeanBase(MgmtBeanFactory factory)
    {
        m_factory = factory;
        m_bean    = null;
    }

    /**
     * Loads a bean with no type validation, i.e. we don't make sure that the
     * loaded bean is of the correct type.
     *
     * @param viewPath        The logical path of the bean to be loaded.
     * @throws MgmtException  Thrown if the bean could not be found.
     *
     * @deprecated  Use replaced by {@link #loadBean(String,String)}
     */
    public void loadBean(String viewPath)
    throws MgmtException
    {
        loadBean(viewPath, null);
    }

    /**
     * Loads a bean and validates that it is of the expected type.
     *
     * @param viewPath        The logical path of the bean to be loaded.
     * @param type            The expected config type of the bean.
     * @throws MgmtException  Thrown if the bean could not be found or the bean
     *                        is not the right type.
     */
    public void loadBean(String viewPath, String type)
    throws MgmtException
    {
        try
        {
            m_bean = (IConfigBean)((AbstractMgmtBeanFactory)m_factory).getConfigServer().loadConfigElement(viewPath);
            m_factory.checkBeanVersion(this.getClass(), m_bean.getConfigType().getVersion());
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to load configuration for " + viewPath + " - " + e.getMessage(), e);
        }

        if(m_bean == null)
        {
            throw new MgmtException("Configuration not found for " + viewPath);
        }
        else
        if ((type != null) && !m_bean.getConfigType().getName().equals(type))
        {
            throw new MgmtException("Loaded configuration '" + viewPath +
              "' is not of the right type '" + m_bean.getConfigType().getName() + "'" +
              " - was expecting '" + type + "'");
        }
    }

    /* package protected */ void setBean(IConfigBean bean)
    {
        m_bean = bean;
    }

    protected void createFolderElement() throws MgmtException
    {
        try
        {
            String path = m_bean.getName();
            path = path.substring(0, path.lastIndexOf("/"));

            //Irene: Cuurently TxnConfigServer doesn't support the commented out  method
            //m_factory.createFolder(path, m_bean.getMetaAttributes());

            m_factory.createFolder(path);
            m_factory.setFolderMetaAttributes(path, m_bean.getMetaAttributes());
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to create folder element - " + e.getMessage(), e);
        }
    }

    protected void setFolderElementAttributes() throws MgmtException
    {
        try
        {
            String path = m_bean.getName();
            path = path.substring(0, path.lastIndexOf("/"));
            HashMap map = m_bean.getMetaAttributes();
            if (map !=  null)
            {
                m_factory.setFolderMetaAttributes(path, map);
            }
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to set folder element attributes- " + e.getMessage(), e);
        }
    }

    public void createBean(String viewPath, String type, String cVersion, String pVersion)
    throws MgmtException
    {
        createBean(viewPath, type, cVersion, pVersion, true);
    }

    public void createBean(String viewPath, String type, String cVersion, String pVersion, boolean setAttributes)
    throws MgmtException
    {
        try
        {
            m_bean = m_factory.getConfigServer().createConfigBean(viewPath, type, cVersion, false);

            if(setAttributes)
            {
                // Set metaAttributes on the configuration
                HashMap map = new HashMap();
                map.put(ConfigServerUtility.TYPE,            type);
                map.put(ConfigServerUtility.CONFIG_VERSION,  cVersion);
                map.put(ConfigServerUtility.PRODUCT_VERSION, pVersion);

                m_bean.setMetaAttributes(map);
            }
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to create configuration for " + viewPath + " - " + e.getMessage(), e);
        }
    }

    private final List getDeleteBeanList() throws MgmtException
    {
        ArrayList list = new ArrayList();

        list.add(m_bean.getName());

        return list;
    }


    /**
     * Called by the domainFactory to allow the bean to perform additional
     * operations on the config server that are required at the same time as
     * deleting this bean.
     * By default we will just delete this single bean
     */
    protected void deleteBean() throws Exception
    {
        m_factory.getConfigServer().removeConfigElement(m_bean.getName());
    }

    /**
     * Called by the domain Factory to allow the bean to perform additional
     * operations on the config server that are required at the same time as
     * saving this bean.
     * By default we will just save this bean to the config Server
     * @param server
     */
    protected void saveBean() throws Exception
    {
        m_factory.getConfigServer().storeConfigElement(m_bean);
    }


    @Override
    public IAttributeMap getAttributeMap()
    {
        return null;
    }

    @Override
    public IAttributeList getAttributeList()
    {
        return null;
    }

    @Override
    public IConfigBean getConfigBean()
    {
        return m_bean;
    }

    @Override
    public String getConfigBeanName()
    {
        if(m_bean != null)
        {
            return m_bean.getName();
        }

        return null;
    }

    @Override
    public String getConfigBeanNameTail()
    {
        String name = getConfigBeanName();

        int index = name.lastIndexOf('/');

        if(index == -1)
        {
            return name;
        }

        return name.substring(index + 1);
    }

    public String getConfigBeanNamePrefix()
    {
        String name = getConfigBeanName();

        int index = name.lastIndexOf('/');

        if(index == -1)
        {
            return "/";
        }

        return name.substring(0, index);
    }

    @Override
    public IConfigPath getConfigPath()
    {
        return null;
    }

    @Override
    public IConfigPath getConfigPath(String name)
    {
        return ConfigFactory.createConfigPath(name);
    }

    @Override
    public IMgmtBase getParent()
    {
        return null;
    }

    @Override
    public MgmtBeanFactory getMgmtBeanFactory()
    {
        return m_factory;
    }

    @Override
    public boolean isNewBean()
    {
        return m_bean.isNew();
    }

    /*
    public boolean isTemplate()
    {
        if(m_bean != null && (m_bean instanceof IConfigPrototype))
            return true;

        return false;
    }
    */

    /*
    public boolean isInstanceOfTemplate()
    {
        if(m_bean != null)
            return m_bean.isPrototypeInstance();

        return false;
    }
    */

    /*
    public IMgmtBeanBase[] getTemplateInstances()
    throws MgmtException
    {
        try
        {
            return ((AbstractMgmtBeanFactory)m_factory).getTemplateInstances(this);
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to get template instances - " + e.getMessage(), e);
        }
    }
    */

    /*
    public IMgmtBeanBase getTemplate()
    throws MgmtException
    {
        try
        {
            return ((AbstractMgmtBeanFactory)m_factory).getTemplate(this);
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to get template - " + e.getMessage(), e);
        }
    }
    */

    @Override
    public void removeAttribute(String name) throws MgmtException
    {
        try
        {
            m_bean.removeAttribute(getConfigPath(name));
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to remove attribute " + name +
                                    " - " + e.getMessage(), e);
        }
    }

    public Object getAttribute(String name) throws MgmtException
    {
        try
        {
            Object ret = m_bean.getAttribute(getConfigPath(name));

            if(ret instanceof IConfigBean)
            {
                ret = ((AbstractMgmtBeanFactory)getMgmtBeanFactory()).createBeanObject((IConfigBean)ret);
            }

            return ret;
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to get attribute " + name +
                                    " - " + e.getMessage(), e);
        }
    }

    public void setAttribute(String name, Object val) throws MgmtException
    {
        try
        {
            if (val != null)
            {
                m_bean.setAttribute(getConfigPath(name), val);
            }
            else
            {
                m_bean.removeAttribute(getConfigPath(name));
            }
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to set attribute " + name +
                                    " - " + e.getMessage(), e);
        }
    }

    @Override
    public void setAttribute(String name, IMgmtBase val) throws MgmtException
    {
        if(val ==null){
            setAttribute(name, (Object)val);
            return;
        }

        Object realVal = null;

        if(val instanceof IMgmtBeanBase)
        {
            realVal = ((IMgmtBeanBase)val).getConfigBean();
        }
        else if(val instanceof IMgmtListBase)
        {
            realVal = val.getAttributeList();
        }
        else if(val instanceof IMgmtSubBeanBase)
        {
            realVal = val.getAttributeMap();
        }

        if(realVal == null)
        {
            throw new MgmtException("Cannot setAttribute. Value does not contain a valid list or map");
        }

        setAttribute(name, (Object)realVal);
    }

    @Override
    public String getStringAttribute(String name) throws MgmtException
    {
        Object obj = getAttribute(name);

        if(obj == null)
        {
            throw new AttributeNotFoundException("String attribute " + name + " is not defined");
        }

        if(!(obj instanceof String))
        {
            throw new AttributeNotFoundException("Attribute " + name + " is of type String");
        }

        return (String)obj;
    }

    @Override
    public void setStringAttribute(String name, String val) throws MgmtException
    {
        setAttribute(name, val);
    }

    @Override
    public boolean getBooleanAttribute(String name) throws MgmtException
    {
        Object obj = getAttribute(name);

        if(obj == null)
        {
            throw new AttributeNotFoundException("Boolean attribute " + name + " is not defined");
        }

        if(!(obj instanceof Boolean))
        {
            throw new AttributeNotFoundException("Attribute " + name + " is of type Boolean");
        }

        return ((Boolean)obj).booleanValue();
    }

    @Override
    public void setBooleanAttribute(String name, boolean val) throws MgmtException
    {
        setAttribute(name, new Boolean(val));
    }

    @Override
    public int getIntegerAttribute(String name) throws MgmtException
    {
        Object obj = getAttribute(name);

        if(obj == null)
        {
            throw new AttributeNotFoundException("Integer attribute " + name + " is not defined");
        }

        if(!(obj instanceof Integer))
        {
            throw new AttributeNotFoundException("Attribute " + name + " is of type Integer");
        }

        return ((Integer)obj).intValue();
    }

    @Override
    public void setIntegerAttribute(String name, int val) throws MgmtException
    {
        setAttribute(name, new Integer(val));
    }

    @Override
    public long getLongAttribute(String name) throws MgmtException
    {
        Object obj = getAttribute(name);

        if(obj == null)
        {
            throw new AttributeNotFoundException("Long attribute " + name + " is not defined");
        }

        if(!(obj instanceof Long))
        {
            throw new AttributeNotFoundException("Attribute " + name + " is of type Long");
        }

        return ((Long)obj).longValue();
    }

    @Override
    public void setLongAttribute(String name, long val) throws MgmtException
    {
        setAttribute(name, new Long(val));
    }

    @Override
    public byte[] getByteArrayAttribute(String name) throws MgmtException
    {
        Object obj = getAttribute(name);

        if(obj == null)
        {
            throw new AttributeNotFoundException("Byte[] attribute " + name + " is not defined");
        }

        if(!(obj instanceof byte[]))
        {
            throw new AttributeNotFoundException("Attribute " + name + " is of type byte[]");
        }

        return (byte[])obj;
    }

    @Override
    public void setByteArrayAttribute(String name, byte[] val) throws MgmtException
    {
        setAttribute(name, val);
    }

    @Override
    public java.math.BigDecimal getBigDecimalAttribute(String name) throws MgmtException
    {
        Object obj = getAttribute(name);

        if(obj == null)
        {
            throw new AttributeNotFoundException("BigDecimal attribute " + name + " is not defined");
        }

        if(!(obj instanceof java.math.BigDecimal))
        {
            throw new AttributeNotFoundException("Attribute " + name + " is of type BigDecimal");
        }

        return (java.math.BigDecimal)obj;
    }

    @Override
    public void setBigDecimalAttribute(String name, java.math.BigDecimal val) throws MgmtException
    {
        setAttribute(name, val);
    }

    @Override
    public IMgmtBeanBase getReferenceAttribute(String name) throws MgmtException
    {
        Object obj = getAttribute(name);

        if(obj == null)
        {
            throw new AttributeNotFoundException("Reference attribute " + name + " is not defined.");
        }

        if(!(obj instanceof IMgmtBeanBase))
        {
            throw new AttributeNotFoundException("Attribute " + name + " is of type Reference");
        }

        return (IMgmtBeanBase)obj;
    }

    @Override
    public void setReferenceAttribute(String name, IMgmtBeanBase val) throws MgmtException
    {
        setAttribute(name, val);
    }

    @Override
    public IMgmtAttributeMetaData getAttributeMetaData(String name) throws MgmtException
    {
        try
        {
            IAttributeMetaData ret = m_bean.getAttributeMetaData(getConfigPath(name));

            return new MgmtAttributeMetaData(ret);
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to get attribute metadata " + name +
                                    " - " + e.getMessage(), e);
        }
    }

    //------------------------------------------------------------------------
    // Recursive routines for automatically obtaining a list of all referenced
    // beans under a particular bean
    //
    protected List getReferencedBeanList(List list, IAttributeMap data)
    {
        for(Iterator iter = data.keySet().iterator(); iter.hasNext(); )
        {
            Object obj = data.get((String)iter.next());

            if(obj instanceof IConfigBean)
            {
                String name = ((IConfigBean)obj).getName();

                if(!list.contains(name))
                {
                    list.add(name);
                }
            }

            if(obj instanceof IAttributeMap)
            {
                getReferencedBeanList(list, (IAttributeMap)obj);
            }
            else if(obj instanceof IAttributeList)
            {
                getReferencedBeanList(list, (IAttributeList)obj);
            }
        }
        return list;
    }

    protected List getReferencedBeanList(List list, IAttributeList data)
    {
        for(Iterator iter = data.iterator(); iter.hasNext(); )
        {
            Object obj = iter.next();

            if(obj instanceof IConfigBean)
            {
                String name = ((IConfigBean)obj).getName();

                if(!list.contains(name))
                {
                    list.add(name);
                }
            }

            if(obj instanceof IAttributeMap)
            {
                getReferencedBeanList(list, (IAttributeMap)obj);
            }
            else if(obj instanceof IAttributeList)
            {
                getReferencedBeanList(list, (IAttributeList)obj);
            }
        }
        return list;
    }

    protected String getNextName(String path, String originalName, String prefix)
    {
        try
        {
            // List names = getMgmtBeanFactory().getConfigurationNames(path);
            List names = getMgmtBeanFactory().list(path);

            // stuff names into a hashMap
            HashMap tempMap = new HashMap();

            for(Iterator iter = names.iterator(); iter.hasNext(); )
            {
                String fsPath = (String)iter.next();
                String name   = fsPath.substring(fsPath.lastIndexOf('/') + 1);
                tempMap.put(name, name);
            }

            String candidate = (prefix == null) ? originalName : (prefix + " of " + originalName);

            if(!tempMap.containsKey(candidate))
            {
                return candidate;
            }

            for(int i = 2; ; i++)
            {
                candidate = (prefix == null) ? (originalName + "_" + (i - 1)):(prefix + "(" + i + ") of " + originalName);

                if(!tempMap.containsKey(candidate))
                {
                    return candidate;
                }
            }
        }
        catch(Exception e)
        {
        }
        return null;
    }

    /**
     * Return a name that is unique across all beans.
     * @return
     */
    @Override
    public String createUniqueName()
    {
        return ConfigFactory.createGUID();
    }

    protected String createUniqueName(String name, boolean appendDefault)
    {
        String result = new String(name + "/"  + ConfigFactory.createGUID());

        if (appendDefault)
        {
            result = result + "/" + ConfigServerUtility.DEFAULT_SUFFIX;
        }

        return result;
    }

    protected String createUniqueName(boolean appendDefault)
    {
        return createUniqueName(getConfigBeanNamePrefix(), appendDefault);
    }

    
    @Override
    protected Object clone()
    throws CloneNotSupportedException
    {
        return (MgmtBeanBase)super.clone();
    }
    
    @Override
    public IMgmtBeanBase clone(String viewName)
    throws MgmtException
    {
        MgmtBeanBase newBean = null;
       
        try
        {
            newBean = (MgmtBeanBase)clone();
            String path = (viewName == null) ? getConfigBean().getName(): viewName;

            if (path.endsWith(ConfigServerUtility.DEFAULT_SUFFIX))
            {
                path = path.substring(0, path.lastIndexOf('/'));
            }

            int index = path.lastIndexOf('/');
            
            String name = (viewName == null) ? 
                           getNextName(path.substring(0, index), path.substring(index + 1), "Copy"):
                           path.substring(index + 1);
                           
            if (viewName != null) //recurcively create folder path
            {
                try
                {
                    m_factory.createFolder(path.substring(0, (index + 1)));
                }
                catch(Exception e){} //nothing to catch, most likely folder path exists
            }
            
            path = path.substring(0, (index + 1)) + name;
            
            IConfigBean cloneBean = (IConfigBean)getConfigBean().clone();

            cloneBean.setName(path);
            cloneBean.setMetaAttributes(m_bean.getMetaAttributes());
            
            newBean.setBean(cloneBean);
    }
    catch(Throwable e)
    {
        throw new MgmtException("Failed to clone bean - " + e.toString(), e);
    }
        return newBean; 
    }
    
    
    /*
    public IMgmtBeanBase clone(boolean convert)
    throws MgmtException
    {
        MgmtBeanBase newBean = null;
        try
        {
            HashMap metaAttributes = getConfigBean().getMetaAttributes();

            newBean = (MgmtBeanBase)clone();
            String path = getConfigBean().getName();

            if (path.endsWith(ConfigServerUtility.DEFAULT_SUFFIX))
                path = path.substring(0, path.lastIndexOf('/'));

            int index = path.lastIndexOf('/');
            String name;

            if (convert)
            {
                // we want to convert the type of the config bean. If it
                // is a template create an instance, if an instance create
                // a template
                if (isTemplate())
                {
                    name = getNextName(path.substring(0, index), path.substring(index + 1), "Instance");
                    path = path.substring(0, (index + 1)) + name;
                    IConfigBean configBean = (IConfigBean)((IConfigPrototype)getConfigBean()).newInstance(path);
                    //[IL] Next two lines are not working, element already stored on newInstance() call
                    //Setting meta-attributes should be done at Config Layer.
                    metaAttributes.put(ConfigServerUtility.TEMPLATE_TYPE, ConfigServerUtility.TEMPLATE_TYPE_INSTANCE);
                    configBean.setMetaAttributes(metaAttributes);
                    //
                    newBean.setBean(configBean);
                }
                else
                {
                    // Some configurations can't be templated...
                    //
                    Boolean canTemplate = (Boolean)getProperty(TEMPLATE_PROPERTY);

                    System.out.println("canTemplate = " + canTemplate);

                    if ((canTemplate == null) || !canTemplate.booleanValue())
                        throw new MgmtException("bean of type " + (String)getProperty(TYPE_PROPERTY) + " does not support templates");

                    name = getNextName(path.substring(0, index), path.substring(index + 1), "Template");
                    path = path.substring(0, (index + 1)) + name;
                    IConfigBean configBean = (IConfigBean)getConfigBean().createPrototype(path);
                    metaAttributes.put(ConfigServerUtility.TEMPLATE_TYPE, ConfigServerUtility.TEMPLATE_TYPE_PROTOTYPE);
                    configBean.setMetaAttributes(metaAttributes);

                    newBean.setBean(configBean);
                }
            }
            else
            {
                name = getNextName(path.substring(0, index), path.substring(index + 1), "Copy");
                path = path.substring(0, (index + 1)) + name;
                if (isInstanceOfTemplate())
                    newBean.setBean((IConfigBean)getConfigBean().clonePrototypeInstance(path));
                else
                {
                    IConfigBean cloneBean = (IConfigBean)getConfigBean().clone();

                    cloneBean.setName(path);

                    newBean.setBean(cloneBean);
                }
            }

            // This cloned bean is new but calling setBean sets the new flag
            // to false for some reason! so we force it here.
            //
            ((MgmtBeanBase)newBean).setIsNewBean(true);
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to clone bean - " + e.toString(), e);
        }

        return newBean;
    }
    */

   
    
    
    @Override
    public Object getProperty(Object propertyId)
    {
        return m_factory.lookupValue(getConfigBean(), propertyId);
    }

    @Override
    public String toString()
    {
        StringBuffer sb = new StringBuffer();

        sb.append(getClass().getName()).append(" (");

        sb.append(", cV").append((String)getProperty(C_VERSION_PROPERTY));
        sb.append(", pV").append((String)getProperty(P_VERSION_PROPERTY));

        /*
        if (isTemplate())
            sb.append(", template");

        if (this.isInstanceOfTemplate())
            sb.append(", instance of template");
        */

        sb.append(", ").append(getConfigBean().getName());

        sb.append(")");

        return sb.toString();
    }
    
    @Override
    public void setAnnotation(String value)
    {
    	if (m_bean == null)
        {
            return;
        }
    		
    	m_bean.setAnnotation(value);
    }
    
    @Override
    public String getAnnotation()
    {
    	if (m_bean == null)
        {
            return null;
        }
    	
    	return m_bean.getAnnotation();
    }
}