package com.sonicsw.ma.mgmtapi.config.impl;

import com.sonicsw.ma.mgmtapi.config.AttributeNotFoundException;
import com.sonicsw.ma.mgmtapi.config.IMgmtAttributeMetaData;
import com.sonicsw.ma.mgmtapi.config.IMgmtBase;
import com.sonicsw.ma.mgmtapi.config.IMgmtBeanBase;
import com.sonicsw.ma.mgmtapi.config.IMgmtListBase;
import com.sonicsw.ma.mgmtapi.config.IMgmtSubBeanBase;
import com.sonicsw.ma.mgmtapi.config.MgmtBeanFactory;
import com.sonicsw.ma.mgmtapi.config.MgmtException;
import com.sonicsw.mx.config.ConfigFactory;
import com.sonicsw.mx.config.IAttributeList;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IAttributeMetaData;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigPath;

public abstract class MgmtSubBeanBase implements IMgmtSubBeanBase
{
    protected IMgmtBase      m_parent;
    protected IConfigBean    m_bean;
    protected IConfigPath    m_path;
    protected IAttributeMap  m_map;
    protected IAttributeList m_list;
    protected String         m_name;

    public MgmtSubBeanBase(IMgmtBase parent, String name, boolean isNew)
    throws MgmtException
    {
        m_name   = name;
        m_parent = parent;
        m_bean   = parent.getConfigBean();

        IConfigPath parentPath = parent.getConfigPath();

        // We are using array, because it could be a name of Attribute that contains ".",
        // such as Sonic.deadMessage. And "." is a separator in IConfigPath.
        if(isNew)
        {
            m_path = null;
        }
        else if(parentPath != null)
        {
            m_path = ConfigFactory.createConfigPath(parentPath).append(new String[] { name });
        }
        else
        {
            m_path = ConfigFactory.createConfigPath().append(new String[] { name });
        }

        Object parentObj = null;

        if(parent.getAttributeMap() != null)
        {
            parentObj = parent.getAttributeMap();
        }
        else if(parent.getAttributeList() != null)
        {
            parentObj = parent.getAttributeList();
        }
        else if(parentPath != null)
        {
            parentObj = m_bean.getAttribute(parentPath);
        }
        else
        {
            parentObj = m_bean;
        }

        if(isNew)
        {
            try
            {
                if(parentObj instanceof IAttributeMap)
                {
                    if(this instanceof IMgmtListBase)
                    {
                        m_list = ((IAttributeMap)parentObj).createAttributeList(name);
                    }
                    else if(this instanceof IMgmtSubBeanBase)
                    {
                        m_map = ((IAttributeMap)parentObj).createAttributeMap(name);
                    }
                }
                else if(parentObj instanceof IAttributeList)
                {
                    if(this instanceof IMgmtListBase)
                    {
                        m_list = ((IAttributeList)parentObj).createAttributeList(name);
                    }
                    else if(this instanceof IMgmtSubBeanBase)
                    {
                        m_map = ((IAttributeList)parentObj).createAttributeMap(name);
                    }
                }
            }
            catch(Throwable e)
            {
                throw new MgmtException("Failed to create list/map for " + name +
                                        " - " + e.getMessage(), e);
            }
        }
        else
        {
            Object ret = null;

            try
            {
                if(parentObj instanceof IConfigBean)
                {
                    ret = ((IConfigBean)parentObj).getAttribute(m_path);
                }
                else if(parentObj instanceof IAttributeMap)
                {
                    ret = ((IAttributeMap)parentObj).getAttribute(name);
                }
                else if(parentObj instanceof IAttributeList)
                {
                    ret = ((IAttributeList)parentObj).getAttribute(Integer.parseInt(name));
                }
            }
            catch(Exception e)
            {
            }

            if (ret == null)
            {
                throw new MgmtException("Failed to get list/map for " + name);
            }
        }
    }

    @Override
    public IMgmtBase getParent()
    {
        return m_parent;
    }

    @Override
    public IConfigBean getConfigBean()
    {
        return m_bean;
    }

    @Override
    public String getConfigBeanName()
    {
        if(m_bean != null)
        {
            return m_bean.getName();
        }

        return null;
    }

    @Override
    public String getConfigBeanNameTail()
    {
        String name = getConfigBeanName();

        int index = name.lastIndexOf('/');

        if(index == -1)
        {
            return name;
        }

        return name.substring(index + 1);
    }

    @Override
    public MgmtBeanFactory getMgmtBeanFactory()
    {
        if(m_parent == null)
        {
            return null;
        }

        return m_parent.getMgmtBeanFactory();
    }

    @Override
    public IAttributeMap getAttributeMap()
    {
        return m_map;
    }

    @Override
    public IAttributeList getAttributeList()
    {
        return m_list;
    }

    @Override
    public IConfigPath getConfigPath()
    {
        return m_path;
    }

    @Override
    public IConfigPath getConfigPath(String name)
    {
        //Sonic00022831, we need to path array, since name can contain '.'
        if(m_path != null)
        {
            return ConfigFactory.createConfigPath(m_path).append(new String[] {name});
        }

        return ConfigFactory.createConfigPath(name);
    }

    protected Object getAttribute(String name) throws MgmtException
    {
        try
        {
            Object ret = null;

            if(m_map != null)
            {
                ret = m_map.getAttribute(name);
            }
            else
            {
                ret = m_bean.getAttribute(getConfigPath(name));
            }

            if(ret instanceof IConfigBean)
            {
                ret = ((AbstractMgmtBeanFactory)getMgmtBeanFactory()).createBeanObject((IConfigBean)ret);
            }

            return ret;
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to get attribute " + name +
                                    " - " + e.getMessage(), e);
        }
    }

    protected void setAttribute(String name, Object val) throws MgmtException
    {
        try
        {
            if (val != null)
            {
                if(m_map != null)
                {
                    m_map.setAttribute(name, val);
                }
                else
                {
                    m_bean.setAttribute(getConfigPath(name), val);
                }
            }
            else
            {
                if(m_map != null)
                {
                    m_map.removeAttribute(name);
                }
                else
                {
                    m_bean.removeAttribute(getConfigPath(name));
                }
            }
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to set attribute " + name +
                                    " - " + e.getMessage(), e);
        }
    }

    @Override
    public void removeAttribute(String name) throws MgmtException
    {
        try
        {
            if(m_map != null)
            {
                m_map.removeAttribute(name);
            }
            else
            {
                m_bean.removeAttribute(getConfigPath(name));
            }
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to remove attribute " + name +
                                    " - " + e.getMessage(), e);
        }
    }

    @Override
    public void setAttribute(String name, IMgmtBase val) throws MgmtException
    {
        Object realVal = null;

        if(val instanceof IMgmtListBase)
        {
            realVal = val.getAttributeList();
        }
        else if(val instanceof IMgmtSubBeanBase)
        {
            realVal = val.getAttributeMap();
        }

        if(realVal == null)
        {
            throw new MgmtException("Cannot setAttribute. Value does not contain a valid list or map");
        }

        setAttribute(name, (Object)realVal);
    }

    @Override
    public String getStringAttribute(String name) throws MgmtException
    {
        Object obj = getAttribute(name);

        if(obj == null)
        {
            throw new AttributeNotFoundException("String attribute " + name + " is not defined");
        }

        if(!(obj instanceof String))
        {
            throw new AttributeNotFoundException("Attribute " + name + " is of type String");
        }

        return (String)obj;
    }

    @Override
    public void setStringAttribute(String name, String val) throws MgmtException
    {
        setAttribute(name, val);
    }

    @Override
    public boolean getBooleanAttribute(String name) throws MgmtException
    {
        Object obj = getAttribute(name);

        if(obj == null)
        {
            throw new AttributeNotFoundException("Boolean attribute " + name + " is not defined");
        }

        if(!(obj instanceof Boolean))
        {
            throw new AttributeNotFoundException("Attribute " + name + " is of type Boolean");
        }

        return ((Boolean)obj).booleanValue();
    }

    @Override
    public void setBooleanAttribute(String name, boolean val) throws MgmtException
    {
        setAttribute(name, new Boolean(val));
    }

    @Override
    public int getIntegerAttribute(String name) throws MgmtException
    {
        Object obj = getAttribute(name);

        if(obj == null)
        {
            throw new AttributeNotFoundException("Integer attribute " + name + " is not defined");
        }

        if(!(obj instanceof Integer))
        {
            throw new AttributeNotFoundException("Attribute " + name + " is of type Integer");
        }

        return ((Integer)obj).intValue();
    }

    @Override
    public void setIntegerAttribute(String name, int val) throws MgmtException
    {
        setAttribute(name, new Integer(val));
    }

    @Override
    public long getLongAttribute(String name) throws MgmtException
    {
        Object obj = getAttribute(name);

        if(obj == null)
        {
            throw new AttributeNotFoundException("Long attribute " + name + " is not defined");
        }

        if(!(obj instanceof Long))
        {
            throw new AttributeNotFoundException("Attribute " + name + " is of type Long");
        }

        return ((Long)obj).longValue();
    }

    @Override
    public void setLongAttribute(String name, long val) throws MgmtException
    {
        setAttribute(name, new Long(val));
    }

    @Override
    public byte[] getByteArrayAttribute(String name) throws MgmtException
    {
        Object obj = getAttribute(name);

        if(obj == null)
        {
            throw new AttributeNotFoundException("Byte[] attribute " + name + " is not defined");
        }

        if(!(obj instanceof byte[]))
        {
            throw new AttributeNotFoundException("Attribute " + name + " is of type byte[]");
        }

        return (byte[])obj;
    }

    @Override
    public void setByteArrayAttribute(String name, byte[] val) throws MgmtException
    {
        setAttribute(name, val);
    }

    @Override
    public java.math.BigDecimal getBigDecimalAttribute(String name) throws MgmtException
    {
        Object obj = getAttribute(name);

        if(obj == null)
        {
            throw new AttributeNotFoundException("BigDecimal attribute " + name + " is not defined");
        }

        if(!(obj instanceof java.math.BigDecimal))
        {
            throw new AttributeNotFoundException("Attribute " + name + " is of type BigDecimal");
        }

        return (java.math.BigDecimal)obj;
    }

    @Override
    public void setBigDecimalAttribute(String name, java.math.BigDecimal val) throws MgmtException
    {
        setAttribute(name, val);
    }

    @Override
    public IMgmtBeanBase getReferenceAttribute(String name) throws MgmtException
    {
        Object obj = getAttribute(name);

        if(obj == null)
        {
            throw new AttributeNotFoundException("Reference attribute " + name + " is not defined.");
        }

        if(!(obj instanceof IMgmtBeanBase))
        {
            throw new AttributeNotFoundException("Attribute " + name + " is of type Reference");
        }

        return (IMgmtBeanBase)obj;
    }

    @Override
    public void setReferenceAttribute(String name, IMgmtBeanBase val) throws MgmtException
    {
        if(val == null)
        {
            setAttribute(name,(Object)null);
        }
        else
        {
            setAttribute(name, val.getConfigBean());
        }
    }

    protected IAttributeMap getRealAttributeMap() throws Exception
    {
        // if we have our own floating list then we can use that.
        if(m_map != null)
        {
            return m_map;
        }

        // otherwise we need to check if there is a floating list / map
        // in any of our parents ?
        IMgmtBase temp = m_parent;
        while(temp != null)
        {
            if(temp.getAttributeMap() != null)
            {
                return (IAttributeMap)temp.getAttributeMap().getAttribute(getConfigPath());
            }

            if(temp.getAttributeList() != null)
            {
                return (IAttributeMap)temp.getAttributeList().getAttribute(getConfigPath());
            }

            temp = temp.getParent();
        }
        return (IAttributeMap)m_bean.getAttribute(getConfigPath());
    }

    protected IAttributeList getRealAttributeList() throws Exception
    {
        // if we have our own floating list then we can use that.
        if(m_list != null)
        {
            return m_list;
        }

        // otherwise we need to check if there is a floating list / map
        // in any of our parents ?
        IMgmtBase temp = m_parent;
        while(temp != null)
        {
            if(temp.getAttributeMap() != null)
            {
                return (IAttributeList)temp.getAttributeMap().getAttribute(getConfigPath());
            }

            if(temp.getAttributeList() != null)
            {
                return (IAttributeList)temp.getAttributeList().getAttribute(getConfigPath());
            }

            temp = temp.getParent();
        }
        return (IAttributeList)m_bean.getAttribute(getConfigPath());
    }

    @Override
    public IMgmtAttributeMetaData getAttributeMetaData(String name)
    throws MgmtException
    {
        try
        {
            IAttributeMetaData ret = null;

            if(m_map != null)
            {
                ret = m_map.getAttributeMetaData(name);
            }
            else
            {
                ret = m_bean.getAttributeMetaData(getConfigPath(name));
            }

            return new MgmtAttributeMetaData(ret);
        }
        catch(Throwable e)
        {
            throw new MgmtException("Failed to get attribute metadata " + name +
                                    " - " + e.getMessage(), e);
        }
    }

}