/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.ma.plugin;

import java.text.Collator;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.StringTokenizer;

import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JTree;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.JWorkspacePanel;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.perms.ManagementSecurityUtils;
import com.sonicsw.ma.gui.table.IContentPane;
import com.sonicsw.ma.gui.table.MessageContentPane;
import com.sonicsw.ma.gui.util.BasicAction;
import com.sonicsw.ma.gui.util.FilterTreeNode;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.mx.config.IConfigServer;

import com.sonicsw.mf.common.config.IElementIdentity;

/**
 * Abstract implementation of IPlugin interface
 */
public abstract class AbstractGUIPlugin extends FilterTreeNode implements IPlugin
{
    protected IContentPane m_pluginContentPane = null;

    private PluginAttributes m_attributes;
    private IPluginContext   m_context;
    private IPluginFilter    m_filter;
    private String           m_name;
    private boolean          m_sorted = true;

    // Used for temporary plugins during the creation of new configurations
    private String m_dummyParentPath;

    private ManagementSecurityUtils.Readability readability;


    public AbstractGUIPlugin(IPluginContext context, String name)
    {
        m_context = context;
        m_name    = name;
    }

    // setParent is called when this tree node is added into its parent tree
    // node. At this point the plugin attributes should be parented.
    @Override
    public void setParent(MutableTreeNode newParent)
    {
        super.setParent(newParent);
        if (getAttributes().isChildBearing() && getChildCount() == 0)
        {
            // if this node doesn't already have a dummy child then we want
            // to add one
            add(new javax.swing.tree.DefaultMutableTreeNode(Boolean.TRUE));
        }
    }

    public final void setSorted(boolean sorted)
    {
        m_sorted = sorted;
    }

    protected int insert(final IPlugin childPlugin)
    {
        int returnIndex = 0;

        insertChild(childPlugin);

        returnIndex = -1;
        if (childPlugin instanceof FilterTreeNode && ((FilterTreeNode)childPlugin).canDisplay())
        {
            returnIndex = getVisibleChildIndex((TreeNode)childPlugin);
        }
        return returnIndex;
    }

    /**
     * Unlike the similar method insert(), this method does not return the 'visible child index'.
     * So it would perform slightly faster. If you do not care about the 'visible child index', use this method
     * @param childPlugin
     */
    protected void insertChild(final IPlugin childPlugin)
    {
        if(m_sorted)
        {
            final int nIndex = getInsertIndex(childPlugin.getPluginName());

        Helper.invoke(new Runnable()
        {
            @Override
            public void run()
            {
                insert((MutableTreeNode)childPlugin, nIndex);
            }
        }, true);
        }
        else
        {
            Helper.invoke(new Runnable()
        {
                @Override
                public void run()
                {
                    add((MutableTreeNode)childPlugin);
                }
            }, true);
        }

        if(childPlugin instanceof AbstractGUIPlugin)
        {
            Helper.invoke(new Runnable()
            {
                @Override
                public void run()
                {
                    ((AbstractGUIPlugin) childPlugin).postInsert();
                }
            });
        }
    }

    protected void postInsert()
    {
        // This check is to make sure that non readable folders or folder like nodes which
        // may have children cannot be expanded by the user.
        // Here, we limit the check to only those nodes who could actually bear children by definition
        // or those which are visible to the user
        final PluginAttributes attributes = getAttributes();
        boolean isChildBearing = attributes != null && attributes.isChildBearing();
        if (!isChildBearing ||  !canDisplay())
        {
            // no need to check! just go back
            return;
        }

        if (!getReadability().readable)
        {
            removeAllChildren();
            if (attributes != null)
            {
                attributes.set(PluginAttributes.IS_CHILD_BEARING, Boolean.FALSE);
            }
        }
    }

    public Map getMetaAttributes()
    {
        Map mapAttributes = null;

        try
        {
            IConfigServer server = getPluginContext().getConfigContext().
                getConfigServer();

            // check to see if a configuration with this name already exists
            mapAttributes = server.getMetaAttributes(getPluginPath());
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }

        return mapAttributes;
    }

    public TreeNode findChildNode(String strName)
    {
        TreeNode nodeResult = null;

        getPluginContext().getWorkspace().getSelectedWorkspacePanel().
            doExpansion(this, false, false, false);

        for (int i = 0; i < getChildCount(); i++)
        {
            TreeNode node = getChildAt(i);

            if (((DefaultMutableTreeNode)node).toString().equals(strName))
            {
                nodeResult = node;
                break;
            }
        }
        return nodeResult;
    }

    protected int addChildPlugin(IPlugin childPlugin, Map metaAttributes)
    {
        if (!isExpanded())
        {
            return -1;
        }

        IPluginFilter filter = getFilter();

        // If we have a filter specified then check to see if we are meant
        // to filter out this child
        if ((filter != null) && !filter.isPluginVisible(JTree.class, childPlugin))
        {
            return -1;
        }

        return insert(childPlugin);
    }

    /**
     * Sorts (if required) and sets all children at one shot.
     * @param childPlugins
     */
    protected void setChildPlugins(List<IPlugin> childPlugins)
    {
        // Note on implementation:
        // This method improves upon the performace of
        // calling addChildPlugin(final IPlugin childPlugin) one by one by avoiding the 'search for the right insert index'
        // on every insert.

        if (!isExpanded())
        {
            return;
        }

        IPluginFilter filter = getFilter();

        for (Iterator<IPlugin> iterator = childPlugins.iterator(); iterator.hasNext();)
        {

            IPlugin childPlugin =  iterator.next();
            if ((filter != null) && !filter.isPluginVisible(JTree.class, childPlugin))
            {
                // Similar behaviour to addChildPlugin(...) method
                iterator.remove();
            }
        }

        if (m_sorted)
        {
            Collections.sort(childPlugins, new Comparator<IPlugin>()
            {
                @Override
                public int compare(IPlugin o1, IPlugin o2)
                {
                    // Same comparison as used in FilterTreeNode.getInsertIndex(...)
                    return Collator.getInstance().compare(o1.getPluginName(), o2.getPluginName());
                }
            });
        }
        boolean sorted = m_sorted;
        // Slight trick to avoid repeated 'index searching' while inserting the child
        m_sorted = false;
        try
        {
            for (Iterator<IPlugin> iterator = childPlugins.iterator(); iterator.hasNext();)
            {
                insertChild(iterator.next());
            }
        }
        finally
        {
            m_sorted = sorted;
        }
    }


    protected IPlugin createChildPlugin(String strName, Map metaAttributes)
    {
        return null;
    }

    //-------------------------------------------------------------------------
    //
    // IPlugin implementation
    //
    //-------------------------------------------------------------------------

    @Override
    public String getPluginName()
    {
        return m_name;
    }

    @Override
    public void setPluginName(String name)
    {
        m_name = name;
    }

    /**
     * Used to set a dummy parent path...so that the plugin appears to have
     * been added in to the tree, hence properly parented.
     *
     * Relies on the temporary plugin not calling getParent() - this will fail
     *
     * @param dummyParentPath  The parent path that would normally be supplied
     *                         by the parent plugin in the tree.
     */
    public void setDummyParentPath(String dummyParentPath)
    {
        m_dummyParentPath = dummyParentPath;
    }

    @Override
    public String getPluginPath()
    {
        StringBuffer sb = new StringBuffer();

        if (m_dummyParentPath != null)
        {
            sb.append(m_dummyParentPath);

            if(!m_dummyParentPath.endsWith("/"))
            {
                sb.append("/");
            }

            sb.append(getPluginName());
        }
        else
        {
            AbstractGUIPlugin pluginParent = (AbstractGUIPlugin)getParent();

            if (pluginParent != null)
            {
                String path = pluginParent.getPluginPath();
                sb.append(path);
            }

            if (!sb.toString().endsWith("/"))
            {
                sb.append("/");
            }

            if (isRoot() == false)
            {
                sb.append(getPluginName());
            }
        }

        return sb.toString();
    }

    public String _getPluginName()
    {
        return getPluginName();
    }

    public String _getPluginPath()
    {
        return getPluginPath();
    }

    @Override
    public final PluginAttributes getAttributes()
    {
        return m_attributes;
    }

    @Override
    public void dispose()
    {
        m_context = null;
        m_attributes = null;
        m_pluginContentPane = null;
    }

    @Override
    public boolean equals(Object object)
    {
        if (object == this)
        {
            return true;
        }

        if (!(object instanceof IPlugin))
        {
            return false;
        }

        IPlugin plugin = (IPlugin)object;

        // Are only the same if they are both visible...
        //
        if ((plugin instanceof FilterTreeNode) &&
            (((FilterTreeNode)plugin).canDisplay() != canDisplay()))
        {
            return false;
        }

        // Are the plugins the same, i.e. do they have the same path?
        if (!plugin.getPluginPath().equals(this.getPluginPath()))
        {
            return false;
        }

        // Plugins can only be equal if they are in the same domain...
        if (!plugin.getPluginContext().getConnectionInfo().equals(this.
            getPluginContext().getConnectionInfo()))
        {
            return false;
        }

        return true;
    }
    
    @Override
    public int hashCode() {
        return Objects.hash(canDisplay(), getPluginPath(), getPluginContext().getConnectionInfo()); 
    }

    @Override
    public IPluginFilter getFilter()
    {
        if (m_filter != null)
        {
            return m_filter;
        }

        TreeNode parent = getParent();

        if ((parent != null) && (parent instanceof AbstractGUIPlugin))
        {
            return ((AbstractGUIPlugin)parent).getFilter();
        }

        return null;
    }

    @Override
    public void setFilter(IPluginFilter filter)
    {
        m_filter = filter;
    }

    @Override
    public boolean isFilterLocal()
    {
        return (m_filter != null);
    }

    @Override
    public String toString()
    {
        return getPluginName();
    }

    //-------------------------------------------------------------------------

    public abstract boolean expand();

    public String getToolTipText()
    {
        List list = getToolTipList();

        if (list == null || list.isEmpty())
        {
            return null;
        }

        StringBuffer sb = new StringBuffer();

        sb.append(Helper.TIP_PROLOG);

        for (int i = 0; i < list.size(); i++)
        {
            sb.append(list.get(i).toString());
        }

            // In DEVELOPER_MODE lets add in all the attributes
        if (MgmtConsole.DEVELOPER_MODE)
        {
            PluginAttributes attributes = m_attributes;

            while (attributes != null)
            {
                sb.append("<tr></tr>");

                for (Iterator iter = attributes.keySet(false).iterator(); iter.hasNext(); )
                {
                    String key = (String)iter.next();
                    sb.append(formatToolTipText(key, attributes.toString(attributes.get(key))));
                }
                attributes = attributes.getParent();
            }
        }
        sb.append(Helper.TIP_EPILOG);

        return sb.toString();
    }

    protected String formatToolTipText(String key, String value)
    {
        return Helper.TABLE_LEFT + key + ":" + Helper.TABLE_MID + value + Helper.TABLE_RIGHT;
    }

    @Override
    public List getToolTipList()
    {
        ArrayList list = new ArrayList();

        list.add(formatToolTipText("Name", getPluginName()));
        list.add(formatToolTipText("Type", getAttributes().getDisplayType()));

        String cVersion = getAttributes().getConfigVersion();
        String pVersion = getAttributes().getProductVersion();

        if (pVersion != null)
        {
            list.add(formatToolTipText("Product Version", pVersion));
        }

        if (cVersion != null)
        {
            list.add(formatToolTipText("Config Version", cVersion));
        }

        return list;
    }

    public final void setAttributes(PluginAttributes attributes)
    {
        m_attributes = attributes;
    }

    /**
     * Subclass should override
     */
    @Override
    public AbstractButton[] getToolbarItems()
    {
        return new AbstractButton[0];
    }

    /**
     * Get the Plugin Context
     */
    @Override
    public IPluginContext getPluginContext()
    {
        return m_context;
    }

    /**
     * There is no default content pane
     */
    @Override
    public IContentPane getPluginContentPane()
    {
        if (m_pluginContentPane == null)
        {
            try
            {
                m_pluginContentPane = createPluginContentPane();
            }
            catch (Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR, "Unable to create content pane.", e, true);
            }
        }
        return m_pluginContentPane;
    }

    public boolean isExpanded()
    {
        final DefaultMutableTreeNode flag = (getChildCount() > 0) ?
            (DefaultMutableTreeNode)getFirstChild() : null;

        if ((flag != null) && (flag.getUserObject()instanceof Boolean))
        {
            return false;
        }

        return true;
    }

    @Override
    public void refresh()
    {
        if (isExpanded())
        {
            JWorkspacePanel workspace = getWorkspacePanel();

            workspace.doCollapse(this);
            workspace.doExpansion(this, true, false, false);
        }
        else
        {
            getTreeModel().nodeChanged(this);
        }
        getWorkspacePanel().refreshContentPaneIfShown(this);
    }

    public boolean pathExists(String strPath)
    {
        // check to see if a configuration with this name already exists
        return getPluginContext().getConfigContext().getConfigServer().pathExists(strPath);
    }

    @Override
    public void onAdded(String name, Map attributes)
    {
        try
        {
            // Create the child plugin - this can return null
            IPlugin child = createChildPlugin(null, attributes);

            int nIndex = addChildPlugin(child, attributes);

            // If we have a child plugin and it succesfully adds to the parent
            // then continue to process
            if (child != null)
            {
                if (nIndex != -1)
                {
                    // Fire a change event to get the child into the tree
                    getTreeModel().nodesWereInserted(this, new int[] {nIndex});
                }

                // If the content pane is currently visible then we want
                // to add the child to the content pane
                IContentPane cp = getContentPaneIfShown(this);

                if (cp != null)
                {
                    if (cp instanceof IPluginChangeListener)
                    {
                        ((IPluginChangeListener)cp).onAdded(child);
                    }
                    else
                    {
                        cp.refresh();
                    }
                }
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }

    @Override
    public void onDeleted(IPlugin child)
    {
        try
        {
            AbstractGUIPlugin aChild = (AbstractGUIPlugin)child;

            // Delete from the table first to avoid problems with the plugin
            // not having a parent
            IContentPane cp = getContentPaneIfShown(this);

            if (cp != null)
            {
                if (cp instanceof IPluginChangeListener)
                {
                    ((IPluginChangeListener)cp).onDeleted(child);
                }
                else
                {
                    cp.refresh();
                }
            }

            boolean selected = getTree().isPathSelected(new TreePath(aChild.getPath()));

            // first check that the child being deleted actually exists in
            // the tree
            int index = getIndex(aChild);
            boolean bDisplayedNode = aChild instanceof FilterTreeNode == false ||
                (aChild instanceof FilterTreeNode && ((FilterTreeNode)aChild).canDisplay());

            if (bDisplayedNode)
            {
                // change the index if displayed in a filtered tree...
                index = getVisibleChildIndex(aChild);
            }

            // Remove the child node
            remove(aChild);

            if (index != -1 && bDisplayedNode)
            {
                // Fire off a tree update for the deleted child
                getTreeModel().nodesWereRemoved(this,
                    new int[] {index}, new Object[] {child});

                // if the tree node was selected then we need to move the selection
                // around
                if (selected)
                {
                    DefaultMutableTreeNode newNode = this;

                    if (index > 0)
                    {
                        // JSP: possible problem here????
                        newNode = (DefaultMutableTreeNode)getChildAt(index - 1);
                    }

                    getTree().setSelectionPath(new TreePath(newNode.getPath()));
                }
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }

    @Override
    public void onUpdated()
    {
        try
        {
            // Fire a change event to update the plugin tree node
            getTreeModel().nodeChanged(this);

            // We first need to check to see if this plugin is displaying
            IContentPane cp = getContentPaneIfShown(this);

            if (cp != null)
            {
                if (cp instanceof IPluginChangeListener)
                {
                    ((IPluginChangeListener)cp).onUpdated(this);
                }
                else
                {
                    cp.refresh();
                }
            }
            else
            {
                AbstractGUIPlugin parent = (AbstractGUIPlugin)getParent();

                // If our parents content pane is currently visible then we want
                // to update the plugin in the content pane
                if (parent != null)
                {
                    cp = getContentPaneIfShown(parent);

                    if (cp != null)
                    {
                        if (cp instanceof IPluginChangeListener)
                        {
                            ((IPluginChangeListener)cp).onUpdated(this);
                        }
                        else
                        {
                            cp.refresh();
                        }
                    }
                }
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }

    @Override
    public void onRenamed(String newPath)
    {
        // Need to check that the new name is in the same folder as the old
        // name. If it is we can just rename the plugin.
        // If it isn't then we need to delete the plugin from its current
        // position and insert in its new position

        String oldPath = getPluginPath();
        String oldBase = oldPath.substring(0, oldPath.lastIndexOf('/'));
        String newBase = newPath.substring(0, newPath.lastIndexOf('/'));

        AbstractGUIPlugin parent = (AbstractGUIPlugin)getParent();
        String newName  = newPath.substring(newPath.lastIndexOf('/')+1);

        if (oldBase.equals(newBase))
        {
            // NOTE WELL: This is sufficient for renaming, but we still need to figure
            // out how to get the parents child list sorted afterwards... - jsp
            setPluginName(newName);

            // Fire a change event to update the plugin tree node
            getTreeModel().nodeChanged(this);

            // If our parents content pane is currently visible then we want
            // to update the plugin in the content pane
            if (parent != null)
            {
                IContentPane cp = getContentPaneIfShown(parent);

                if (cp != null)
                {
                    if (cp instanceof IPluginChangeListener)
                    {
                        ((IPluginChangeListener)cp).onRenamed(this);
                    }
                    else
                    {
                        cp.refresh();
                    }
                }
            }
        }
        else
        {
            // We must find the new parent plugin for the "newly" renamed plugin
            // and add it in there...
            //
            AbstractGUIPlugin newParent = (AbstractGUIPlugin)((AbstractGUIPlugin)getRoot()).goToPlugin(newBase);

            if (newParent != null)
            {
                int newIndex = newParent.getInsertIndex(newName);

                getTreeModel().removeNodeFromParent(this);
                setPluginName(newName);
                getTreeModel().insertNodeInto(this, newParent, newIndex);

                if (newParent != null)
                {
                    IContentPane cp = getContentPaneIfShown(newParent);

                    if (cp != null)
                    {
                        if (cp instanceof IPluginChangeListener)
                        {
                            ((IPluginChangeListener)cp).onRenamed(this);
                        }
                        else
                        {
                            cp.refresh();
                        }
                    }
                }
            }
        }
    }

    //-------------------------------------------------------------------------

    public IPlugin goToPlugin(String path)
    {
        IPlugin         node   = this;
        StringTokenizer tokens = new StringTokenizer(path, "/");

        while (tokens.hasMoreTokens())
        {
            if  ((node = (IPlugin)((AbstractGUIPlugin)node).findChildNode(tokens.nextToken())) == null)
            {
                break;
            }
        }

        if (node != null)
        {
            TreePath pathToNode = new TreePath(((DefaultMutableTreeNode)node).getPath());

            getTree().expandPath(pathToNode);
            getTree().invalidate();
        }
        else
        {
            Helper.logErrorMessage("Failed to find configuration node with path '" + path + "'");
        }

        return (IPlugin)node;
    }

    /**
     * By default :), the default action should return the properties menu item
     * action (if set).
     *
     * @return  The default action (usually the properties action)
     */
    @Override
    public BasicAction getDefaultAction()
    {
        JComponent[] c = getPropertiesMenuItems();

        if ((c != null) && (c.length > 0))
        {
            if (c[c.length-1] instanceof AbstractButton)
            {
                Action action = ((AbstractButton)c[c.length-1]).getAction();

                if (action instanceof BasicAction)
                {
                    return (BasicAction)action;
                }
            }
        }

        return null;
    }

    @Override
    public JComponent[] getMenuItems(int type)
    {
        JComponent[] ret = null;

        if (type == PLUGIN_TYPE)
        {
            ret = getPluginMenuItems();
        }
        else if (type == EDIT_TYPE)
        {
            ret = getEditMenuItems();
        }
        else if (type == VIEW_TYPE)
        {
            ret = getViewMenuItems();
        }
        else if (type == SYSTEM_TYPE)
        {
            ret = getSystemMenuItems();
        }
        else if (type == PROPERTIES_TYPE)
        {
            ret = getPropertiesMenuItems();
        }
        else if (type == NEW_TYPE)
        {
            ret = getNewMenuItems();
        }
        else if (type == PERMISSIONS_TYPE){
            return getPermissionsMenuItems();
        }


        return ret;
    }


    protected IContentPane createPluginContentPane()
    {
        if (!getReadability().readable) {
            return new MessageContentPane(this, readability.message);
        }
        return createReadablePluginContentPane();
    }

    /**
     * To be overriden by those plugins whose read permissions could be denied.
     *
     * @return
     */
    protected IContentPane createReadablePluginContentPane() {
        return null;
    }

    protected ManagementSecurityUtils.Readability getReadability() {
        if(readability!=null)
        {
            return readability;
        }
        if(isReadDeniable()){
            return readability = ManagementSecurityUtils.getReadability(this);
        }
        return readability = ManagementSecurityUtils.Readability.READABLE;
    }

    protected boolean isReadDeniable() {
        return ManagementSecurityUtils.allowsConfigPerms(this);
    }

    protected JComponent[] getPluginMenuItems()
    {
        return null;
    }

    protected JComponent[] getNewMenuItems()
    {
        return null;
    }

    protected JComponent[] getPropertiesMenuItems()
    {
        return null;
    }

    protected JComponent[] getEditMenuItems()
    {
        return null;
    }

    protected JComponent[] getSystemMenuItems()
    {
        return null;
    }

    protected JComponent[] getViewMenuItems()
    {
        return null;
    }

    protected JComponent[] getPermissionsMenuItems() {
        return ManagementSecurityUtils.getPermissionsMenuItems(this);

    }


    //-------------------------------------------------------------------------

    /**
     * This method is called when the user has entered a new name for this
     * plugin. The plugin receiving this call will perform any additional
     * operations prior to calling into the config server to rename the
     * config element
     * @param newName the fully qualified new name for this plugin
     */
    @Override
    public void renamePlugin(IConfigServer configServer, String newPath)
        throws Exception
    {
        MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
            "Rename plugin not yet implemented.", true);
    }

    /**
     * This method is called when the plugin (and any data representing it)
     * is to be deleted. The implementation is responsible for removing any
     * child plugin's and associated data.
     *
     * The plugin(s) are removed from the tree/content pane asynchronously
     * by notification from the Directory Service.
     *
     * @param server      (transaction) config server to use for the delete
     * @throws Exception  if for some reason the operation cannot be performed
     *                    or completed then an exception will be thrown
     */
    @Override
    public void deletePlugin(IConfigServer configServer)
        throws Exception
    {
        MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
            "Delete plugin not yet implemented.", true);
    }

    /**
     * This method is called when the plugin (and any data representing it)
     * is to be copied. The implementation is responsible for creating the
     * new plugin and associated data.
     *
     * In most cases, it is the data that is the underlying data that needs
     * to be created...and this creation kicks off notification events that
     * result in plugins being added to the appropriate views.
     *
     * @param server      (transaction) config server to use for the copy
     * @param path        the new location of the plugin copy
     * @param copyData    implementation-specific data needed to control
     *                    the copy operation, e.g. instance, template copies
     * @throws Exception  if for some reason the operation cannot be performed
     *                    or completed then an exception will be thrown
     */
    @Override
    public void copyPlugin(IConfigServer server, String path, Object copyData)
        throws Exception
    {
        MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
            "Copy plugin not yet implemented.", true);
    }

    //-------------------------------------------------------------------------
    //
    // IMonitorPlugin implementation
    //
    //-------------------------------------------------------------------------

    public String getMonitorPath()
    {
        return getPluginPath();
    }

    @Override
    public String[] getAllSupportedMonitors()
    {
        return new String[0];
    }

    @Override
    public boolean isMonitorSupported(String monitorId)
    {
        return false;
    }

    @Override
    public Object getMonitorModel(String monitorId)
    {
        return null;
    }

    protected abstract JWorkspacePanel getWorkspacePanel();

    protected JTree getTree()
    {
        return getWorkspacePanel().getTree();
    }

    protected DefaultTreeModel getTreeModel()
    {
        return (DefaultTreeModel)getWorkspacePanel().getTree().getModel();
    }

    protected IContentPane getContentPaneIfShown(IPlugin plugin)
    {
        return getWorkspacePanel().getContentPaneIfShown(plugin);
    }

    //-------------------------------------------------------------------------

    protected IElementIdentity createElementIdentity(String path, String type)
    {
        return new DummyElementIdentity(path, type);
    }

    class DummyElementIdentity
        implements IElementIdentity
    {
        private String m_name;
        private String m_type;

        public DummyElementIdentity(String name, String type)
        {
            m_name = name;
            m_type = type;
        }

        @Override
        public long getCreationTimestamp()
        {
            return System.currentTimeMillis();
        }

        @Override
        public long getVersion()
        {
            return INITIAL_VERSION;
        }

        @Override
        public String getType()
        {
            return m_type;
        }

        @Override
        public String getReleaseVersion()
        {
            return null;
        }

        @Override
        public boolean equalVersion(IElementIdentity elementID)
        {
            return false;
        }

        @Override
        public boolean equalEntity(IElementIdentity elementID)
        {
            return false;
        }

        @Override
        public String[] getNameComponents()
        {
            StringTokenizer st = new StringTokenizer(m_name, "/");
            String[] res = new String[st.countTokens()];
            int i = 0;

            while (st.hasMoreTokens())
            {
                res[i++] = st.nextToken();
            }

            return res;
        }

        @Override
        public String getName()
        {
            return m_name;
        }
    }


}