/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.ma.plugin;

import java.util.Map;

import com.sonicsw.ma.gui.util.Helper;

import com.sonicsw.mf.common.dirconfig.IDirElement;

public abstract class AbstractPluginFactory implements IPluginFactory
{
    private PluginAttributes  m_attributes;
    private IPluginFactory[]  m_childFactories;

    public AbstractPluginFactory()
    {
    }

    @Override
    public final PluginAttributes getAttributes()
    {
        return m_attributes;
    }

    public final void setAttributes(PluginAttributes attributes)
    {
        m_attributes = attributes;
    }

    /**
     * Called when the factory is being created to pass it an additional
     * configuration settings from the DS
     * @param element
     * @throws java.lang.Exception
     */
    @Override
    public void init(IDirElement element) throws Exception
    {
    }

    @Override
    public final IPluginFactory[] getChildFactories()
    {
        if(m_childFactories == null)
        {
            m_childFactories = createChildFactories();

            if(m_childFactories != null)
            {
                // In each of the child factories we need to set the same list
                // of supported PRODUCT_VERSIONS as we have in this factory
                for(int i = 0; i < m_childFactories.length; i++)
                {
                    m_childFactories[i].getAttributes().set(PluginAttributes.PRODUCT_VERSIONS,
                        getAttributes().get(PluginAttributes.PRODUCT_VERSIONS));
                }
            }
        }
        return m_childFactories;
    }

    protected IPluginFactory[] createChildFactories()
    {
        return new IPluginFactory[0];
    }

    public IPlugin newInstance(IPluginContext context, String id, Map data)
    throws Exception
    {
        return null;
    }

    /**
     * Creates a plugin.
     *
     * @param context  The context provides the plugin access to resources it
     *                 needs to function within the framework
     * @param id       A string that uniquely identifies the plugin
     * @param pVersion A string that specifies the product version for the new
     *                 plugin. A PluginFactory can potentially create many
     *                 product versions of configurations, all with the same
     *                 configuration version. If the pVersion is null then
     *                 the factory will create the a plugin for the first
     *                 product version specified in the supported version list.
     * @param data     A map of additional creation data
     * @return         A plugin instance
     */
    @Override
    public IPlugin createPlugin(IPluginContext context, String id, String pVersion, Map data)
    throws Exception
    {
        String[] versions = getAttributes().getProductVersions();

        if(pVersion == null)
        {
            // We don't know what product version to create, so create the one with
            // the first version number supported by this factory. Factories should
            // always list their supported versions with the default version
            // first
            pVersion = versions[0];

            Helper.logDebugMessage("AbstractPluginFactory::createPlugin() " + getClass().getName() + " no version specified for " + id + " - use " + pVersion);
        }
        else
        {
            if (versions == null || versions.length == 0)
            {
                throw new Exception("AbstractPluginFactory::createPlugin(): " +
                                    getClass().getName() +
                                    " - no product versions defined");
            }

            // check to see if the product version specified is one that is supported
            // by this plugin factory
            boolean found = false;
            for (int i = 0; i < versions.length; i++)
            {
                if (versions[i].equals(pVersion))
                {
                    found = true;
                    break;
                }
            }

            if (!found)
            {
                throw new Exception("AbstractPluginFactory::createPlugin(): " +
                                    getClass().getName() +
                                    " - product version " + pVersion +
                                    " not supported");
            }
        }

        // Create the required plugin
        IPlugin plugin = newInstance(context, id, data);

        if (plugin != null)
        {
            // Put the plugin product version into the plugin
            plugin.getAttributes().set(PluginAttributes.PRODUCT_VERSION, pVersion);

            // Set this factory as its attribute parent...
            plugin.getAttributes().setParent(getAttributes());
        }
        return plugin;
    }
}