/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.ma.plugin;

import java.util.HashMap;

import javax.swing.JInternalFrame;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.IConfigurableTool;
import com.sonicsw.ma.gui.IConfigurableWorkspaceTool;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.domain.DomainConnectionModel;

/**
 * <p>Title: ClasspathToolInfo</p>
 * <p>Description: Creates a TooInfo for Classpath based internally launched java code.</p>
 * <p>Copyright: Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.</p>
 * <p>Company: Sonic Software Corporation</p>
 * @author Jeffrey S. Pace
 * @version 1.0
 */
public class ClasspathToolInfo extends ToolInfo
{
    protected String m_classname;
    protected ClassLoader m_classloader;

    /**
     * Creates a ClassPathToolInfo
     * @param name - name of the tool
     * @param description - description of the tool.
     * @param classname - class name of the tool to launch.
     */
    public ClasspathToolInfo(String name, String description, String classname)
    {
        this(null, name, description, classname, null, false);
    }

    /**
     * Creates a ClassPathToolInfo
     * @param context
     * @param name - name of the tool
     * @param description - description of the tool.
     * @param classname - class name of the tool to launch.
     * @param classpath - classpath to reference for the tool.
     * @param configure
     */
    public ClasspathToolInfo(IPluginContext context,
                             String name,
                             String description,
                             String classname,
                             ClassLoader classloader,
                             boolean configure)
    {
        m_classname   = classname;
        m_classloader = classloader;

        commonInit(context, name, description, configure);
    }

    /**
     * Determines when this tool can be enabled.
     * @return boolean
     */
    @Override
    public boolean canEnable()
    {
        boolean res = false;

        try
        {
            ClassLoader cl  = m_classloader != null ? m_classloader : getClass().getClassLoader();

            Class.forName(m_classname, false, cl);

            res = true;
        }
        catch (Exception e)
        {
        }
        return res;
    }

    /**
     * Execution of the Tool.
     */
    @Override
    public void execute()
    {
        Thread t = new Thread(new Runnable()
        {
            @Override
            public void run()
            {
                try
                {
                    ClassLoader cl      = m_classloader != null ? m_classloader : getClass().getClassLoader();
                    Class       cls     = Class.forName(m_classname, false, cl);
                    Class       toolCls = Class.forName("com.sonicsw.ma.gui.IConfigurableTool", false, cl);

                    if(toolCls.isAssignableFrom(cls))
                    {
                        // This is an IConfigurableTool interface
                        IConfigurableTool obj = (IConfigurableTool)cls.newInstance();
                        JInternalFrame    frame;

                        if(m_configure && m_context != null)
                        {
                            frame = obj.create(buildConfiguration(m_context.getConnectionInfo()));
                        }
                        else
                        {
                            frame = obj.create();
                        }

                        if(frame.isVisible())
                        {
                            if(frame.isIcon())
                            {
                                frame.setIcon(false);
                            }

                            frame.moveToFront();
                        }
                        else
                        {
                            MgmtConsole.getMgmtConsole().showInternalFrame(frame, true);
                        }
                    }
                    else
                    {
                        // This is a class we want to create
                        Object obj = cls.newInstance();

                        if(obj instanceof JInternalFrame)
                        {
                            MgmtConsole.getMgmtConsole().showInternalFrame((JInternalFrame)obj, true);
                        }
                    }
                }
                catch (Exception e)
                {
                    MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                                         "Failed to launch tool " + m_classname, e, true);
                }
            }
        });

        t.start();
    }

    protected HashMap buildConfiguration(DomainConnectionModel model)
    {
        HashMap map = new HashMap();

        String str;

        map.put("DOMAIN", model.getDomainName());
        map.put("URL", model.getUrl());
        str = model.getUserName();
        if(str != null && str.length() > 0)
        {
            map.put("USER", str);
        }
        str = model.getPassword();
        if(str != null && str.length() > 0)
        {
            map.put("PASSWORD", str);
        }
        map.put("TIMEOUT", new Integer(model.getTimeout()));
        map.put("USE_DRA", new Boolean(model.isUseDRA()));
        str = model.getManagementNode();
        if(str != null && str.length() > 0)
        {
            map.put("MANAGEMENT_NODE", str);
        }
        map.put("LOAD_BALANCING", new Boolean(model.isLoadBalancing()));

        return map;
    }
    
    @Override
    public String toString()
    {
        StringBuffer sb = new StringBuffer();
        sb.append(getClass().getName());
        sb.append(" [");
        sb.append("classname: ").append(m_classname);
        sb.append(" ]");
        
        return sb.toString();
    }

    public boolean isVisibleInMenu()
    {
        if (!canEnable())
        {
            return false;
        }
        
        try
        {
            ClassLoader cl      = m_classloader != null ? m_classloader : getClass().getClassLoader();
            Class       cls     = Class.forName(m_classname, false, cl);
              
            // If its a Workspace-enabled Configurable Tool then we don't want to execute it as a standalone tool
            if (IConfigurableWorkspaceTool.class.isAssignableFrom(cls))
            {
                return false;
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        
        return true;
    }
    
    public Class getRootPluginClass()
    {
        if (!canEnable())
        {
            return null;
        }
        
        try
        {
            ClassLoader cl      = m_classloader != null ? m_classloader : getClass().getClassLoader();
            Class       cls     = Class.forName(m_classname, false, cl);
              
            if (IConfigurableWorkspaceTool.class.isAssignableFrom(cls))
            {
                IConfigurableWorkspaceTool cwTool = (IConfigurableWorkspaceTool)cls.newInstance();
                
                return cwTool.getConfigRootNodeClass();
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        
        return null;
    }
}
