/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.ma.plugin;

import java.util.HashMap;

import com.sonicsw.mf.common.dirconfig.IDirElement;

/**
 * This class represents information about a ConfigPluginFactory tree.
 * It is initialized from a root plugin factory, and contains a hashmap
 * of all of the children plugins and a default menu structure that
 * represents the menu tree in hierarchical fashion
 */
public class ConfigFactoryInfo
{
    private ClassLoader          m_classLoader;
    private IConfigPluginFactory m_rootFactory;
    private HashMap              m_childFactories = new HashMap();

    public ConfigFactoryInfo(String className, ClassLoader classLoader, IDirElement element)
        throws Exception
    {
        m_classLoader = classLoader;
        m_rootFactory = (IConfigPluginFactory)Class.forName(className,
                                                            true,
                                                            m_classLoader).newInstance();

        m_rootFactory.init(element);

        recurseChildren(m_rootFactory);
    }

    public IConfigPluginFactory getRootFactory()
    {
        return m_rootFactory;
    }

    public HashMap getChildFactoryMap()
    {
        return m_childFactories;
    }

    /**
     * Get child factory given a type and version
     */
    public IConfigPluginFactory getChildFactory(String type, String cVersion, String pVersion)
    {
        // Check the root factory to see if this is the correct version
        // All child factories have the same config version as the root.
        // If no config version is specified then we are just going to return
        // the first thing we find !!!
        if(cVersion == null || cVersion.equals(m_rootFactory.getAttributes().getConfigVersion()))
        {
            IConfigPluginFactory factory = (IConfigPluginFactory)m_childFactories.get(type);

            if(pVersion == null)
            {
                return factory;
            }

            if(factory != null)
            {
                // Check that this factory supports the specified product version
                String[] versions = factory.getAttributes().getProductVersions();
                for(int i = 0; i < versions.length; i++)
                {
                    if(versions[i].equals(pVersion))
                    {
                        return factory;
                    }
                }
            }
        }
        return null;
    }

    /**
     * Recurse the children building up the list of child factories
     */
    private void recurseChildren(IConfigPluginFactory factory)
    {
        IPluginFactory[] children = factory.getChildFactories();

        if (children == null)
        {
            return;
        }

        for (int i = 0; i < children.length; i++)
        {
            IConfigPluginFactory child      = (IConfigPluginFactory)children[i];
            PluginAttributes     attributes = child.getAttributes();

            if(!m_childFactories.containsKey(attributes.getType()))
            {
                m_childFactories.put(attributes.getType(), child);
            }

            recurseChildren(child);
        }
    }
    
    @Override
    public String toString()
    {
        StringBuffer sb = new StringBuffer();
        
        IConfigPluginFactory rootFactory = getRootFactory();
        PluginAttributes attributes = rootFactory.getAttributes();
        
        sb.append("ConfigFactoryInfo (");
        sb.append("type=").append(attributes.getType());
        sb.append(", cV=").append(attributes.getConfigVersion());
        sb.append(", pV=").append(attributes.getProductVersion());
        sb.append(", label=").append(attributes.getDisplayType());
        sb.append(", family=").append(attributes.getProductFamilyId());
        sb.append(")");
        
        return sb.toString();
    }
}
