package com.sonicsw.ma.plugin;

/**
 * Defines the interface for supporting a monitoring plugin.
 *
 * It really provides generic access to arbitrary "read-only" data.
 * There is no way to set a monitor model back into the supporting plugin,
 * i.e. only a getter is implemented.
 */
public interface IMonitorPlugin
{
    public static final String MONITOR_ID_METRICS       = "_Metrics";
    public static final String MONITOR_ID_METRICS_CONFIG       = "_MetricsConfig";
    public static final String MONITOR_ID_NOTIFICATIONS = "_Notifications";

    /**
     * Returns the name of the plugin denoted by its pathname.
     * This is just the last name in the pathname's name sequence.
     * If the pathname's name sequence is empty, then an empty string is returned.
     *
     * @return  The name of the plugin denoted by this abstract pathname, or
     *          the empty string if this pathname's name sequence is empty
     */
    public String getPluginName();

    /**
     * Converts this abstract pathname into a pathname string. The resulting
     * string uses the '\' ame-separator character to separate the names in
     * the name sequence.
     *
     * @return  The string form of this abstract pathname
     */
    public String getPluginPath();

    /**
     * Returns a list of all supported monitors in the form of an array of
     * unique string identifiers.
     *
     * @return  A string array of unique monitor id's or a zero-length array
     *          if no monitors are supported.
     */
    public String[] getAllSupportedMonitors();

    /**
     * Returns a boolean indicating if the monitor "monitorId" is supported
     * by this plugin. An easy way to determine support for a given monitor.
     *
     * @param monitorId  A string that uniquely identifies a monitor
     * @return           true if the monitor is supported otherwise false
     */
    public boolean isMonitorSupported(String monitorId);

    /**
     * Used to fetch the monitor model. This model can be anything at all -
     * it is up to the monitor handler to interpret.
     *
     * If the monitor "monitorId" is not supported then this method will
     * return null.
     *
     * @param monitorId  A string that uniquely identifies a monitor
     * @return           The monitor model object or null if the monitor is
     *                   not supported.
     */
    public Object getMonitorModel(String monitorId);
}
