package com.sonicsw.ma.plugin;

import com.sonicsw.mx.config.ConfigServerUtility;

import com.sonicsw.mf.common.runtime.IState;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

public class PluginAttributes
{
    public static final String TYPE                 = ConfigServerUtility.TYPE;
    public static final String CONFIG_VERSION       = ConfigServerUtility.CONFIG_VERSION;
    public static final String PRODUCT_VERSION      = ConfigServerUtility.PRODUCT_VERSION;
    public static final String PRODUCT_VERSIONS     = "PRODUCT_VERSIONS"; // String[]
    public static final String DISPLAY_TYPE         = "DISPLAY_TYPE";
    public static final String PLUGIN_REVISION      = "PLUGIN_REVISION";
    public static final String IS_TEMPLATABLE       = "TEMPLATABLE";     // Boolean   was IS_REUSABLE in v5
    public static final String IS_CREATABLE         = "CREATABLE";       // Boolean
    public static final String IS_SYSTEM            = "SYSTEM";          // String
    public static final String IS_CHILD_BEARING     = "CHILD_BEARING";   // Boolean
    public static final String IS_COMPONENT         = "COMPONENT";       // Boolean
    public static final String IS_RELOADABLE        = "RELOADABLE";      // Boolean
    public static final String STATE                = "STATE";           // IState
    public static final String IS_STATEABLE         = "STATEABLE";       // Boolean
    public static final String IS_RUNTIME_ROOT      = "RUNTIME_ROOT";    // Boolean
    public static final String MBEAN_INFO           = "MBEAN_INFO";      // MBeanInfo
    public static final String METRIC_INFO          = "METRIC_INFO";     // IMetricInfo[]
    public static final String IS_DYNAMIC_RUNTIME   = "DYNAMIC_RUNTIME"; // Boolean
    public static final String IS_UPGRADEABLE       = "UPGRADEABLE";     // Boolean
    public static final String PRODUCT_FAMILY_ID    = "PRODUCT_FAMILY_ID";  // String
    public static final String PRIMARY_CONFIGURATION = "PRIMARY_CONFIGURATION"; // Boolean

    private PluginAttributes m_parent;
    private HashMap m_map;

    public PluginAttributes()
    {
        this(null);
    }

    public PluginAttributes(PluginAttributes parent)
    {
        this(parent, new HashMap());
    }

    public PluginAttributes(PluginAttributes parent, Object[][] attributes)
    {
        this(parent);

        put(attributes);
    }

    public PluginAttributes(PluginAttributes parent, HashMap attributes)
    {
        m_parent = parent;
        m_map    = attributes;
    }

    public Object get(String key)
    {
        Object res = hasParent() ? getParent().get(key) : null;

        if (m_map.containsKey(key))
        {
            res = m_map.get(key);
        }

        return res;
    }

    public void set(String key, Object value)
    {
        m_map.put(key, value);
    }

    public Set keySet()
    {
        return keySet(true);
    }

    public Set keySet(boolean inherit)
    {
        Set res = null;

        if (inherit && hasParent())
        {
            res = new HashSet(getParent().keySet(inherit));

            res.addAll(m_map.keySet());
        }
        else
        {
            res = m_map.keySet();
        }

        return res;
    }

    public boolean containsKey(String key)
    {
        return keySet().contains(key);
    }

    public int size()
    {
        return keySet().size();
    }

    public boolean isEmpty()
    {
        return keySet().isEmpty();
    }

    public final void put(PluginAttributes attributes)
    {
        if (attributes != null)
        {
            Iterator i = attributes.keySet().iterator();

            while (i.hasNext())
            {
                String key = (String)i.next();

                set(key, attributes.get(key));
            }
        }
    }

    public final void put(Map attributes)
    {
        if (attributes != null)
        {
            m_map.putAll(attributes);
        }
    }

    public final void put(Map attributes, String[] excludeKeys)
    {
        put(m_map, attributes, excludeKeys);
    }

    /**
     * Takes the key value pairs from the source Map and adds (puts) them
     * into the destination Map...excluding any pairs from the source Map that
     * match the excludeKeys list.
     *
     * @param dst          The destination Map where the pairs are to be set
     * @param src          The source of the new key value pairs
     * @param excludeKeys  A list of keys to exclude when merging both Maps
     * @return             The updated destination Map
     */
    public static Map put(Map dst, Map src, String[] excludeKeys)
    {
        if ((dst != null) && (src != null))
        {
            Iterator i = src.keySet().iterator();

            while (i.hasNext())
            {
                String  key     = (String)i.next();
                boolean exclude = false;

                if (excludeKeys != null)
                {
                    for (int j = 0; j < excludeKeys.length; j++)
                    {
                        if (excludeKeys[j] == key)
                        {
                            exclude = true;
                            break;
                        }
                    }
                }

                if (!exclude)
                {
                    dst.put(key, src.get(key));
                }
            }
        }

        return dst;
    }

    public final void put(Object[][] attributes)
    {
        if (attributes != null)
        {
            for (int i = 0; i < attributes.length; i++)
            {
                Object[] attribute = (Object[])attributes[i];

                if ((attribute == null) || (attribute.length != 2))
                {
                    continue;
                }

                set(attribute[0].toString(), attribute[1]);
            }
        }
    }

    protected boolean hasParent()
    {
        return (getParent() != null);
    }

    public PluginAttributes getParent()
    {
        return m_parent;
    }

    protected void setParent(PluginAttributes parent)
    {
        m_parent = parent;
    }

    @Override
    public String toString()
    {
        StringBuffer buffer = new StringBuffer();

        for(Iterator iter = m_map.keySet().iterator(); iter.hasNext();)
        {
            String key = (String)iter.next();

            if(buffer.length() > 0)
            {
                buffer.append(", ");
            }

            buffer.append(key).append('=').append(toString(m_map.get(key)));
        }
        return buffer.toString();
    }

    private String formatStringArray(String[] val)
    {
        StringBuffer buffer = new StringBuffer();

        for(int i = 0; i < val.length; i++)
        {
            if(buffer.length() > 0)
            {
                buffer.append(", ");
            }

            buffer.append(val[i]);
        }
        return buffer.toString();
    }

    public String toString(Object val)
    {
        String ret = null;

        if(val instanceof String[])
        {
            ret = formatStringArray((String[])val);
        }
        else if(val instanceof IState)
        {
            ret = ((IState)val).getStateString();
        }
        else if(val instanceof String)
        {
            ret = (String)val;
        }
        else if(val instanceof Boolean)
        {
            ret = val.toString();
        }
        else
        {
            ret = "?";
        }

        return ret;
    }

    public boolean getBooleanValue(String key)
    {
        Boolean val = (Boolean)get(key);

        return (val != null) ? val.booleanValue() : false;
    }

    //-------------------------------------------------------------------------

    public String getType()
    {
        return (String)get(TYPE);
    }

    public String getDisplayType()
    {
        return (String)get(DISPLAY_TYPE);
    }

    public String getConfigVersion()
    {
        return (String)get(CONFIG_VERSION);
    }

    public String getProductVersion()
    {
        return (String)get(PRODUCT_VERSION);
    }

    public String[] getProductVersions()
    {
        final Object productVersionObject = get(PRODUCT_VERSIONS);

        if (productVersionObject instanceof Collection) {
            final Collection<String> productVersionCollection = (Collection<String>) productVersionObject;
            return productVersionCollection.toArray(new String[productVersionCollection.size()]);
        } else {
            return (String[]) productVersionObject;
        }
    }

    public String getProductVersionsString()
    {
        return formatStringArray(getProductVersions());
    }

    public String getProductFamilyId()
    {
        return (String)get(PRODUCT_FAMILY_ID);
    }

    public String getPluginRevision()
    {
        return (String)get(PLUGIN_REVISION);
    }

    public String getConfigProductVersion()
    {
        String cVersion = getConfigVersion();
        String pVersion = getProductVersion();

        if(pVersion == null && cVersion == null)
        {
            return null;
        }

        if(pVersion == null)
        {
            pVersion = "Unknown";
        }

        if(cVersion == null)
        {
            cVersion = "Unknown";
        }

        return pVersion + " (" + cVersion + ")";
    }

    public boolean isTemplatable()
    {
        return getBooleanValue(IS_TEMPLATABLE);
    }

    public boolean isCreatable()
    {
        return getBooleanValue(IS_CREATABLE);
    }

    public boolean isUpgradeable()
    {
        return getBooleanValue(IS_UPGRADEABLE);
    }

    public boolean isSystem()
    {
        boolean bIsSystem = false;
        String strResult = (String)get(IS_SYSTEM);

        if (strResult != null)
        {
            bIsSystem = strResult.equals("true");
        }

        return bIsSystem;
    }

    public boolean isChildBearing()
    {
        return getBooleanValue(IS_CHILD_BEARING);
    }

    public IState getState()
    {
        return (IState)get(STATE);
    }

    public final void setState(IState state)
    {
        set(STATE, state);
    }

    public boolean isStateable()
    {
        return getBooleanValue(IS_STATEABLE);
    }

    public boolean isRuntimeRoot()
    {
        return getBooleanValue(IS_RUNTIME_ROOT);
    }

    public boolean isReloadable()
    {
        return getBooleanValue(IS_RELOADABLE);
    }

    public boolean isComponent()
    {
        return getBooleanValue(IS_COMPONENT);
    }

    public boolean isDynamicRuntime()
    {
        return getBooleanValue(IS_DYNAMIC_RUNTIME);
    }
    
    public boolean isPrimaryConfiguration()
    {
        return getBooleanValue(PRIMARY_CONFIGURATION);
    }
}
