/**
 * Copyright (c) 2003 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.ma.plugin;

import java.io.File;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.StringTokenizer;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.domain.DomainConnectionModel;
import com.sonicsw.ma.gui.util.Helper;

/**
 * <p>Title: PluginClassLoader</p>
 * <p>Description: This class maintains the URLS of the plugin jars.</p>
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: Sonic Software Corporation</p>
 * @author Jeffrey S. Pace
 * @version 1.0
 */
public class PluginClassLoader
    extends URLClassLoader
{
    /**
     * PluginClassLoader
     * @param classpath
     * @param cs
     * @throws Exception
     */
    public PluginClassLoader(String classpath, DomainConnectionModel connectionInfo)
        throws Exception
    {
        super(new URL[0]);

        buildURLs(classpath, connectionInfo);
    }
    
    /**
     * buildURLs
     * @param classpath
     * @param cs
     * @throws Exception
     */
    public final void buildURLs(String classpath, DomainConnectionModel connectionInfo)
        throws Exception
    {
        Helper.logDebugMessage("Creating PluginClassLoader for " + classpath);

        // Need to tokenize and split up the classpath into individual URLs
        StringTokenizer st = new StringTokenizer(classpath, ";");

        while (st.hasMoreTokens())
        {
            String strPluginPath = st.nextToken();

            if (PluginJarFile.isValidFile(strPluginPath))
            {
                // looks like a valid definition in the DS...
                PluginJarFile pluginJarFile = new PluginJarFile(strPluginPath, connectionInfo);

                // load into cache manager...
                JarFileCache.add(pluginJarFile);
                
                //Only add the url if its not already in the list...
                if (!isURLPresent(pluginJarFile.toURL()))
                {
                    addURL(pluginJarFile.toURL());
                }
            }
            else
            {
                File file = new File(strPluginPath);
                
                // is not a valid DS file.  add it as a regular File...
                if (!isURLPresent(file.toURL()))
                {
                    addURL(file.toURL());
                }
            }
            Helper.logDebugMessage("Load of plugin from '" + strPluginPath + "' complete.");
        }
    }

    private boolean isURLPresent(URL url)
    {
        URL[] urls = getURLs();
        
        for (int i = 0; i < urls.length; i++)
        {
            if ((urls[i] != null) && urls[i].equals(url))
            {
                return true;
            }
        }
            
        return false;
    }
    
    /**
     * Load the class with the specified name, searching using the following
     * algorithm until it finds and returns the class.  If the class cannot
     * be found, returns <code>ClassNotFoundException</code>.
     * <ul>
     * <li>Call <code>findLoadedClass(String)</code> to check if the
     *     class has already been loaded.  If it has, the same
     *     <code>Class</code> object is returned.</li>
     * <li>Call <code>findClass()</code> to find this class in our locally
     *     defined repositories.</li>
     * <li>Call the <code>loadClass()</code> method of our parent
     *     class loader, if any.</li>
     * </ul>
     * If the class was found using the above steps, and the
     * <code>resolve</code> flag is <code>true</code>, this method will then
     * call <code>resolveClass(Class)</code> on the resulting Class object.
     *
     * @param name Name of the class to be loaded
     * @param resolve If <code>true</code> then resolve the class
     *
     * @exception ClassNotFoundException if the class was not found
     */
    @Override
    public Class loadClass(String name, boolean resolve)
    throws ClassNotFoundException
    {
        // Otherwise we are going to load from the plugins jars first and then
        // look to the classpath as a last resort
        Class clazz = null;

        if (name.startsWith("["))
        {
            clazz = Class.forName(name, false, this);
            if (resolve)
            {
                resolveClass(clazz);
            }
            return clazz;
        }
        
        // If we want to use locally built CLASSPATH classes first
        if(MgmtConsole.LOCAL_CLASSES)
        {
            return super.loadClass(name, resolve);
        }

        // (1) Check our previously loaded class cache
        clazz = findLoadedClass(name);
        if (clazz != null)
        {
            if (resolve)
            {
                resolveClass(clazz);
            }
            return (clazz);
        }

        // (2) Search local repositories
        try
        {
            clazz = findClass(name);
            if (clazz != null)
            {
                if (resolve)
                {
                    resolveClass(clazz);
                }
                return (clazz);
            }
        }
        catch (ClassNotFoundException e) { }

        // (3) Delegate to parent unconditionally - use super class to do this
        return super.loadClass(name, resolve);
    }
    
    /**
     * Finds the resource with the specified name on the URL search path.
     *
     * @param name the name of the resource
     * @return a <code>URL</code> for the resource, or <code>null</code> 
     * if the resource could not be found.
     */
    @Override
    public URL getResource(String name) {
      URL result = super.findResource(name);
      if (null == result) {
          result = super.getResource(name);
      }
      return result;
    }
}