package com.sonicsw.ma.plugin;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.StringTokenizer;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.WorkspaceWindow;
import com.sonicsw.ma.gui.domain.DomainConnectionModel;
import com.sonicsw.ma.gui.util.Helper;

import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.dirconfig.IDirElement;

/**
 * Provides context based services for plugins to a MA application.
 */
public class PluginContext implements IPluginContext
{
    private static final String ELEMENT_ID_JMS    = "JMS";
    private static final String ELEMENT_ID_SOAP   = "SOAP";
    private static final String ELEMENT_ID_DIRECT = "DIRECT";

    private static final String MF_LIBRARY_BASE = IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_LIBRARY_DIR + IMFDirectories.MF_DIR_SEPARATOR;

    private static final String MF_LIBRARY_PLUGINS = new String(MF_LIBRARY_BASE) + "plugins";
    private static final String MF_LIBRARY_TOOLS   = new String(MF_LIBRARY_BASE) + "tools";
    private static final String MF_LIBRARY_EDITORS = new String(MF_LIBRARY_BASE) + "editors";
    private static final String MF_LIBRARY_EXTENSIONS = new String(MF_LIBRARY_BASE) + "extensions";

    private static final AttributeName AN_PLUGIN_CONFIG    = new AttributeName("CONFIG_PLUGIN_FACTORY_CLASS");
    private static final AttributeName AN_PLUGIN_RUNTIME   = new AttributeName("RUNTIME_PLUGIN_FACTORY_CLASS");
    private static final AttributeName AN_PLUGIN_CLASSPATH = new AttributeName("PLUGIN_FACTORY_CLASSPATH");

    private static final AttributeName AN_TOOL_NAME       = new AttributeName("NAME");
    private static final AttributeName AN_TOOL_DESC       = new AttributeName("DESCRIPTION");
    private static final AttributeName AN_TOOL_CLASS      = new AttributeName("CLASS");
    private static final AttributeName AN_TOOL_CLASSPATH  = new AttributeName("CLASSPATH");
    private static final AttributeName AN_TOOL_APP        = new AttributeName("APP");
    private static final AttributeName AN_TOOL_CONFIGURE  = new AttributeName("CONFIGURE");

    private static final AttributeName AN_EDITOR_EXTENSION  = new AttributeName("EXTENSION");
    private static final AttributeName AN_EDITOR_DESC       = new AttributeName("DESCRIPTION");
    private static final AttributeName AN_EDITOR_CLASS      = new AttributeName("CLASS");
    private static final AttributeName AN_EDITOR_CLASSPATH  = new AttributeName("CLASSPATH");

    private static final AttributeName AN_TOOL_EXTENSION_DESC      = new AttributeName("DESCRIPTION");
    private static final AttributeName AN_TOOL_EXTENSION_CLASSNAME = new AttributeName("CLASSNAME");
    private static final AttributeName AN_TOOL_EXTENSION_CLASSPATH = new AttributeName("CLASSPATH");
    
    private boolean               m_destroy = false;
    private IRuntimeContext       m_runtimeContext;
    private IConfigContext        m_configContext;
    private DomainConnectionModel m_connectionInfo;
    private WorkspaceWindow       m_workspace;
    private MFLibraryMap          m_library = new MFLibraryMap();

    public PluginContext(DomainConnectionModel connectionInfo)
    {
        m_connectionInfo = connectionInfo;
    }

    /**
     * Get the Configuration Specific Context
     */
    @Override
    public IConfigContext getConfigContext()
    {
        return m_configContext;
    }

    /**
     * Get the Runtime Specific Context
     */
    @Override
    public IRuntimeContext getRuntimeContext()
    {
        return m_runtimeContext;
    }

    /**
     * Get information about the connection
     */
    @Override
    public DomainConnectionModel getConnectionInfo()
    {
        return m_connectionInfo;
    }

    @Override
    public MgmtConsole getMgmtConsole()
    {
        return MgmtConsole.getMgmtConsole();
    }

    @Override
    public MFLibraryMap getLibrary()
    {
        return m_library;
    }

    @Override
    public WorkspaceWindow getWorkspace()
    {
        return m_workspace;
    }

    public void setWorkspace(WorkspaceWindow workspace)
    {
        m_workspace = workspace;
    }

    /**
     * Initialize the Context
     */
    @Override
    public void create() throws Exception
    {
        m_connectionInfo.connect();

        if(m_destroy)
        {
            return;
        }

        m_runtimeContext = new RuntimeContext(this);
        m_configContext  = new ConfigContext(this);

        initPluginFactoryMap();
        initToolsMap();
        initEditorsMap();
        initToolExtensions();
    }

    @Override
    public void destroy()
    {
        m_destroy        = true;
        m_workspace      = null;
        m_library        = null;
        m_runtimeContext = null;
        m_configContext  = null;
    }

    /**
     * Init the plugins from the information stored in the DS
     */
    private void initPluginFactoryMap() throws Exception
    {
        IDirectoryAdminService dsp     = (IDirectoryAdminService)getConnectionInfo().getDirectoryService();
        IDirElement[]          element = dsp.getAllElements(MF_LIBRARY_PLUGINS, false);

        for (int i = 0; i < element.length; i++)
        {
            if(m_destroy)
            {
                return;
            }

            try
            {
                String config    = (String)element[i].getAttribute(AN_PLUGIN_CONFIG);
                String runtime   = (String)element[i].getAttribute(AN_PLUGIN_RUNTIME);
                String classpath = (String)element[i].getAttribute(AN_PLUGIN_CLASSPATH);

                ClassLoader cl = null;

                // create a classloader for the plugin factory
                if(classpath != null && classpath.length() != 0)
                {
                    cl = new PluginClassLoader(classpath, m_connectionInfo);
                }
                else
                {
                    cl = this.getClass().getClassLoader();
                }

                if(config != null)
                {
                    addConfigFactory(config, cl, element[i]);
                }

                if(runtime != null)
                {
                    addRuntimeFactory(runtime, cl, element[i]);
                }
            }
            catch (Throwable e)
            {
                getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                    "Could not load plugin factory: " + (String)element[i].getAttribute(AN_PLUGIN_CLASSPATH), e, true);
            }
        }
    }

    private void initToolsMap() throws Exception
    {
        IDirectoryAdminService dsp     = (IDirectoryAdminService)getConnectionInfo().getDirectoryService();
        IDirElement[]          element = dsp.getAllElements(MF_LIBRARY_TOOLS, false);

        for (int i = 0; i < element.length; i++)
        {
            if(m_destroy)
            {
                return;
            }

            try
            {
                String  name      = (String)element[i].getAttribute(AN_TOOL_NAME);
                String  desc      = (String)element[i].getAttribute(AN_TOOL_DESC);
                String  app       = (String)element[i].getAttribute(AN_TOOL_APP);
                String  cls       = (String)element[i].getAttribute(AN_TOOL_CLASS);
                String  classpath = (String)element[i].getAttribute(AN_TOOL_CLASSPATH);
                Boolean config    = (Boolean)element[i].getAttribute(AN_TOOL_CONFIGURE);

                ClassLoader cl = null;

                if(classpath != null && classpath.length() != 0)
                {
                    cl = new PluginClassLoader(classpath, m_connectionInfo);
                }
                else
                {
                    cl = this.getClass().getClassLoader();
                }

                m_library.addToolInfo(ToolInfo.createInfo(this,
                    cl, name, desc, app, cls, config != null ? config.booleanValue() : false));
            }
            catch (Throwable e)
            {
            }
        }
    }

    private void initEditorsMap() throws Exception
    {
        try
        {
            IDirectoryAdminService dsp = (IDirectoryAdminService)getConnectionInfo().getDirectoryService();
            IDirElement[] element = dsp.getAllElements(MF_LIBRARY_EDITORS, false);

            for (int i = 0; i < element.length; i++)
            {
                if (m_destroy)
                {
                    return;
                }

                try
                {
                    ArrayList extns = new ArrayList();
                    StringTokenizer tn = new StringTokenizer(((String) element[i].getAttribute(AN_EDITOR_EXTENSION)).trim(), ",", false);
                    while(tn.hasMoreTokens())
                    {
                        extns.add(tn.nextToken().trim());
                    }
                    
                    String desc = (String) element[i].getAttribute(AN_EDITOR_DESC);
                    String cls = (String) element[i].getAttribute(AN_EDITOR_CLASS);
                    String classpath = (String) element[i].getAttribute(AN_EDITOR_CLASSPATH);

                    ClassLoader cl = null;

                    if(classpath != null && classpath.length() != 0)
                    {
                        cl = new PluginClassLoader(classpath, m_connectionInfo);
                    }
                    else
                    {
                        cl = this.getClass().getClassLoader();
                    }
                    
                    for(Iterator it = extns.iterator(); it.hasNext(); )
                    {
                        m_library.addEditorInfo(new EditorInfo(this, cl, (String)it.next(), desc, cls));
                    }
                }
                catch (Throwable e)
                {
                }
            }
        }
        catch(Exception e)
        {
        }
    }


    private void addConfigFactory(String className, ClassLoader cl, IDirElement element)
    {
        try
        {
            ConfigFactoryInfo info = new ConfigFactoryInfo(className, cl, element);

            m_library.addConfigRootFactoryInfo(info);

            PluginAttributes attributes = info.getRootFactory().getAttributes();

            StringBuffer msg = new StringBuffer("Loaded config factory for ");
            msg.append(attributes.getDisplayType());
            msg.append(" (product=");
            msg.append(attributes.getProductVersionsString());
            msg.append(" config=");
            msg.append(attributes.getConfigVersion());
            msg.append(" build=");
            msg.append(attributes.getPluginRevision());
            msg.append(")");

            getMgmtConsole().notifyMessage(IApplication.MESSAGE_INFO, msg.toString(), false);
        }
        catch(Throwable e)
        {
            getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                "Failed to load configuration factory for " + className, e, false);
        }
    }

    private void addRuntimeFactory(String className, ClassLoader cl, IDirElement element)
    {
        try
        {
            RuntimeFactoryInfo info = new RuntimeFactoryInfo(className, cl, element);

            m_library.addRuntimeRootFactoryInfo(info);

            PluginAttributes attributes = info.getRootFactory().getAttributes();

            StringBuffer msg = new StringBuffer("Loaded runtime factory for ");
            msg.append(attributes.getDisplayType());
            msg.append(" (product=");
            msg.append(attributes.getProductVersionsString());
            msg.append(" config=");
            msg.append(attributes.getConfigVersion());
            msg.append(" build=");
            msg.append(attributes.getPluginRevision());
            msg.append(")");

            getMgmtConsole().notifyMessage(IApplication.MESSAGE_INFO, msg.toString(), false);
        }
        catch(Throwable e)
        {
            getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                 "Failed to load runtime factory for " + className, e, false);
        }
    }
    
    private void initToolExtensions() throws Exception
    {
        try
        {
            IDirectoryAdminService dsp = (IDirectoryAdminService)getConnectionInfo().getDirectoryService();
            IDirElement[] element = dsp.getAllElements(MF_LIBRARY_EXTENSIONS, false);

            for (int i = 0; i < element.length; i++)
            {
                if (m_destroy)
                {
                    return;
                }

                try
                {
                    String desc = (String) element[i].getAttribute(AN_TOOL_EXTENSION_DESC);
                    String classname = (String) element[i].getAttribute(AN_TOOL_EXTENSION_CLASSNAME);
                    String classpath = (String) element[i].getAttribute(AN_TOOL_EXTENSION_CLASSPATH);

                    ClassLoader cl = null;

                    if(classpath != null && classpath.length() != 0)
                    {
                        cl = new PluginClassLoader(classpath, m_connectionInfo);
                    }
                    else
                    {
                        cl = this.getClass().getClassLoader();
                    }
                    
                    Helper.logDebugMessage("Loading Tool Extension: " + desc + " (" + classname + ")");
                    
                    IToolExtension ext = (IToolExtension)Class.forName(classname, true, cl).newInstance();
                    ext.init(this);
                }
                catch (Throwable e)
                {
                    getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                                  "Failed to load tool extension ", e, false);
                }
            }
        }
        catch(Exception e)
        {
            getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                         "Failed to load tool extensions ", e, false);
        }
    }
}