/**
 * Copyright (c) 2003 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.ma.plugin;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.net.MalformedURLException;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.domain.DomainConnectionModel;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.JWaitCursor;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigBeanFile;
import com.sonicsw.mx.config.IConfigElement;

/**
 * <p>Title: PluginJarFile</p>
 * <p>Description: This class manages plugin files accessible via the Sonic DS and interaction with the JarFileCache. </p>
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: Sonic Software Corporation</p>
 * @author Jeffrey S. Pace
 * @version 1.0
 */
public class PluginJarFile
    extends File
{
    private static final String SONICFS_PREFIX = "sonicfs://";
    private static final String DRIVE_SEPARATOR = ":/";

    private static final String[] EXTENSIONS = { ".jar", ".zip" };

    private String m_strOriginalPluginPath = null;
    private DomainConnectionModel m_connectionInfo = null;
    private boolean m_bInUse = false;

    /**
     * PluginJarFile
     * @param strPluginPath
     * @throws Exception
     */
    public PluginJarFile(String strPluginPath, DomainConnectionModel connectionInfo)
        throws Exception
    {
        super(convertToCacheFilename(strPluginPath, connectionInfo));

        m_strOriginalPluginPath = strPluginPath;
        m_connectionInfo = connectionInfo;
    }

    /**
     * determines if the path is using the sonicfs: descriptor.
     * @param strPluginPath
     * @return
     */
    public static boolean hasSonicFSPath(String strPluginPath)
    {
        return strPluginPath.startsWith(SONICFS_PREFIX);
    }

    /**
     * validates that strPluginPath is a path to a valid jar file type.
     * @param strPluginPath
     * @return true if valid
     */
    public static boolean isValidFile(String strPluginPath)
    {
        if (hasSonicFSPath(strPluginPath))
        {
            for(int i = 0; i < EXTENSIONS.length; i++)
            {
                if (strPluginPath.endsWith(EXTENSIONS[i]))
                {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * gets the path relative to the cache root.
     * @return path relative to the cache root.
     * @throws MalformedURLException
     */
    public String getRelativePath()
    {
        String strRelativePath = null;

        if (hasSonicFSPath(m_strOriginalPluginPath))
        {
            strRelativePath = m_strOriginalPluginPath.substring(SONICFS_PREFIX.length());
        }
        else
        {
            Helper.logErrorMessage("PluginJarFile is not sonicfs: type.");
        }
        return strRelativePath;
    }

    /**
     * gets the original path.
     * @return original path.
     */
    public String getOriginalPluginPath()
    {
        return m_strOriginalPluginPath;
    }

    /**
     * creates the absolute path to the file as it will be contained in the cache.
     * @param strPluginPath
     * @return
     * @throws MalformedURLException
     */
    private static String convertToCacheFilename(String strPluginPath, DomainConnectionModel connectionInfo)
        throws MalformedURLException
    {
        if(!hasSonicFSPath(strPluginPath))
        {
            throw new MalformedURLException("Invalid file path: " + strPluginPath);
        }

        String strLocalFilename = strPluginPath.substring(SONICFS_PREFIX.length());

        return JarFileCache.getCachePath() +
               File.separator +
               String.valueOf(connectionInfo.getUniqueId()) +
               strLocalFilename;
    }

    /**
     * needsUpdating
     * @return
     * @throws ConfigServiceException
     */
    protected boolean needsUpdating()
        throws ConfigServiceException
    {
        boolean bNeedsUpdating = true;

        if (isCached())
        {
            // its already in the cache, now depends on the modified state...
            if (m_connectionInfo.getConfigServer() != null)
            {
                IConfigBean dsFile = (IConfigBean)m_connectionInfo.getConfigServer().loadConfigElement(getRelativePath());

                if(dsFile == null)
                {
                    return false;
                }

                // MODIFIED will be a constant in the SonicFS file plugin, when that is created
                long localModified = lastModified();
                long dsModified = ((Long)dsFile.getAttribute("LAST_MODIFIED_TIME")).longValue();

                bNeedsUpdating = dsModified > localModified;
            }
            else
            {
                // fill in when we have coded the alternate connection to the DS
            }
        }
        return bNeedsUpdating;
    }

    /**
     * isCached
     * @return
     */
    protected boolean isCached()
    {
        return exists();
    }

    /**
     * cacheFile
     * @throws Exception
     */
    protected void cacheFile()
        throws Exception
    {
        // make sure the parent path has been created
        getParentFile().mkdirs();

        exportFile();
    }

    /**
     * exportFile
     * @throws Exception
     */
    protected void exportFile()
        throws Exception
    {
        FileOutputStream os = null;
        InputStream is = null;

        JWaitCursor wc = new JWaitCursor(MgmtConsole.getMgmtConsole());

        String relativePath = getRelativePath();

        MgmtConsole.getMgmtConsole().notifyMessage(
            MgmtConsole.MESSAGE_STATUS,
            "Writing " + relativePath + " into plugin cache...",
            false);

        try
        {
            IConfigElement element = m_connectionInfo.getConfigServer().loadConfigElement(relativePath);
            IConfigBeanFile bean = (IConfigBeanFile)element;

            if (bean == null)
            {
                throw new Exception("File '" + relativePath + "' not found in DS.");
            }

            is = bean.getContents();

            if (is != null)
            {
                os = new FileOutputStream(PluginJarFile.this);

                byte[] buffer = new byte[1000];
                while (true)
                {
                    int length = is.read(buffer);

                    if (length == -1)
                    {
                        break;
                    }

                    os.write(buffer, 0, length);
                }
            }
        }
        finally
        {
            if (os != null)
            {
                os.flush();
                os.close();
            }

            if (is != null)
            {
                is.close();
            }

            wc.release();
        }
    }
}