package com.sonicsw.mf.comm;

import java.util.Map;

import com.sonicsw.mf.common.IConsumer;

/**
 * The IConnectorClient interface describes the methods to be implemented by
 * classes that map MF management communications over a transport (e.g. JMS).
 */
public interface IConnectorClient
extends IConnectionListener
{
    public static final String TRACE_MASK_VALUES = "16=request/reply,32=request/reply failures,64=connection failures,128=connection state changes,";
    public static final int TRACE_DETAIL = 1;
    public static final int TRACE_REQUEST_REPLY = 16;
    public static final int TRACE_REQUEST_REPLY_FAILURES = TRACE_REQUEST_REPLY << 1;
    public static final int TRACE_CONNECTION_FAILURES = TRACE_REQUEST_REPLY_FAILURES << 1;
    public static final int TRACE_CONNECTION_STATE_CHANGES = TRACE_CONNECTION_FAILURES << 1;

    /**
     * The system property name used to define if the connection layer should log significant connection
     * messages to the java console.
     * <p>
     * As an example : -Dsonicsw.mf.logToConsole=true
     */
    public static final String MF_LOG_CONNECTION_MESSAGES_PROPERTY = "sonicsw.mf.logConnectionMessages";

    /**
     * Set the routing node that the containers hosting the Directory Service and Agent
     * Manager are connected to for management communications.
     */
    public void setManagementNode(String node);

    /**
     * Dynamically changes the underlying connection to one created from
     * the given environment. The environment is a set of name/value pairs matching
     * settable attributes on a SonicMQ TopicConnectionFactory.
     *
     * Any request/reply in process at the time the new factories are applied
     * will be cancelled.
     *
     * @param env     The properties used to create a connection factory.
     * @param timeout The length of time in which the initial connection must be established.
     */
    public void connect(Map env, long timeout)
    throws Exception;

    /**
     * Returns true if there is a live connection under the hood.
     */
    public boolean isConnected();

    /**
     * Gets the exception listener for the connector.
     */
    public IExceptionListener getExceptionListener();

    /**
     * Sets the exception listener for the connector.
     */
    public void setExceptionListener(IExceptionListener listener);

    /**
     * Gets the connection listener for the connector.
     */
    public IConnectionListener getConnectionListener();

    /**
     * Sets the connection listener for the connector.
     */
    public void setConnectionListener(IConnectionListener listener);

    /**
     * Gets the orphaned reply listener for the connector.
     */
    public IOrphanedReplyListener getOrphanedReplyListener();

    /**
     * Sets the orphaned reply listener for the connector.
     */
    public void setOrphanedReplyListener(IOrphanedReplyListener listener);

    /**
     * To indicate that the connector is about to be closed so in general failures will not be
     * retried (and potentially reported).
     */
    public void closePending();

    /**
     * Cleanup and closeout the connector. Once called the connector is not reusable.
     *
     * All registrations created with the connector will be removed.
     */
    public void close();

    /**
     * Modifies the 30000 milliseconds default timeout the connector waits for responses from remote requests. The minimum is
     * 10000 milliseconds (silently enforced).
     *
     * @param timeout in milliseconds
     *
     */
    public void setRequestTimeout(long timeout);

    /**
     * Adds a directed listener to the specified notification from the specified source.
     * Directed notifications are those which are generated at the request of a
     * specific listener; this is in contrast to broadcast notifications.
     *
     * A directed notification listener must also register his interest in receiving
     * notification directly with the source (which will only send notifications to
     * registered listeners).
     *
     * @param listener         A listener that handles directed notifications.
     *
     * @return                 A consumer object on which close() should be called to
     *                         remove and cleanup on behalf of the listener.
     *
     * @see com.sonicsw.mf.common.IConsumer#close()
     */
    public IConsumer addDirectedNotificationListener(INotificationListener listener);

    /**
     * Invoke a management request with the default (ADMINISTRATOR) role and connector timeout.
     *
     * @see #invoke(String, String, String, String, Object[], String[], String, long)
     */
    public Object invoke(String commsType, String namespace, String target, String operationName, Object[] params, String[] signature)
    throws InvokeTimeoutException, Exception;

    /**
     * Invoke a management request with the default (ADMINISTRATOR) role and specific timeout.
     *
     * @see #invoke(String, String, String, String, Object[], String[], String, long)
     */
    public Object invoke(String commsType, String namespace, String target, String operationName, Object[] params, String[] signature, long timeout)
    throws InvokeTimeoutException, Exception;

    /**
     * Invoke a management request with the given role and connector timeout.
     *
     * @see #invoke(String, String, String, String, Object[], String[], String, long)
     */
    public Object invoke(String commsType, String namespace, String target, String operationName, Object[] params, String[] signature, String role)
    throws InvokeTimeoutException, Exception;

    /**
     * Invoke a management operation on the specified component. The connector will
     * not check to see if the specified component is collocated with the requester
     * (it is the requester that needs to handle that efficiency).
     *
     * @param commsType     Communications types are those supported by MF (although not
     *                      explicitely checked). At time of writing the following types may be
     *                      used:
     *
     *                        - "mf" (intended for internal container-to-container use)
     *                        - "jmx" (intended for JMX connector use)
     * @param namespace     The container namespace in which the target exists (used to
     *                      create the topic name used for communications .. will be of the 
     *                      form:
     *                        <domain name>.<container name>
     * @param target        The stringified JMX object name
     * @param operationName The name of the operation to invoke.
     * @param params        An array of serializable objects that form the operation
     *                      parameters. Java primative types should be wrapped in their
     *                      assocaited wrapper class. Serializable object should follow
     *                      MF guidelines regarding serialization where possible to
     *                      ensure best version compatibility.
     * @param signature     An array of class names describing the parameters.
     * @param role          The role to be associated with a request.
     * @param timeout       The duration after which the connector will give up waiting for
     *                      a response and instead throw a timeout exception.
     *
     * @return              The return value from the operation. If the operation
     *                      returns a primative type, the return value will be wrapped
     *                      its associated wrapper class. If the operation returns void
     *                      this operation will return null.
     *
     * @throws InvokeTimeoutException
     *                      Thrown when the response to the invoke request has not been
     *                      received within a particular timeframe.
     * @throws Exception    Thrown for all other exceptions.
     */
    public Object invoke(String commsType, String namespace, String target, String operationName, Object[] params, String[] signature, String role, long timeout)
    throws InvokeTimeoutException, Exception;

    /**
     * Sets a debug mask for the connector.
     */
    public void setTraceMask(int traceMask);

    /**
     * Gets the debug mask for the connector.
     */
    public int getTraceMask();

   /**
    * Gets the node name of the broker this connector is connected to
    */
    public String getLocalRoutingNodeName();
 
    /**
     * Gets the enterprise setting of the broker this connector is connected to
     */
     public boolean isConnectionEnterpriseEnabled();


}
