package com.sonicsw.mf.comm.jms;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.PrintWriter;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.rmi.server.UID;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;

import javax.jms.BytesMessage;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;

import com.sonicsw.mx.util.LoaderInputStream;

import com.sonicsw.mf.comm.ConnectTimeoutException;
import com.sonicsw.mf.comm.IConnectionListener;
import com.sonicsw.mf.comm.IConnectorClient;
import com.sonicsw.mf.comm.IDurableConnectorConsumer;
import com.sonicsw.mf.comm.IExceptionListener;
import com.sonicsw.mf.comm.INotificationListener;
import com.sonicsw.mf.comm.IOrphanedReplyListener;
import com.sonicsw.mf.comm.IRetryCallback;
import com.sonicsw.mf.comm.InvokeTimeoutCommsException;
import com.sonicsw.mf.comm.InvokeTimeoutException;
import com.sonicsw.mf.common.IConsumer;
import com.sonicsw.mf.common.MFConnectAbortedException;
import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.MFSecurityException;
import com.sonicsw.mf.common.MFServiceNotActiveException;
import com.sonicsw.mf.common.runtime.IContainerIdentity;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.runtime.impl.CanonicalName;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;

import progress.message.client.EPasswordExpired;
import progress.message.client.ESecurityGeneralException;
import progress.message.client.ESecurityPolicyViolation;
import progress.message.client.EUnauthorizedClient;
import progress.message.jclient.TopicConnectionFactory;

/**
 * The ConnectorClient class is a concrete implementation of the IConnectorClient
 * interface and is responsible for mapping MF management communications
 * over JMS. It is intended for use by management applications via their respective
 * client side connector (e.g. JMX/JMS connector client). It has knowledge of how to
 * map request/reply and notifications over JMS.
 *
 * The ConnectorClient transparently provides a level of QoS beyond that of regular
 * SonicMQ JMS messaging. Clients of the ConnectorClient are unaware of temporary
 * loss (and re-establishment) of connection.
 *
 * @see com.sonicsw.mf.framework.comm.jms.ConnectorServer
 */
public class ConnectorClient
implements IConnectorClient, IDurableConnectorConsumer, MessageListener
{
    private static boolean DEBUG_TRACE_CONNECTION_STATE = false;
    private static boolean DEBUG_TRACE_REQUEST = false;
    private static boolean DEBUG_TRACE_INVOCATION = false;
    private static boolean DEBUG_TRACE_RETRY_CALLBACK = false;
    
    private static String REQUEST_TIMEOUT_OVERRIDE_PROPERTY = "sonicsw.mf.requestTimeoutOverride";
    private static String REQUEST_TIMEOUT_OVERRIDE = null;
    
    protected int m_traceMask;
    protected boolean m_logConnectionMessages = false;

    // duh - the MF domain name
    protected String m_domainName;
    // duh - the MF container name
    protected String m_containerName;
    // the routing node through which containers hosting the DS/AM are communicating
    protected String m_managementNode;  // a.k.a. "primary node name"

    // duh - the request timeout
    protected long m_requestTimeout = REQUEST_TIMEOUT_DEFAULT;

    // d'oh - the connect timeout
    protected long m_connectTimeout = CONNECT_TIMEOUT_DEFAULT;
    
    // the delivery mode. The default for mgmt comms is PERSISTENT, but JNDICLIENT uses NON_PERSISTENT for speed
    // no d'oh - nothing is that obvious to me.
    private int m_deliveryMode = javax.jms.DeliveryMode.PERSISTENT;// progress.message.jclient.DeliveryMode.DISCARDABLE;
    
    private boolean m_isDurable = true; // default is to do durable subscriptions

    // d'oh - the socket connect timeout
    protected long m_socketConnectTimeout = SOCKET_CONNECT_TIMEOUT_DEFAULT;

    // the generic subject for request/replies/notifications destined exclusively to this peer/client
    private String m_directedSubscriptionSubject;
    private String m_directedReplyToSubject;
    // the subscription name to be used for durable subscriptions .. management clients use
    // non-durable subscriptions
    private String m_subscriptionName;
    // the connectID ID (needs to be identifiable so we do not count these in connection count at the broker)
    private String m_connectID;
    // default role (if not JNDI roles)
    private String m_defaultRole;

    // table of notification listeners to delegates
    private Hashtable m_notificationListeners = new Hashtable();

    // the underlying connector
    protected DurableConnector m_durableConnector;

    // the connection and services state manager
    protected ConnectionStateManager m_stateManager;

    // the underlying subscription (in the JMSConnectorServer's case we want to make best attempts to close
    // this early in the shutdown process so we don't get anymore requests
    protected IConsumer m_subscription;

    // table of outstanding requests
    protected HashMap m_activeRequests = new HashMap();

    // listener to permanent connection failures
    protected IExceptionListener m_exceptionListener;
    // listener to connection conditions
    protected IConnectionListener m_connectionListener;
    // listener to orphaned replies (those received after the request has timed out waiting for the reply)
    protected IOrphanedReplyListener m_orphanedReplyListener;

    // flag to indicate if this ConnectorClient instance resides in an MF container
    protected boolean m_inContainer = false;

    // connection factories used to set up connections for management requests
    private TopicConnectionFactory m_connectionFactory;

    // reference to reply callback object
    private IRetryCallback m_retryCallback = null;
    private Object m_requestLock = new Object();
    private RetryCallbackManager m_retryCallbackManager = null;
    
    private IDurableConnectorFactory m_durableConnectorFactory = new DurableConnectorFactory();
    private ITopicConnectionFactoryFactory m_topicConnectionFactoryFactory = new TopicConnectionFactoryFactory();
    
    
    private static boolean m_debugRequestTimeout = false;
    private static boolean m_requestTimeoutPrinted = false;

    //
    // Constants
    //

    private static final ClassLoader m_loader = ConnectorClient.class.getClassLoader();

    // default timeout for invoke
    public static final long REQUEST_TIMEOUT_DEFAULT = getRequestDefaultTimeOut();
    public static final String REQUEST_TIMEOUT_DEFAULT_PROPERTY = "sonicsw.mf.requestTimeoutDefault";
    // minimum timeout for invoke
    public static final long REQUEST_TIMEOUT_MINIMUM = 10000;

    // default timeout for connection setup
    public static final long CONNECT_TIMEOUT_DEFAULT = 10000;
    // minimum timeout for connect setup
    public static final long CONNECT_TIMEOUT_MINIMUM = 10000;

    // default timeout for socket connection setup (max per URL in connection URL list)
    public static final long SOCKET_CONNECT_TIMEOUT_DEFAULT = 0;
    // minimum timeout for socket connect 
    public static final long SOCKET_CONNECT_TIMEOUT_MINIMUM = 0;

    // subject prefix for MF JMS comms
    public static final String MF_SUBJECT_ROOT = "SonicMQ.mf.";
    // client ID prefix
    public static final String MF_CONNECTID_PREFIX = "SonicMQ/mf/";

    // service strings
    public static final String MF_DIRECTORY_SERVICE_STR = "DIRECTORY SERVICE";
    public static final String MF_AGENT_MANAGER_STR = "AGENT MANAGER";
    public static final String MF_DIRECTORY_SERVICE_ID = "ID=DIRECTORY SERVICE";
    public static final String MF_AGENT_MANAGER_ID = "ID=AGENT MANAGER";

    private static final int REQUEST_ARRAY_LENGTH = 4;

    public static final int REQUEST_ID_INDEX = 0;
    public static final int REQUEST_TARGET_INDEX = 1;
    public static final int REQUEST_ATTEMPT_COUNT_INDEX = 2;
    public static final int REQUEST_REPLY_MESSAGE_INDEX = 3;

    public static final int MAX_REQUEST_ATTEMPT_COUNT = 3;

    // message properties used in mapping

    // general prefix
    protected static final String JMS_PROPERTY_PREFIX = "JMS_SonicMQ_mf_";
    // comms type (e.g. mf (internal) or JMX)
    public static final String JMS_COMMS_TYPE_PROPERTY = JMS_PROPERTY_PREFIX + "comms_type";
    // content type (e.g. request, reply, etc.)
    public static final String JMS_CONTENT_TYPE_PROPERTY = JMS_PROPERTY_PREFIX + "content_type";
    public static final short NOTIFICATION_CONTENT_TYPE = 0;
    public static final short REQUEST_CONTENT_TYPE = 1;
    public static final short REPLY_CONTENT_TYPE = 2;
    // reply to subject
    public static final String JMS_REPLY_TO_PROPERTY = JMS_PROPERTY_PREFIX + "reply_subject";
    public static final String MY_MGMT_NODE_PROPERTY = JMS_PROPERTY_PREFIX + "my_mgmt_node";
    // oneway request (only provided when request is oneway .. i.e. no synchronous reply is required)
    public static final String JMS_ONEWAY_REQUEST_PROPERTY = JMS_PROPERTY_PREFIX + "oneway_request";
    // notification listener ID .. directed listeners provide this when they make a subscription
    public static final String JMS_LISTENER_ID_PROPERTY = JMS_PROPERTY_PREFIX + "listener_ID";
    // the target of the operation
    public static final String JMS_REQUEST_TARGET_PROPERTY = JMS_PROPERTY_PREFIX + "target";
    // the operation name
    public static final String JMS_REQUEST_OPERATION_PROPERTY = JMS_PROPERTY_PREFIX + "operation";
    // the original request timeout value
    public static final String JMS_REQUEST_TIMEOUT_PROPERTY = JMS_PROPERTY_PREFIX + "timeout";
    // the time the request was received
    public static final String JMS_REQUEST_RECEIVED_PROPERTY = JMS_PROPERTY_PREFIX + "received";
    // the time the request was completed/aborted
    public static final String JMS_REQUEST_REPLIED_PROPERTY = JMS_PROPERTY_PREFIX + "replied";

    public static final String NEWLINE = System.getProperty("line.separator");

    private static boolean DISABLE_FT_CLIENT;
    private static String DISABLE_FT_CLIENT_PROPERTY = "sonicsw.mf.disableFTClient";
    private static String FLOW_TO_DISK;
    private static String FLOW_TO_DISK_PROPERTY = "sonicsw.mf.flowToDisk";
    private static boolean REPLACE_IPADDRESS_OR_HOSTNAME_WITH_LOCALHOST;
    private static String REPLACE_IPADDRESS_OR_HOSTNAME_WITH_LOCALHOST_PROPERTY = "sonicsw.mf.replaceIPAddressOrHostNameWithLocalHost";

    private static boolean INFINITE_RETRY;
    private static String INFINITE_RETRY_PROPERTY = "sonicsw.mf.infiniteRetry";
    private static final ThreadLocal<SimpleDateFormat> DATE_PARSER_THREAD_LOCAL = new ThreadLocal<SimpleDateFormat>() {
        @Override
        protected SimpleDateFormat initialValue() {
            return new SimpleDateFormat("yy/MM/dd HH:mm:ss");
        }
    };

    static
    {
        String disableFTClient = System.getProperty(DISABLE_FT_CLIENT_PROPERTY, "false");
        DISABLE_FT_CLIENT = (disableFTClient != null && disableFTClient.equals("true"));
        FLOW_TO_DISK = System.getProperty(FLOW_TO_DISK_PROPERTY);

        String replaceWithLocalHost = (String) System.getProperty(REPLACE_IPADDRESS_OR_HOSTNAME_WITH_LOCALHOST_PROPERTY,"false");
        REPLACE_IPADDRESS_OR_HOSTNAME_WITH_LOCALHOST = (replaceWithLocalHost.equalsIgnoreCase("true")) ? true : false;

        String infiniteRetry = System.getProperty(INFINITE_RETRY_PROPERTY, "false");
        INFINITE_RETRY = (infiniteRetry != null && infiniteRetry.equals("true"));
        
        // sonicsw.mf.requestTimeoutOverride overrides sonicsw.mf.requestTimeoutDefault
        REQUEST_TIMEOUT_OVERRIDE = System.getProperty(REQUEST_TIMEOUT_OVERRIDE_PROPERTY);
        
        String debugTimeout = System.getProperty("debug.request.timeout");
        if (debugTimeout != null)
        {
            m_debugRequestTimeout = (new Boolean(debugTimeout)).booleanValue();
        }
    }

    private static long getRequestDefaultTimeOut() {
        
        String requestTimeoutDefault = System.getProperty(REQUEST_TIMEOUT_DEFAULT_PROPERTY);
        String overrideTimeOut = System.getProperty(REQUEST_TIMEOUT_OVERRIDE_PROPERTY);
        if (overrideTimeOut != null)
        {
            return (new Long(overrideTimeOut).longValue());
        }
        if (requestTimeoutDefault != null)
        {
            return (new Long(requestTimeoutDefault).longValue());
        }
        return 30000;
    }
    /**
     * Allocates a connector object intended for use by management clients
     * (e.g. JMX JMS connector client).
     */
    public ConnectorClient(String clientType)
    {
        String ipID = "0_0_0_0";
        try
        {
            ipID = InetAddress.getLocalHost().getHostAddress().replace('.', '_');
        } catch(Throwable e) { } // default already set

        // define a unique subscription subject for this client
        String uniqueID = clientType + '.' + ipID + '.' + new UID().toString();
        m_directedSubscriptionSubject = MF_SUBJECT_ROOT + uniqueID;
        m_directedReplyToSubject = m_directedSubscriptionSubject;
        uniqueID = uniqueID.replace('.', '/');
        m_subscriptionName = uniqueID;

        initialize(uniqueID);
        
        m_defaultRole = clientType;
    }
    
    /**
     * Allocates a connector object intended for use by management clients
     * (e.g. JMX JMS connector client).
     * 
     * This constructor is only intended to be used by unit tests, hence it is package private
     */
    
    ConnectorClient(IDurableConnectorFactory durableConnectorFactory, ITopicConnectionFactoryFactory topicConnectionFactoryFactory, String clientType)
    {
        this(clientType);
        m_durableConnectorFactory = durableConnectorFactory;
        m_topicConnectionFactoryFactory = topicConnectionFactoryFactory;
    }

    /**
     * Allocates a connector object and associates it with a container instance.
     * This is intended for use by the ConnectorServer class.
     *
     * @param containerIdentity The runtime identity of the container.
     */
    public ConnectorClient(IContainerIdentity containerIdentity)
    {
        m_domainName = containerIdentity.getDomainName();
        m_containerName = containerIdentity.getContainerName();

        // define a generic subscription for directed comms mapped on JMS (request/reply)
        String uniqueID = m_domainName + '.' + m_containerName;

        m_directedSubscriptionSubject = MF_SUBJECT_ROOT + "*." + uniqueID;
        m_directedReplyToSubject = MF_SUBJECT_ROOT + "." + uniqueID;
        uniqueID = m_domainName + '/' + m_containerName;
        m_subscriptionName = uniqueID;
        
        initialize(uniqueID);
        
        m_defaultRole = "MFCLIENT";
    }
    
    /**
     * Allocates a connector object and associates it with a container instance.
     * This is intended for use by the ConnectorServer class.
     * 
     * This constructor is only intended to be used by unit tests, hence it is package private
     * 
     * @param containerIdentity The runtime identity of the container.
     */
    
    ConnectorClient(IDurableConnectorFactory durableConnectorFactory, ITopicConnectionFactoryFactory topicConnectionFactoryFactory, IContainerIdentity containerIdentity)
    {
        this(containerIdentity);
        m_durableConnectorFactory = durableConnectorFactory;
        m_topicConnectionFactoryFactory = topicConnectionFactoryFactory;
    }
    
    private void initialize(String uniqueID)
    {
        m_logConnectionMessages = Boolean.getBoolean(IConnectorClient.MF_LOG_CONNECTION_MESSAGES_PROPERTY);

        // check to see if system property for tracing has been enabled (needed for client-side tracing)
        String tr = System.getProperty("enableConnectorClientConnectionTracing");
        if ((tr != null) && (tr.equalsIgnoreCase("true")))
        {
            DEBUG_TRACE_CONNECTION_STATE = true;
        }

        m_connectID = MF_CONNECTID_PREFIX + uniqueID;
    }
    
    public long getRequestTimeout() { return m_requestTimeout; }
    @Override
    public void setRequestTimeout(long milliseconds)
    {
    	// if the user has used the override, that has precedence over a value set
    	// programatically.
    	if (REQUEST_TIMEOUT_OVERRIDE == null)
    	{
            m_requestTimeout = (milliseconds >= REQUEST_TIMEOUT_MINIMUM) ? milliseconds : REQUEST_TIMEOUT_MINIMUM;
            setPingInterval();
    	}
    }

    private void setPingInterval()
    {
        // set the ping interval at the greater of 90% of the request timeout or 20 seconds
        int pingInterval = (int)((getRequestTimeout() * 9) / 10000);
        if (pingInterval < 20)
        {
            pingInterval = 20;
        }
        
        if (m_connectionFactory != null)
        {
            ((progress.message.jclient.TopicConnectionFactory)m_connectionFactory).setPingInterval(pingInterval);
        }
    }

    public long getConnectTimeout() { return m_connectTimeout; }
    public void setConnectTimeout(long milliseconds)
    {
        m_connectTimeout = (milliseconds >= CONNECT_TIMEOUT_MINIMUM) ? milliseconds : CONNECT_TIMEOUT_MINIMUM;

        // Ensure that the new connect timeout value is passed down
        // to the connection factories that are used to create JMS connections.
        try
        {
            if (m_connectionFactory != null)
            {
                setSonicMQFaultTolerance(m_connectionFactory);
            }
        }
        catch(Exception e) { } // should not occur
    }
    
    public void setDeliveryMode(int deliveryMode)
    {
    	m_deliveryMode = deliveryMode;
    }
    
    public void setDurable(boolean durable)
    {
    	m_isDurable = durable;
    }
    

    public long getSocketConnectTimeout() { return m_socketConnectTimeout; }
    public void setSocketConnectTimeout(long milliseconds)
    {
        if (milliseconds <= 0)
        {
            m_socketConnectTimeout = SOCKET_CONNECT_TIMEOUT_MINIMUM;
        }
        else
        if (milliseconds >= 2000)
        {
            m_socketConnectTimeout = milliseconds;
        }
        else
        {
            m_socketConnectTimeout = 2000;
        }
    }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#setManagementNode(String)
     */
    @Override
    public void setManagementNode(String node) { m_managementNode = node; }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#connect(Map, long)
     */
    @Override
    public void connect(Map properties, long timeout)
    throws Exception
    {
        // the connector is not reentrant .. i.e. you can't connect the same connector twice, you must create a
        // new instance
        if (m_durableConnector != null)
        {
            throw new IllegalStateException("Cannot attempt to connect same connector more than once");
        }
        
        // add in a unique ConnectID (override if needed)
        properties.put("ConnectID", m_connectID);

        // create a copy of the input map to use for the primary connection factory
        Map factoryProperties = (Map) new HashMap(properties);

        // create the connection factory
        m_connectionFactory = m_topicConnectionFactoryFactory.createConnectionFactory();

        // set the primary connection factory properties
        setConnectionFactoryProperties(m_connectionFactory, factoryProperties);

        // create the connector
        try
        {
            m_durableConnector = m_durableConnectorFactory.createDurableConnector(this, "Management", timeout);
            m_durableConnector.setDeliveryMode(m_deliveryMode);
        }
        catch (MFConnectAbortedException cae)
        {
            if (m_durableConnector != null)
            {
                m_durableConnector.cleanup();
            }

            // rethrow the exception
            throw cae;
        }
        catch (MFSecurityException se)
        {
            // clean up DurableConnector, if it exists
            if (m_durableConnector != null)
            {
                m_durableConnector.cleanup();
            }

            // invoke any registered IConnectionListener [ditto for IExceptionListener, though it is a deprecated interface]
            if (m_connectionListener != null)
            {
                m_connectionListener.onFailure(se);
            }
            if (m_exceptionListener != null)
            {
                m_exceptionListener.onException(se);
            }

            // rethrow the exception
            throw se;
        }


        if (Thread.interrupted()) // check if current thread has been interrupted
        {
            Thread.currentThread().interrupt(); // Since a call to Thread.interrupted() clears the interrupted status of the thread, invoke interrupt to set the status back to "interrupted" (and to notify any waiting threads via an InterruptedException)
        }
        else
        {
            if (timeout > 0 && !m_durableConnector.isConnected())
            {
                m_durableConnector.cleanup();
                throw new ConnectTimeoutException("Timeout occured while attempting to connect");
            }

            try
            {
                if (DEBUG_TRACE_CONNECTION_STATE)
                {
                    System.out.println("ConnectorClient.connect: subscription subject = " + m_directedSubscriptionSubject + ", subscription name = " + m_subscriptionName);
                }
                m_subscription = m_durableConnector.subscribe(m_directedSubscriptionSubject, m_isDurable ? this.m_subscriptionName : null, this, null);
                if (DEBUG_TRACE_CONNECTION_STATE)
                {
                    System.out.println("ConnectorClient.connect: connection successfully established, m_subscription = " + m_subscription);
                }
            }
            catch(JMSException e)
            {
                Exception le = ((JMSException)e).getLinkedException();
                if (le != null && (le instanceof ESecurityPolicyViolation || le instanceof ESecurityGeneralException || le instanceof EPasswordExpired || le instanceof EUnauthorizedClient))
                {
                    MFSecurityException se = new MFSecurityException("Authentication failure");
                    se.setLinkedException(le);
                    throw se;
                }
                throw e;
            }
        }
    }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#isConnected()
     */
    @Override
    public boolean isConnected()
    {
        if (m_durableConnector == null)
        {
            return false;
        }

        try
        {
            return m_durableConnector.isConnected();
        }
        catch(NullPointerException e)
        {
            return false;
        }
    }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#getExceptionListener()
     */
    @Override
    public IExceptionListener getExceptionListener() { return m_exceptionListener; }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#setExceptionListener(IExceptionListener)
     */
    @Override
    public final void setExceptionListener(IExceptionListener listener) { m_exceptionListener = listener; }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#getConnectionListener()
     */
    @Override
    public IConnectionListener getConnectionListener() { return m_connectionListener; }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#setConnectionListener(IConnectionListener)
     */
    @Override
    public final void setConnectionListener(IConnectionListener listener) { m_connectionListener = listener; }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#getOrphanedReplyListener()
     */
    @Override
    public IOrphanedReplyListener getOrphanedReplyListener() { return m_orphanedReplyListener; }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#setOrphanedReplyListener(IOrphanedReplyListener)
     */
    @Override
    public void setOrphanedReplyListener(IOrphanedReplyListener listener) { m_orphanedReplyListener = listener; }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#registerRetryCallback(IRetryCallback)
     */
    public synchronized void registerRetryCallback(IRetryCallback rcb)
    {
        if (m_retryCallback != null)
        {
            throw new IllegalArgumentException("Retry callback already registered; only a single callback may be registered");
        }

        m_retryCallback = rcb;

        m_retryCallbackManager = new RetryCallbackManager();

        if (DEBUG_TRACE_RETRY_CALLBACK)
        {
            System.out.println("ConnectorClient.registerRetryCallback: m_retryCallback = " + m_retryCallback);
        }
    }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#deregisterRetryCallback()
     */
    public synchronized void deregisterRetryCallback()
    {
        m_retryCallback = null;

        m_retryCallbackManager = null;

        if (DEBUG_TRACE_RETRY_CALLBACK)
        {
            System.out.println("ConnectorClient.deregisterRetryCallback: m_retryCallback = " + m_retryCallback);
        }
    }

    @Override
    public void closePending()
    {
        if (m_durableConnector != null)
        {
            m_durableConnector.cleanupPending();
        }
    }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#close()
     */
    @Override
    public void close()
    {
        if (m_durableConnector != null)
        {
            m_durableConnector.cleanup();
        }

        m_notificationListeners.clear();

        synchronized(m_activeRequests)
        {
            Iterator iterator = m_activeRequests.values().iterator();
            while (iterator.hasNext())
            {
                Object requestArray = iterator.next();
                synchronized(requestArray)
                {
                    requestArray.notifyAll();
                }
            }
            m_activeRequests.clear();
        }

        synchronized(this)
        {
            if (m_retryCallback != null)
            {
                synchronized(m_requestLock)
                {
                    if (m_retryCallbackManager.waiters > 0)
                    {
                        m_retryCallbackManager.callbackAction = IRetryCallback.CANCEL_REQUEST;
                        m_requestLock.notifyAll();
                    }
                }
                deregisterRetryCallback();
            }
        }
    }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#addDirectedNotificationListener(INotificationListener)
     * @see com.sonicsw.mf.common.IConsumer#close()
     */
    @Override
    public IConsumer addDirectedNotificationListener(INotificationListener listener)
    {
        NotificationListenerDelegate delegate = new NotificationListenerDelegate(listener);
        m_notificationListeners.put(Integer.valueOf(listener.hashCode()), delegate);

        // no subscription is required since the notifications will be directed to the regular
        // topic the client is listening on

        return delegate;
    }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#invoke(String, String, String, String, Object[], String[])
     */
    @Override
    public Object invoke(String commsType, String namespace, String target, String operationName, Object[] params, String[] signature)
    throws Exception
    {
        return invoke(commsType, namespace, target, operationName, params, signature, m_defaultRole, m_requestTimeout);
    }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#invoke(String, String, String, String, Object[], String[], long)
     */
    @Override
    public Object invoke(String commsType, String namespace, String target, String operationName, Object[] params, String[] signature, long timeout)
    throws Exception
    {
        return invoke(commsType, namespace, target, operationName, params, signature, m_defaultRole, timeout);
    }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#invoke(String, String, String, String, Object[], String[], String)
     */
    @Override
    public Object invoke(String commsType, String namespace, String target, String operationName, Object[] params, String[] signature, String role)
    throws Exception
    {
        return invoke(commsType, namespace, target, operationName, params, signature, role, m_requestTimeout);
    }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#invoke(String, String, String, String, Object[], String[], String, long)
     */
    @Override
    public Object invoke(String commsType, String namespace, String target, String operationName, Object[] params, String[] signature, String role, long requestTimeout)
    throws Exception
    {
        while (true)
        {
            long invokeStart = 0;
            long invokeRequestBuilt = 0, invokeRequestSent = 0, invokeReplyReceived = 0;
            if (DEBUG_TRACE_INVOCATION)
            {
                invokeStart = System.currentTimeMillis();
                invokeRequestBuilt = invokeRequestSent = invokeReplyReceived = invokeStart;
            }
            BytesMessage requestMessage = null;
            Object[] requestArray = null;
            String requestID = null;
            Object returnValue = null;

            // input request timeout of zero means use the default timeout
            // the override takes precedence over a value set programatically
            if ((requestTimeout <= 0) || (REQUEST_TIMEOUT_OVERRIDE != null))
            {
                requestTimeout = m_requestTimeout;
            }
            
            if (m_debugRequestTimeout  && !m_requestTimeoutPrinted)
            {
                boolean log = m_logConnectionMessages;
                m_logConnectionMessages = true;
                logMessage("ConnectorClient.invoke target == " + target + " using request timeout " + requestTimeout, Level.INFO);
                m_requestTimeoutPrinted = true;
                m_logConnectionMessages = log;
            }
            // determine overall timeout value (i.e. multiple requests may be made in the event of a failure)
            long overallTimeout = determineOverallTimeoutValue(requestTimeout);
            try
            {
                // setup the request
                try
                {
                    requestMessage = buildRequestMessage(commsType, target, operationName, params, signature);
                }
                catch (Exception e)
                {
                    if ((m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
                    {
                        logMessage("Failed to marshal request, trace follows...", e, Level.TRACE);
                    }
                    throw e;
                }
                catch (Error e)
                {
                    if ((m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
                    {
                        logMessage("Failed to marshal request, trace follows...", e, Level.TRACE);
                    }
                    throw e;
                }

                requestID = requestMessage.getJMSCorrelationID();

                if (DEBUG_TRACE_INVOCATION)
                {
                    System.out.println("ConnectorClient.invoke: requestID = " + requestID);
                }

                    // create the array of request information (to the proper sending of the request and response)
                requestArray = buildRequestArray(target, requestID);
                synchronized (m_activeRequests)
                {
                    //m_activeRequests.put(requestID, requestID);
                    m_activeRequests.put(requestID, requestArray);
                }

                if (DEBUG_TRACE_INVOCATION)
                {
                    System.out.println("ConnectorClient.invoke: added active request entry for requestID: " + requestID);
                }

                boolean isRequestAttemptComplete = false;
                long requestAttemptTimeout = 0;
                long requestAttemptStartTime = System.currentTimeMillis();

                // create the "appropriate" topic name to which the request will be published
                CanonicalName cName = new CanonicalName(namespace);
                String topicName = getNodePrefix(cName) + MF_SUBJECT_ROOT + role + '.' + cName.getDomainName() + '.' + cName.getContainerName(); 
                
                while (!isRequestAttemptComplete)
                {
                    synchronized (requestArray)
                    {
                        requestArray[REQUEST_REPLY_MESSAGE_INDEX] = null;
                    }

                    // check if we've exceeded the maximum allowable timeout to establish a connection...
                    // (the timeout check will not be performed if a retry callback has been registered,
                    // as the callback object will determine whether to keep trying or not)
                    if (DEBUG_TRACE_RETRY_CALLBACK)
                    {
                        System.out.println("ConnectorClient.invoke: m_retryCallback = " + m_retryCallback);
                    }
                    if (m_retryCallback == null)
                    {
                        if (System.currentTimeMillis() > (requestAttemptStartTime + overallTimeout))
                        {
                            if ((m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
                            {
                                logMessage("Timeout of synchronous invoke occurred while trying to establish management connection, target" + " - " + operationName + "()", Level.TRACE);
                            }

                            isRequestAttemptComplete = true;
                            cleanupRequestArray(requestArray);
                            throw new InvokeTimeoutException("Request timed out while trying to establish management connection, target" + " - " + operationName + "()");
                        }
                    }

                    // send the request
                    try
                    {
                        // determine the appropriate timeout value to provide to the "publish" invocation
                        requestAttemptTimeout = determineOverallTimeoutValue(requestTimeout);

                        if (DEBUG_TRACE_INVOCATION)
                        {
                            System.out.println("ConnectorClient.invoke: operationName = " + operationName + ", role = " + role);
                        }

                        // publish the request
                        requestMessage.setLongProperty(JMS_REQUEST_TIMEOUT_PROPERTY, requestAttemptTimeout);

                        if (DEBUG_TRACE_INVOCATION)
                        {
                            System.out.println("topic=" + topicName + ", target=" + target + ", operationName=" + operationName + ", replyTo=" + getReplyTo());
                        }

                        if (DEBUG_TRACE_INVOCATION)
                        {
                            invokeRequestBuilt = System.currentTimeMillis();
                        }
                        m_durableConnector.publish(topicName, (Message)requestMessage, requestAttemptTimeout, requestTimeout, requestArray);
                        if (DEBUG_TRACE_INVOCATION)
                        {
                            invokeRequestSent = System.currentTimeMillis();
                        }

                        if (DEBUG_TRACE_INVOCATION)
                        {
                            System.out.println("ConnectorClient.invoke: published request to topicName: " + topicName);
                        }
                    }
                    catch (InvokeTimeoutException e)
                    {
                        isRequestAttemptComplete = handleRequestAttemptFailure(target, operationName, e, cName);

                        if (DEBUG_TRACE_INVOCATION)
                        {
                            System.out.println("ConnectorClient.invoke: request cycle complete = " + isRequestAttemptComplete);
                        }

                        if (isRequestAttemptComplete)
                        {
                            // have exhausted attempts to submit request
                            throw e instanceof InvokeTimeoutCommsException ? new InvokeTimeoutCommsException(target + " - " + operationName + "()") : new InvokeTimeoutException(target + " - " + operationName + "()");
                        }
                        else {
                            continue; // back to top of "while"
                        }
                    }
                    catch (JMSException e)
                    {
                        isRequestAttemptComplete = true;
                        Exception le = e.getLinkedException();
                        if (DEBUG_TRACE_INVOCATION)
                        {
                            le.printStackTrace();
                        }
                        if (le != null && (le instanceof ESecurityPolicyViolation || le instanceof ESecurityGeneralException || le instanceof EPasswordExpired || le instanceof EUnauthorizedClient))
                        {
                            MFSecurityException se = new MFSecurityException("Authorization failure");
                            se.setLinkedException(le);
                            throw se;
                        }
                        throw e;
                    }
                    catch (Exception e)
                    {
                        if (DEBUG_TRACE_INVOCATION)
                        {
                            e.printStackTrace();
                        }
                        throw e;
                    }
                    catch (Error e)
                    {
                        if (DEBUG_TRACE_INVOCATION)
                        {
                            e.printStackTrace();
                        }
                        throw e;
                    }

                    // wait for the reply
                    BytesMessage replyMessage = null;
                    try
                    {
                        if (DEBUG_TRACE_INVOCATION)
                        {
                            System.out.println("ConnectorClient.invoke: preparing to wait for reply to requestID: " + requestID);
                        }

                        // incoming responses will replace the value of the entry in the request table
                        // so we can check that once we fall through the wait
                        synchronized (requestArray)
                        {
                            if (requestArray[REQUEST_REPLY_MESSAGE_INDEX] == null)
                            {
                                requestArray.wait(requestAttemptTimeout);
                            }
                            
                            replyMessage = (BytesMessage)requestArray[REQUEST_REPLY_MESSAGE_INDEX];
                        }
                        if (DEBUG_TRACE_INVOCATION)
                        {
                            invokeReplyReceived  = System.currentTimeMillis();
                        }
                        if (replyMessage == null) // timeout occurred (otherwise, response message field in request's object array would be non-null)
                        {
                            if (DEBUG_TRACE_INVOCATION)
                            {
                                System.out.println("ConnectorClient.invoke: timeout for requestID: " + requestID);
                            }

                            DurableConnector connector = m_durableConnector;
                            String message = target + " - " + operationName + "()";
                            InvokeTimeoutException ite = (connector == null || !connector.isConnected()) ? new InvokeTimeoutCommsException(message) : new InvokeTimeoutException(message);
                            isRequestAttemptComplete = handleRequestAttemptFailure(target, operationName, ite, cName);
                            if (isRequestAttemptComplete)
                            {
                                if ((m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
                                {
                                    StringBuffer msg = new StringBuffer("Timeout of synchronous invoke occurred");
                                    if ((m_traceMask & IConnectorClient.TRACE_DETAIL) > 0)
                                    {
                                        msg.append(", details...");
                                        msg.append("\n\ttarget = ").append(target);
                                        msg.append("\n\trequest = ").append(operationName);
                                    }
                                    logMessage(msg.toString(), Level.TRACE);
                                }
                                throw ite;
                            }
                            else {
                                continue; // back to the top...to try again
                            }
                        }
                    }
                    catch (InterruptedException e)
                    {
                        if (DEBUG_TRACE_INVOCATION)
                        {
                            System.out.println("ConnectorClient.invoke: InterruptedException thrown while waiting for reply...");
                        }
                    } // don't believe this needs to be handled

                    // unpack the response
                    try
                    {
                        returnValue = unpackResponse(replyMessage);
                    }
                    catch (Exception e)
                    {
                        isRequestAttemptComplete = true;
                        if ((m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
                        {
                            logMessage("Failed to unmarshal reply, trace follows...", e, Level.TRACE);
                        }
                        throw e;
                    }
                    catch (Error e)
                    {
                        isRequestAttemptComplete = true;
                        if ((m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
                        {
                            logMessage("Failed to unmarshal reply, trace follows...", e, Level.TRACE);
                        }
                        throw e;
                    }

                    // check for "MFServiceNotActiveException", so that we can redirect the request if necessary (if FT enabled)
                    // only a service request could result in this exception being returned...
                    if (returnValue instanceof MFServiceNotActiveException)
                    {
                        if (DEBUG_TRACE_INVOCATION)
                        {
                            System.out.println("ConnectorClient.invoke: received MFServiceNotActiveException while publishing service request, set up for next FT attempt, requestID: " + requestID);
                        }

                        isRequestAttemptComplete = handleRequestAttemptFailure(target, operationName, (MFRuntimeException)returnValue, cName);
                        if (isRequestAttemptComplete)
                        {
                            throw new MFServiceNotActiveException("Requested service not in active state, target: " + target);
                        }
                        else {
                            continue; // back to the top...to try again
                        }
                    }
                    else if (returnValue instanceof Error) // unexpected fatal condition - need to re-throw
                    {
                        isRequestAttemptComplete = true;
                        if ((m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
                        {
                            logMessage("Error thrown during attempted operation, trace follows...", (Throwable)returnValue, Level.TRACE);
                        }
                        throw (Error)returnValue;
                    }
                    else if (returnValue instanceof Exception)
                    {
                        isRequestAttemptComplete = true;
                        if ((m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
                        {
                            logMessage("Exception thrown during attempted operation, trace follows...", (Throwable)returnValue, Level.TRACE);
                        }
                        throw (Exception)returnValue;
                    }
                    else
                    {
                        // N.B.  Apparently the "return value" obtained from a response message
                        //       may be "null" (or "void").  Unless the request message contains
                        //       some field to indicate that the expected return value may be
                        //       "null", there is no easy way to figure out if a return value of
                        //       "null" is expected.  So, if the return value isn't one of the
                        //       Exception types checked above, we'll have to assume success, and
                        //       exit the "while" loop...
                        if (DEBUG_TRACE_INVOCATION)
                        {
                            System.out.println("ConnectorClient.invoke: successfully received proper response to request, requestID: " + requestID);
                        }

                            // must have received a proper response if this point is reached.  Need to get out of
                            // the while loop, and return the response.
                        isRequestAttemptComplete = true;
                        break;
                    }

                } // end of "while"
            }
            catch(InvokeTimeoutException e)
            {
                if (INFINITE_RETRY)
                {
                    logMessage("Request failure, retrying...", Level.WARNING);
                    continue;
                }
                throw e;
            }
            finally
            {
                synchronized (m_activeRequests)
                {
                    requestArray = (Object[])m_activeRequests.remove(requestID);
                    cleanupRequestArray(requestArray);
                }
            }
            if (DEBUG_TRACE_INVOCATION)
            {
                long invokeDone = System.currentTimeMillis();
                System.out.println("Invoke request for operation " + operationName + 
                        ".\n\ttook " + (invokeDone - invokeStart) + " ms." +
                        "\n\tbuilding the request took " + (invokeRequestBuilt - invokeStart) + " ms." +
                        "\n\tsending the request took " + (invokeRequestSent - invokeRequestBuilt) + " ms." +
                        "\n\treceiving the reply took " + (invokeReplyReceived - invokeRequestSent) + " ms." +
                        "\n\tunpacking the reply took " + (invokeDone- invokeReplyReceived) + " ms."
                );
            }
            return returnValue;
        }
    }

    /**
     * Invoke a management operation on the specified component and if indicated do not
     * wait for the response. This can only be invoked where the operation returns void and
     * the caller does not need to have synchronous knowledge that the operation
     * failed (when failure occurs). Otherwise this method functions in a similar fashion
     * its synchronous counterpart.
     *
     * @see ConnectorClient#invoke(String, String, String, String, Object[], String[], long)
     */
    public void invokeOneway(String commsType, String namespace, String target, String operationName, Object[] params, String[] signature, Long timeout)
    throws InvokeTimeoutException, Exception
    {
        invokeOneway(commsType, namespace, target, operationName, params, signature, timeout.longValue());
    }

    /**
     * Invoke a management operation on the specified component and if indicated do not
     * wait for the response. This can only be invoked where the operation returns void and
     * the caller does not need to have synchronous knowledge that the operation
     * failed (when failure occurs). Otherwise this method functions in a similar fashion
     * its synchronous conterpart.
     *
     * @see ConnectorClient#invoke(String, String, String, String, Object[], String[], long)
     * @see ConnectorClient#invokeOneway(String, String, String, String, Object[], String[], long, long)
     */

    public void invokeOneway(String commsType, String namespace, String target, String operationName, Object[] params, String[] signature, long timeout)
    throws InvokeTimeoutException, Exception
    {
        invokeOneway(commsType, namespace, target, operationName, params, signature, timeout, timeout);  // use the request timeout value as the time-to-live value
    }

    /**
     * Invoke a management operation on the specified component and if indicated do not
     * wait for the response. This can only be invoked where the operation returns void and
     * the caller does not need to have synchronous knowledge that the operation
     * failed (when failure occurs). Otherwise this method functions in a similar fashion
     * its synchronous counterpart.
     *
     * @see ConnectorClient#invoke(String, String, String, String, Object[], String[], long)
     */
    public void invokeOneway(String commsType, String namespace, String target, String operationName, Object[] params, String[] signature, long timeout, long ttl)
    throws InvokeTimeoutException, Exception
    {
        BytesMessage requestMessage = null;
        String requestID = null;
        Object[] requestArray = null;

        CanonicalName cName = new CanonicalName(namespace);
        String topicName = getNodePrefix(cName) + MF_SUBJECT_ROOT + m_defaultRole + '.' + cName.getDomainName() + '.' + cName.getContainerName();

        // Can simply invoke DC's "getStateDependentPublisher" method here, supplying a dummy request ID
        requestID = "2";  // dummy requestID
        requestArray = buildRequestArray(target, requestID);

        // timeout of zero means use the default timeout
        // always use the override if the user set it
        if ((timeout == 0) || (REQUEST_TIMEOUT_OVERRIDE != null))
        {
            timeout = m_requestTimeout;
        }
        
        if (m_debugRequestTimeout  && !m_requestTimeoutPrinted)
        {
            boolean log = m_logConnectionMessages;
            m_logConnectionMessages = true;
            logMessage("ConnectorClient.invoke target == " + target + " using request timeout " + timeout, Level.INFO);
            m_requestTimeoutPrinted = true;
            m_logConnectionMessages = log;
        }
        // setup the request
        try
        {
            requestMessage = buildRequestMessage(commsType, target, operationName, params, signature);
            requestMessage.setBooleanProperty(JMS_ONEWAY_REQUEST_PROPERTY, true);
        }
        catch(Throwable e)
        {
            if ((m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
            {
                logMessage("Failed to marshal (oneway) request, trace follows...", e, Level.TRACE);
            }
            throw new Exception("Failed to marshal (oneway) request.");
        }

        // send the request
        try
        {
            m_durableConnector.publish(topicName, requestMessage, timeout, ttl, requestArray);
        }
        catch(InvokeTimeoutCommsException e)
        {
            throw new InvokeTimeoutCommsException(cName + " : " + operationName + "()");
        }
        catch(InvokeTimeoutException e)
        {
            throw new InvokeTimeoutException(cName + " : " + operationName + "()");
        }
        catch(Throwable e)
        {
            if ((m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
            {
                logMessage("Failed to publish (oneway) request, trace follows...", e, Level.TRACE);
            }
            
            // TODO: Tempted to set cause, but nervous that this could lead to issues with
            // ClassNotFoundException if the exception gets serialized to another component.
            // Playing it safe for now since I've not had time to investigate. 
            throw new Exception("Failed to publish (oneway) request: " + e);
        }
    }

    public void dumpRequestArray(Object[] requestArray)
    {
        System.out.println("ConnectorClient.dumpRequestArray: ");
        System.out.println("ConnectorClient.dumpRequestArray: requestID = " + requestArray[ConnectorClient.REQUEST_ID_INDEX]);
        System.out.println("ConnectorClient.dumpRequestArray: reply message = " + requestArray[ConnectorClient.REQUEST_REPLY_MESSAGE_INDEX]);  // this is the field that will contain a reference to the reply message (if the request is successfully processed)
        System.out.println("ConnectorClient.dumpRequestArray: target = " + requestArray[ConnectorClient.REQUEST_TARGET_INDEX]);
        System.out.println("ConnectorClient.dumpRequestArray: request attempt count = " + requestArray[ConnectorClient.REQUEST_ATTEMPT_COUNT_INDEX]);
    }

    private void cleanupRequestArray(Object[] requestArray)
    {
        if (requestArray != null)
        {
            for (int i = 0; i < requestArray.length; i++)
            {
                requestArray[i] = null;
            }
        }
    }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#getTraceMask()
     */
    @Override
    public int getTraceMask() { return m_traceMask; }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#setTraceMask(int)
     */
    @Override
    public void setTraceMask(int traceMask)
    {
        m_traceMask = traceMask;
        if (m_traceMask < 0)
        {
            m_traceMask = 0;
        }
    }

    @Override
    public String getLocalRoutingNodeName()
    {
        String node = null;

        if (m_durableConnector != null)
        {
            node = m_durableConnector.getLocalRoutingNodeName();
        }
        if (node == null && m_containerName != null)
        {
            int index = m_containerName.indexOf('@');
            if (index > 0 && ++index < m_containerName.length())
            {
                node = m_containerName.substring(index);
            }
        }
        return node;
    }
    
    @Override
    public boolean isConnectionEnterpriseEnabled()
    {
        if (m_durableConnector != null) {
            return m_durableConnector.isConnectionEnterpriseEnabled();
        } else {
        	return false;
        }
    }
    
    public String getReplyTo()
    {
        StringBuffer sb = new StringBuffer();
        String localRoutingNodeName = getLocalRoutingNodeName();
        if (localRoutingNodeName != null)
        {
            sb.append(localRoutingNodeName).append("::");
        }
        sb.append(m_directedReplyToSubject);
        return sb.toString();
    }

    public static String getPseudoContainerID(String globalID, String instanceID)
    {
        StringBuffer pseudoContainerID = new StringBuffer();
        if (instanceID != null)
        {
            pseudoContainerID.append('[');
            pseudoContainerID.append(instanceID);
            pseudoContainerID.append(']');
        }
        pseudoContainerID.append(globalID);

        return pseudoContainerID.toString();
    }

    /**
     * Build a standard request message. Takes same args as passed to invoke() methods.
     */
    protected BytesMessage buildRequestMessage(String commsType, String target, String operationName, Object[] params, String[] signature)
    throws Exception
    {
        BytesMessage requestMessage = null;
        String requestID = new UID().toString();

        // create the message
        requestMessage = new progress.message.jimpl.BytesMessage();

        // setup the header/properties
        requestMessage.setJMSCorrelationID(requestID);

        if (m_managementNode != null)
        {
            requestMessage.setStringProperty(MY_MGMT_NODE_PROPERTY, m_managementNode);
        }

        requestMessage.setStringProperty(JMS_REPLY_TO_PROPERTY, getReplyTo());
        requestMessage.setStringProperty(JMS_COMMS_TYPE_PROPERTY, commsType);
        requestMessage.setShortProperty(JMS_CONTENT_TYPE_PROPERTY, REQUEST_CONTENT_TYPE);
        requestMessage.setStringProperty(JMS_REQUEST_TARGET_PROPERTY, target);
        requestMessage.setStringProperty(JMS_REQUEST_OPERATION_PROPERTY, operationName);

        if (DEBUG_TRACE_INVOCATION)
        {
            System.out.println("ConnectorClient.buildInvokeRequest: replyTo = " + requestMessage.getStringProperty(JMS_REPLY_TO_PROPERTY) +
                                          ", MY_MGMT_NODE = " + requestMessage.getStringProperty(MY_MGMT_NODE_PROPERTY) +
                                          ", target = " + requestMessage.getStringProperty(JMS_REQUEST_TARGET_PROPERTY) +
                                          ", operation = " + requestMessage.getStringProperty(JMS_REQUEST_OPERATION_PROPERTY) +
                                          " for requestID = " + requestID);
        }


        // fill the body
        requestMessage.writeBoolean(signature.length == 0); // write true if its an empty payload
        if (signature.length > 0)
        {
            ByteArrayOutputStream arrayOut = new ByteArrayOutputStream(5120);
            synchronized(arrayOut) // performance optimization
            {            
                ObjectOutputStream out = new ObjectOutputStream(arrayOut);
                out.writeObject(signature);
                out.writeObject(params);
                out.flush();
                requestMessage.writeInt(arrayOut.size());
                requestMessage.writeBytes(arrayOut.toByteArray());
                out.close();  // close the stream
            }
        }
        
        return requestMessage;
    }

    protected String getNodePrefix(CanonicalName cName)
    {
        // does the container have a remote node destination in it, if so put that into the node prefix
        String targetContainer = cName.getContainerName();
        int index = targetContainer.indexOf('@');
        if (index > 0 && ++index < targetContainer.length())
        {
            return targetContainer.substring(index) +  "::";
        }
        
        // (else) is the management node defined, if so then use that as a prefix
        if (m_managementNode != null)
        {
            return m_managementNode +  "::";
        }

        // (else) no node prefix required
        return "";
    }

    @Override
    public void logMessage(String message, int severityLevel)
    {
        if (m_logConnectionMessages)
        {
            StringBuffer timestampedMessage = new StringBuffer();

            timestampedMessage.append('[');
            timestampedMessage.append(DATE_PARSER_THREAD_LOCAL.get().format(new Date(System.currentTimeMillis())));
            timestampedMessage.append("]");
            timestampedMessage.append(" (");
            timestampedMessage.append(Level.LEVEL_TEXT[severityLevel]);
            timestampedMessage.append(") ");
            timestampedMessage.append(message);

            System.err.println(timestampedMessage.toString());
            System.err.flush();
        }
    }

    @Override
    public void logMessage(String message, Throwable exception, int severityLevel)
    {
        StringBuffer buffer = new StringBuffer(message);
        buffer.append(NEWLINE);
        ByteArrayOutputStream out = new ByteArrayOutputStream(128);
        synchronized(out) // performance optimization
        {
            PrintWriter writer = new PrintWriter(out, true);
            printStackTrace(exception, writer);
            byte[] data = out.toByteArray();
    
            try
            {
                out.close();
            }
            catch(IOException io){};

            buffer.append(new String(data));
        }
        
        logMessage(buffer.toString(), severityLevel);
    }

    @Override
    public synchronized TopicConnectionFactory getConnectionFactory() { return m_connectionFactory; }

    public synchronized boolean isInContainer() { return m_inContainer; }

    private void setSonicMQFaultTolerance(TopicConnectionFactory tcFactory)
    throws RuntimeException, NoSuchMethodException, InvocationTargetException, IllegalAccessException
    {
        if (!DISABLE_FT_CLIENT)
        {
            // enable SonicMQ (as opposed to MF) Fault Tolerance for the TopicConnectionFactory
            Boolean b = Boolean.TRUE;
            Method setter = null;
            setter = progress.message.jclient.TopicConnectionFactory.class.getMethod("setFaultTolerant", new Class[]{ b.getClass() });
            setter.invoke(tcFactory, new Object[]{ b });

            // set the initial and reconnect timeout
            Integer timeout = Integer.valueOf((int)m_connectTimeout / 1000);
            setter = progress.message.jclient.TopicConnectionFactory.class.getMethod("setFaultTolerantReconnectTimeout", new Class[]{ timeout.getClass() });
            setter.invoke(tcFactory, new Object[]{ timeout });
            setter = progress.message.jclient.TopicConnectionFactory.class.getMethod("setInitialConnectTimeout", new Class[]{ timeout.getClass() });
            setter.invoke(tcFactory, new Object[]{ timeout });
        }
    }

    private void setConnectionFactoryProperties(TopicConnectionFactory factory, Map factoryProperties)
    throws NoSuchMethodException, NoSuchFieldException, InvocationTargetException, IllegalAccessException
    {
        // set the monitor interval (for flow control debugging) if specified as a System property....
        String monitorInterval = System.getProperty("monitorInterval");
        if (monitorInterval != null)
        {
            int val = Integer.parseInt(monitorInterval);
            if (val >= 0)
            {
                ((progress.message.jclient.TopicConnectionFactory)factory).setMonitorInterval(Integer.valueOf(val));
            }
        }

        Method setter = null;

        // force load balancing on (default) .. it will get switched off if the user checked off for either
        // a container or SMC connection
        ((progress.message.jclient.TopicConnectionFactory)factory).setLoadBalancing(true);

        // Now set, through reflection, the rest of the parameters
        Iterator iterator = factoryProperties.entrySet().iterator();

        while (iterator.hasNext())
        {
            Map.Entry entry = (Map.Entry)iterator.next();
            try
            {
                String name = (String)entry.getKey();
                Object value = entry.getValue();
                if (name.equals(IContainerConstants.CONNECTIONURLS_ATTR))
                {
                    value = fixupURLs((String)value);
                }
                if (name.startsWith("LoadBalancing"))
                {
                    name = "LoadBalancingBoolean";
                }
                if(name.startsWith("EnableCompression"))
                {
                    name = "EnableCompressionBoolean";
                }
                setter = factory.getClass().getMethod("set" + name, new Class[]{ value.getClass() });
                setter.invoke(factory, new Object[]{ value });
            }
            catch (NoSuchMethodException e)
            {
                throw new NoSuchMethodException("The TopicConnectionFactory class does not have a '" +
                                                 (String)entry.getKey() + "' " + entry.getValue().getClass().getName() +
                                                 " property.");
            }
        }

        // Set params that can't be changed
        ((progress.message.jclient.TopicConnectionFactory)factory).setSequential(true);
        ((progress.message.jclient.TopicConnectionFactory)factory).setPersistentDelivery(true);
        
        ((progress.message.jclient.TopicConnectionFactory)factory).setInitialConnectTimeout(Integer.valueOf((int)m_connectTimeout / 1000));
        ((progress.message.jclient.TopicConnectionFactory)factory).setSocketConnectTimeout(Integer.valueOf((int)m_socketConnectTimeout));
        
        // Sonic00039954 Set the ping interval in the factory
        setPingInterval();

        // force (SonicMQ, as opposed to MF) FT to be enabled for the "client"
        setSonicMQFaultTolerance(factory);

        // force flow-to-disk feature on for MF comms, unless specifically disabled (use reflection in order to allow this to work with pre-6.0 versions of SonicMQ)
        Field flowToDiskOn = progress.message.jclient.Constants.class.getField("FLOW_TO_DISK_ON");
        Field flowToDiskOff = progress.message.jclient.Constants.class.getField("FLOW_TO_DISK_OFF");
        Field flowToDiskUseBrokerSetting = progress.message.jclient.Constants.class.getField("FLOW_TO_DISK_USE_BROKER_SETTING");
        int onVal = flowToDiskOn.getInt(null);  // sets FLOW_TO_DISK "on" for the subscriber despite the Broker's setting
        int offVal = flowToDiskOff.getInt(null);  // sets FLOW_TO_DISK "off" for the subscriber despite the Broker's setting
        int useBrokerVal = flowToDiskUseBrokerSetting.getInt(null);  // sets FLOW_TO_DISK such that Broker's setting for this property is used for the subscriber
        int flowToDiskVal = onVal;  // sets FLOW_TO_DISK "on" for the subscriber despite the Broker's setting
        if (FLOW_TO_DISK != null)
        {
            flowToDiskVal = Integer.parseInt(FLOW_TO_DISK);
            if ((flowToDiskVal != onVal) && (flowToDiskVal != offVal) && (flowToDiskVal != useBrokerVal))
            {
                flowToDiskVal = onVal;
            }
        }
        ((progress.message.jclient.TopicConnectionFactory)factory).setFlowToDisk(Integer.valueOf(flowToDiskVal));
    }

    public long determineOverallTimeoutValue(long requestTimeout)
    {
        if (m_durableConnector.isConnected())
        {
            return requestTimeout;
        }
        return requestTimeout + m_connectTimeout;  // allow additional time for connection to be set up...
    }

    public Object[] buildRequestArray(String target, String requestID)
    {
        if (DEBUG_TRACE_REQUEST)
        {
            System.out.println("ConnectorClient.buildRequestArr: target = " + target + ", requestID: " + requestID);
        }

        Object[] requestArray = new Object[ REQUEST_ARRAY_LENGTH ];
        requestArray[ConnectorClient.REQUEST_ID_INDEX] = (Object)requestID;
        requestArray[ConnectorClient.REQUEST_REPLY_MESSAGE_INDEX] = null;
        requestArray[ConnectorClient.REQUEST_TARGET_INDEX] = target;
        requestArray[ConnectorClient.REQUEST_ATTEMPT_COUNT_INDEX] = Short.valueOf((short)1);

        return requestArray;
    }

    private boolean handleRequestAttemptFailure(String target, String operationName, MFRuntimeException e, CanonicalName cName)
    throws InvokeTimeoutException, RuntimeException
    {
        if (DEBUG_TRACE_RETRY_CALLBACK)
        {
            System.out.println("ConnectorClient.handleRequestAttemptFailure: m_retryCallback = " + m_retryCallback);
        }

        if (m_retryCallback == null)
        {
            return true;  // no more attempts to be made for request
        }
        else
        {
            return testRetryCallback(cName, operationName, e);
        }
    }

    private boolean testRetryCallback(CanonicalName cName, String operationName, Exception e)
    throws InvokeTimeoutException, RuntimeException
    {
        boolean requestCycleComplete = false;

        if (DEBUG_TRACE_RETRY_CALLBACK)
        {
            System.out.println("ConnectorClient.testRetryCallback: cName = " + cName.getCanonicalName());
        }

        Object lock = m_requestLock;
        RetryCallbackManager manager = m_retryCallbackManager;

        boolean performCallback = false;
        try
        {
            synchronized (lock)
            {
                manager.waiters++;
                if (manager.waiters == 1)
                {
                    performCallback = true;
                }
            }
            if (performCallback)
            {
                short callbackAction = performCallback(cName, e);
                synchronized(lock)
                {
                    manager.callbackAction = callbackAction;
                    lock.notifyAll();
                }
            }
            else
            {
                synchronized(lock)
                {
                    while (manager.callbackAction == -1)
                    {
                        try
                        {
                            lock.wait();
                        }
                        catch (InterruptedException ie)
                        {
                            if (DEBUG_TRACE_RETRY_CALLBACK)
                            {
                                System.out.println("ConnectorClient.testRetryCallback: InterruptedException ignored");
                            }
                        }
                    }
                }
            }
        }
        finally
        {
            synchronized (lock)
            {
                try
                {
                    requestCycleComplete = manager.callbackAction == IRetryCallback.CANCEL_REQUEST ? true : false;

                    if (DEBUG_TRACE_RETRY_CALLBACK)
                    {
                        System.out.println("ConnectorClient.testRetryCallback: callbackAction = " + manager.callbackAction + ", requestCycleComplete = " + requestCycleComplete);
                    }
                }
                finally
                {
                    manager.waiters--;
                    if (manager.waiters == 0)
                     {
                        manager.callbackAction = -1; // reset it for next time
                    }
                }
            }
        }

        return requestCycleComplete;
    }

    private short performCallback(CanonicalName cName, Exception e)
    {
        // determine the acceptable return values for the callback
        short[] acceptableReturnValues = null;
        if (DEBUG_TRACE_RETRY_CALLBACK)
        {
            System.out.println("ConnectorClient.performCallback: cName = " + cName.getCanonicalName());
        }
        acceptableReturnValues = new short[2];
        acceptableReturnValues[0] = IRetryCallback.CANCEL_REQUEST;
        acceptableReturnValues[1] = IRetryCallback.RETRY_REQUEST;

        // get the "current" connection URL(s) [the ones used for the current connection or connection attempt]
        String currentConnectionURLs = ((progress.message.jclient.TopicConnectionFactory)getConnectionFactory()).getConnectionURLs();

        if (DEBUG_TRACE_RETRY_CALLBACK)
        {
            System.out.println("ConnectorClient.performCallback: current connection URLs = " + currentConnectionURLs);
        }

        String targetName = cName.getContainerName().endsWith(cName.getComponentName()) ? cName.getContainerName() : cName.getCanonicalName();

        if (DEBUG_TRACE_RETRY_CALLBACK)
        {
            System.out.println("ConnectorClient.performCallback: formulated targetName = " + targetName);
        }

        // invoke the callback
        short callbackAction = m_retryCallback.onRequestFailure(targetName, currentConnectionURLs, e, acceptableReturnValues);

        if (DEBUG_TRACE_RETRY_CALLBACK)
        {
            System.out.println("ConnectorClient.handleCallback: returned action = " + callbackAction);
        }

        // make sure that the returned value is one of the acceptable values
        // Note: IRetryCallback.RETRY_REQUEST_FAILOVER is deprecated but we must still support for now
        if (callbackAction < IRetryCallback.CANCEL_REQUEST || callbackAction > IRetryCallback.RETRY_REQUEST_FAILOVER)
        {
            throw new RuntimeException("Invalid action specified for retry of failed request: request will be cancelled.");
        }

        return callbackAction;
    }

    private void printStackTrace(Throwable exception, PrintWriter writer)
    {
        exception.printStackTrace(writer);
        if (exception instanceof InvocationTargetException)
        {
            printStackTrace(((InvocationTargetException)exception).getTargetException(), writer);
        }
    }

    //
    // MessageListener interface
    //

    /**
     * Receive either:
     *
     *  - a reply (mapped over JMS) and match it with the originating request
     *  - a directed notification and match that with a listener
     */
    @Override
    public void onMessage(Message message)
    {
        if (DEBUG_TRACE_REQUEST)
        {
            System.out.println("ConnectorClient.onMessage: reply received...");
        }

        // ignore duplicates
        if (isRedelivered(message))
        {
            return;
        }

        try
        {
            short contentType = message.getShortProperty(JMS_CONTENT_TYPE_PROPERTY);
            // handle based on message content type
            switch(contentType)
            {
                case(REPLY_CONTENT_TYPE):
                    if (DEBUG_TRACE_REQUEST)
                    {
                        System.out.println("ConnectorClient.onMessage: reply received, BytesMessage...");
                    }
                    handleReplyMessage((BytesMessage)message);
                    break;
                case(NOTIFICATION_CONTENT_TYPE):
                    if (DEBUG_TRACE_REQUEST)
                    {
                        System.out.println("ConnectorClient.onMessage: reply received, DirectedNotification...");
                    }
                    handleDirectedNotification(message);
                    break;
                case(REQUEST_CONTENT_TYPE):
                    logMessage("Request content type " + message.getJMSDestination().toString(), Level.WARNING);
                    break;
                default:
                    if (DEBUG_TRACE_REQUEST)
                    {
                        System.out.println("ConnectorClient.onMessage: reply received, unknown content type...");
                    }
                    logMessage("Unknown message content type: " + contentType, Level.WARNING);
            }
        }
        catch(Throwable e)
        {
            if ((m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
            {
                logMessage("Failed to identify message content type, trace follows", e, Level.TRACE);
            }
        }
    }

    // We set the session to be DUPS OK .. and we should ignore duplicates we are sent
    protected boolean isRedelivered(Message message)
    {
        try
        {
            return message.getJMSRedelivered();
        }
        catch(JMSException e)
        {
            if ((m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
            {
                logMessage("Bad MF JMS message redelivered (dropping message), trace follows...", e, Level.TRACE);
            }
            return true;
        }
    }

    private void handleReplyMessage(BytesMessage message)
    {
        String requestID = null;
        try
        {
            requestID = message.getJMSCorrelationID();

            Object[] requestArray = null;

            synchronized(m_activeRequests)
            {
                requestArray = (Object[])m_activeRequests.get(requestID);

                if (DEBUG_TRACE_REQUEST)
                {
                    System.out.println("ConnectorClient.handleReplyMessage: requestArray = " + Arrays.toString(requestArray) + " for requestID:" + requestID);
                }

                if (requestArray == null)
                {
                    IOrphanedReplyListener listener = m_orphanedReplyListener;
                    if (listener != null)
                    {
                        handleOrphanedReplyMessage(listener, message);
                    }
                    return; // ignore as the request has timed out or otherwise been canceled
                }
            }

            synchronized(requestArray)
            {
                requestArray[REQUEST_REPLY_MESSAGE_INDEX] = (Object)message;

                requestArray.notifyAll();
            }

            if (DEBUG_TRACE_REQUEST)
            {
                System.out.println("ConnectorClient.handleReplyMessage: notified waiter of requestArray object...");
            }
        }
        catch(JMSException e)
        {
            if ((m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
            {
                logMessage("Failed to identify reply ID, trace follows...", e, Level.TRACE);
            }
        }
        catch(Throwable e)
        {
            if ((m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
            {
                logMessage("Failed to handle reply, trace follows...", e, Level.TRACE);
            }
        }
    }

    private Object unpackResponse(BytesMessage replyMessage)
    throws Exception
    {
        if (replyMessage == null || replyMessage.readBoolean())
        {
            return null;
        }

        // THST
        ClassLoader loader = Thread.currentThread().getContextClassLoader();
        byte[] bytes = new byte[replyMessage.readInt()];
        replyMessage.readBytes(bytes);
        ByteArrayInputStream arrayIn = new ByteArrayInputStream(bytes);

        synchronized(arrayIn) // performance optimization
        {
            ObjectInputStream in = null;

            // if the container name is not null then this is the JMSConnectorServer that extends this class and
            // class loader xfers are done elsewhere
            // if the loader is null since the client did not use setContextClassLoader() (or by some chance
            // the loader is the same as the connectors loader) then regular deserialization can occur
            if (m_containerName != null || loader == null || loader == m_loader)
            {
                in = new ObjectInputStream(arrayIn);
            }
            else
            {
                in = new LoaderInputStream(arrayIn, loader);
            }
            
            return in.readObject();
        }
    }

    private void handleOrphanedReplyMessage(IOrphanedReplyListener listener, BytesMessage replyMessage)
    {
        String target = null;
        String operationName = null;
        long requestReceived = 0;
        long requestReplied = 0;
        Object returnValue = null;

        try
        {
            target = replyMessage.getStringProperty(JMS_REQUEST_TARGET_PROPERTY);
            operationName = replyMessage.getStringProperty(JMS_REQUEST_OPERATION_PROPERTY);
            requestReceived = replyMessage.getLongProperty(JMS_REQUEST_RECEIVED_PROPERTY);
            requestReplied = replyMessage.getLongProperty(JMS_REQUEST_REPLIED_PROPERTY);
            returnValue = unpackResponse(replyMessage);
        }
        catch(Exception e)
        {
            if ((m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
            {
                logMessage("Failed to unmarshal orphaned reply, trace follows...", e, Level.TRACE);
            }
        }

        if (returnValue instanceof Exception)
        {
            listener.onFailure(target, operationName, (Exception)returnValue, requestReceived, requestReplied);
        }
        else
        {
            listener.onSuccess(target, operationName, returnValue, requestReceived, requestReplied);
        }
    }

    private void handleDirectedNotification(Message message)
    {
        try
        {
            // extract the target listener
            Integer listenerHash = Integer.valueOf(message.getIntProperty(JMS_LISTENER_ID_PROPERTY));
            NotificationListenerDelegate delegate = (NotificationListenerDelegate)m_notificationListeners.get(listenerHash);
            delegate.onMessage(message);
        }
        catch(Throwable e)
        {
            if ((m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
            {
                logMessage("Failed to identify notification listener (dropping notification), trace follows...", e, Level.TRACE);
            }
        }
    }

    @Override
    public void onFailure(Exception e)
    {
        close();
        if (m_connectionListener != null)
        {
            m_connectionListener.onFailure(e);
        }
        if (m_exceptionListener != null)
        {
            m_exceptionListener.onException(e);
        }
    }

    @Override
    public void onReconnect(String localRoutingNode)
    {
        if (m_connectionListener != null)
        {
            m_connectionListener.onReconnect(localRoutingNode);
        }
    }

    @Override
    public void onDisconnect()
    {
        if (m_connectionListener != null)
        {
            m_connectionListener.onDisconnect();
        }
    }

    // Sonic00009105 Request is to allow more flexibility from the connector than we allow
    //               in the underlying connection factory. Issue is folks entering
    //               "<url>, <url>", whilst the connection factory does not allow spaces
    //               in the comma delimited list.
    // Sonic00017435 Replace hosts that evaluate to the local IP address with "localhost"
    //               to avoid the issue exposed in this defect
    private String fixupURLs(String urls)
    {
        StringBuffer sb = new StringBuffer();
        StringTokenizer st = new StringTokenizer(urls, ", ");
        InetAddress localAddr = null;
        try
        {
            localAddr = InetAddress.getLocalHost();
            InetAddress.getByName("localhost"); // this must be defined and accessible
        }
        catch(UnknownHostException e)
        {
            localAddr = null; // to prevent conversion below
        }

        for (int i = 0; st.hasMoreTokens(); i++)
        {
            if (i > 0)
            {
                sb.append(',');
            }
            String url = st.nextToken();
            String host = "";
            // append protocol if supplied
            int index = url.indexOf("://");
            if (index > -1)
            {
                sb.append(url.substring(0, index + 3));
                url = url.substring(index + 3);
            }
            index = url.indexOf(":");
            host = index < 0 ? url : url.substring(0, index);

            // convert the host if need be then append the host
            if (REPLACE_IPADDRESS_OR_HOSTNAME_WITH_LOCALHOST)
            {
                // this will only be done if (a) the flag has been set explicitly, and
                // (b) the container using this ConnectorClient instance is collocated
                // with the Management Broker, and (c) the host/IP address supplied in the URL
                // matches that of "localhost".
                try
                {
                    if (localAddr != null && localAddr.equals(InetAddress.getByName(host)))
                    {
                        host = "localhost";
                    }
                }
                catch(UnknownHostException e) { }
            }
            sb.append(host);
            // append the port if supplied
            if (index > -1)
            {
                sb.append(url.substring(index));
            }
        }

        return sb.toString();
    }
        
    private class TopicConnectionFactoryFactory implements ITopicConnectionFactoryFactory
    {
        @Override
        public TopicConnectionFactory createConnectionFactory() throws JMSException
        {
            return new progress.message.jclient.TopicConnectionFactory();
        }
    }
    
    private class DurableConnectorFactory implements IDurableConnectorFactory
    {
        @Override
        public DurableConnector createDurableConnector(IDurableConnectorConsumer consumer, String consumerType, long timeout) throws InterruptedException
        {
            return new DurableConnector(consumer, consumerType, timeout);
        }
    }
    
    interface IDurableConnectorFactory
    {
        public DurableConnector createDurableConnector(IDurableConnectorConsumer consumer, String consumerType, long timeout) throws InterruptedException;
    }
    
    interface ITopicConnectionFactoryFactory
    {
        TopicConnectionFactory createConnectionFactory() throws JMSException;
    }

    /**
     * Class acts as a structure to hold a MF notification listener's details and
     * act as a delegate to the listener.
     *
     * This class includes the mapping between JMS based notifications and the neutral
     * MF based notification format.
     */
    private class NotificationListenerDelegate
    implements MessageListener, IConsumer
    {
        INotificationListener listener;

        private NotificationListenerDelegate(INotificationListener listener)
        {
            this.listener = listener;
        }

        /**
         * Receive a notification mapped to a JMS message and remap to the MF
         * neutral notification format.
         */
        @Override
        public void onMessage(Message message)
        {
            try
            {
                // THST
                BytesMessage notificationMessage = (BytesMessage)message;
                byte[] bytes = new byte[notificationMessage.readInt()];
                notificationMessage.readBytes(bytes);
                ByteArrayInputStream arrayIn = new ByteArrayInputStream(bytes);

                synchronized(arrayIn) // performance optimization
                {
                    ObjectInputStream in = new ObjectInputStream(arrayIn);

                    final INotification notification = (INotification)in.readObject();
                    final Object handback = in.readObject();
    
                    new Thread()
                    {
                        @Override
                        public void run()
                        {
                            NotificationListenerDelegate.this.listener.handleNotification(notification, handback);
                        }
                    }.start();
                    in.close();
                }
            }
            catch(Throwable e)
            {
                if ((ConnectorClient.this.m_traceMask & IConnectorClient.TRACE_REQUEST_REPLY_FAILURES) > 0)
                {
                    ConnectorClient.this.logMessage("Failed to read notification, trace follows...", e, Level.TRACE);
                }
            }
        }

        //
        // IConsumer interface
        //

        @Override
        public void close()
        {
            ConnectorClient.this.m_notificationListeners.remove(this);
        }
    }

    private final class RetryCallbackManager
    {
        private int waiters = 0;
        private short callbackAction = -1;
    }
}
