// Copyright (c) 2009 Progress Software Corporation. All Rights Reserved.
package com.sonicsw.mf.comm.jms;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.StringTokenizer;

import javax.jms.DeliveryMode;
import javax.jms.ExceptionListener;
import javax.jms.IllegalStateException;
import javax.jms.JMSException;
import javax.jms.JMSSecurityException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicPublisher;
import javax.jms.TopicSession;
import javax.jms.TopicSubscriber;

import com.sonicsw.mf.comm.IConnectionListener;
import com.sonicsw.mf.comm.IConnectorClient;
import com.sonicsw.mf.comm.IDurableConnectorConsumer;
import com.sonicsw.mf.comm.IGlobalComponentListener;
import com.sonicsw.mf.comm.InvokeTimeoutCommsException;
import com.sonicsw.mf.comm.InvokeTimeoutException;
import com.sonicsw.mf.common.IConsumer;
import com.sonicsw.mf.common.MFConnectAbortedException;
import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.MFSecurityException;
import com.sonicsw.mf.common.MgmtMsgTooBigException;
import com.sonicsw.mf.common.runtime.IStateListener;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.runtime.NonRecoverableStateChangeException;
import com.sonicsw.mf.common.runtime.RecoverableStateChangeException;

import progress.message.client.EAnonymousConnectionDisallowed;
import progress.message.client.EInvalidApplicationId;
import progress.message.client.EInvalidSubjectSyntax;
import progress.message.client.EInvalidUserId;
import progress.message.client.ENetworkFailure;
import progress.message.client.EParameterIsNull;
import progress.message.client.EPasswordExpired;
import progress.message.client.ESecurityGeneralException;
import progress.message.client.ESecurityPolicyViolation;
import progress.message.client.EUnauthorizedClient;
import progress.message.client.EUnknownBrokerHost;
import progress.message.client.EUserAlreadyConnected;
import progress.message.jclient.Connection;
import progress.message.jclient.ConnectionStateChangeListener;
import progress.message.jclient.Constants;
import progress.message.jclient.TopicConnection;

public class DurableConnector
{
    private static boolean DEBUG_TRACE_CONNECTION_STATE = false;
    private static final boolean DEBUG_TRACE_PUBLISH = false;
    private static final boolean DEBUG_TRACE_SUBSCRIBE = false;

    private IDurableConnectorConsumer m_consumer;
    private String m_consumerType;
    private String m_localRoutingNodeName;
    private boolean m_connectionEnterpriseEnabled;

    private ConnectionProxy m_connectionProxy;
    private ConnectThread m_connectThread;
    private Object m_connectThreadLockObj = new Object();
    private boolean connectThreadLocked = true;
    
    private long m_lastPublishTime = System.currentTimeMillis();

    private boolean m_isCleanupPending = false;

    private Exception m_lastConnectionException;
    private boolean m_connectionRetryLogged = false;

    private int m_deliveryMode = DeliveryMode.PERSISTENT;// progress.message.jclient.DeliveryMode.DISCARDABLE; //default

    private JMSException m_permanentFailureException;

    private ConnectionStateManager m_stateManager;
    
    private MessageSizeValidator m_msgSizeValidator;

    public static final long DURABLE_SUBSCRIPTION_TTL;
    private static final long DURABLE_SUBSCRIPTION_TTL_DEFAULT = 7200000L; // 2 hrs
    private static final String DURABLE_SUBSCRIPTION_TTL_PROPERTY = "sonicsw.mf.durableSubscriptionTTL";
    private static final String SKIP_INITIAL_CONNECT_PROPERTY = "sonicsw.mf.skipInitialConnect";

    static
    {
        String durableSubscriptionTTL = System.getProperty(DURABLE_SUBSCRIPTION_TTL_PROPERTY);
        DURABLE_SUBSCRIPTION_TTL = durableSubscriptionTTL == null ? DURABLE_SUBSCRIPTION_TTL_DEFAULT : Long.parseLong(durableSubscriptionTTL);
    }

    /**
     * 
     * @param consumer The consumer (owner) of this connector
     * @param consumerType The man readable name for the consumer type (e.g. "Management", "Fault detection", etc.). The value gets used in logged messages.
     * @param timeout
     * @param stateManager
     * @throws InterruptedException
     */
    public DurableConnector(IDurableConnectorConsumer consumer, String consumerType, long timeout) throws InterruptedException
    {
        long startTime = System.currentTimeMillis();  // get current time (used later to figure out how long to wait for [re]connect attempt to be made)

        // check to see if system property for tracing has been enabled (needed for client-side tracing)
        String tr = System.getProperty("enableDurableConnectorConnectionTracing");
        if ((tr != null) && (tr.equalsIgnoreCase("true")))
        {
            DEBUG_TRACE_CONNECTION_STATE = true;
        }

        m_consumer = consumer;
        m_consumerType = consumerType;
        
        m_msgSizeValidator = new MessageSizeValidator(m_consumer);
        
        m_stateManager = new ConnectionStateManager();

        // create and register state listeners
        m_stateManager.registerStateListener(new ConnectionStateListener(), "ConnectorClient");

        // Set the connection up
        TopicConnection connection = null;
        try
        {
        	String skipInitialConnect = System.getProperty(SKIP_INITIAL_CONNECT_PROPERTY, "false");
        	if ("false".equalsIgnoreCase(skipInitialConnect))
        	{
                connection = initialConnect();
        	}

        }
        catch(BadParamConnectionException e)
        {
            // We cannot recover from this simply by continuing to try to connect
            MFRuntimeException re = new MFRuntimeException(e.getMessage());
            if (e.getLinkedException() != null)
            {
                re.setLinkedException(e.getLinkedException());
            }
            throw re;
        }
        catch(JMSException e)
        {
            if (DEBUG_TRACE_CONNECTION_STATE)
            {
                System.out.println("DurableConnector: JMSException thrown during attempt to establish initial connection: " + e.getMessage());
            }
            logFailure(e);
        }

        // create the ConnectionProxy object(s)
        m_connectionProxy = new ConnectionProxy(connection);
        if (connection != null)
        {
            try
            {
                m_connectionProxy.setupContext(connection);  // set the proxy's context
            }
            catch (JMSException je)  // this is a fatal condition and we need to bail...
            {
                if (DEBUG_TRACE_CONNECTION_STATE)
                {
                    System.out.println("JMSException thrown during initial attempt to establish connection context, je = " + je.getMessage());
                }
                m_permanentFailureException = je;
                m_connectionProxy.cleanup();
                RuntimeException rte = new RuntimeException("JMSException encountered while trying to establish initial connection context");
                rte.initCause(je);
                throw rte;
            }
        }

        // The connection parameter might be null if initialConnect() failed.  The new Connection
        // object will keep trying to connect.
        m_connectThread = new ConnectThread(m_connectionProxy);

        synchronized(m_connectThreadLockObj)
        {
            // start the connect thread
            m_connectThread.start();

            // only wait on initial connection if this durable connector is being Synchronized by
            // a JMX client and then if we're not connected, cleanup
            if ((m_consumer.getClass().equals(ConnectorClient.class)) && (connection == null))
            {
                try
                {
                    if (timeout == 0)
                    {
                        while (connectThreadLocked)
                        {
                            m_connectThreadLockObj.wait(timeout); // if user specified zero as the timeout, we will use that value as-is...
                        }
                    }
                    else
                    {
                        long waitTime = timeout - (System.currentTimeMillis() - startTime);  // acount for time that has elapsed since this method was invoked....
                        while (waitTime > 0 && connectThreadLocked)
                        {
                            // don't want to wait until thread is interrupted unless user specified value of zero for the timeout
                            m_connectThreadLockObj.wait(waitTime);
                            waitTime = timeout - (System.currentTimeMillis() - startTime);
                        }
                    }
                }
                catch(InterruptedException e) // could only be interrupted in order to close
                {
                    cleanup();
                    throw new MFConnectAbortedException("Connect attempt to management broker interrupted.");
                }
                finally
                {
                    connectThreadLocked = true;
                }
            }
        }
    }

    // allows to set NON_PERSISTENT delivery mode
    public void setDeliveryMode(int deliveryMode)
    {
    	m_deliveryMode = deliveryMode;
    }

    public void cleanupPending()
    {
        m_isCleanupPending = true;
    }

    public final void cleanup()
    {
        cleanupPending();

        if (m_connectThread != null)
        {
            m_connectThread.cleanup();
        }

        if (m_connectionProxy != null)
        {
            m_connectionProxy.cleanup();
        }
    }

    public void publish(String subject, Message message, long timeout, long ttl, Object[] requestArray)
    throws JMSException, MgmtMsgTooBigException
    {
        Throwable traceEx = null;
        if (DEBUG_TRACE_PUBLISH)
        {
            System.out.println("DurableConnector.publish: subject = " + subject + ", request array target = " + requestArray[ConnectorClient.REQUEST_TARGET_INDEX]);
        }

        long timeoutAtTime = System.currentTimeMillis() + timeout;

        if (DEBUG_TRACE_PUBLISH)
        {
            System.out.println("DurableConnector.publish: timeoutAtTime= " + timeoutAtTime);
        }

        while (true)
        {
            try
            {
                TopicPublisher publisher = m_connectionProxy.getTopicPublisher();

                if (DEBUG_TRACE_PUBLISH)
                {
                    System.out.println("DurableConnector.publish: publisher = " + publisher);
                }

                // out of time ?
                if (System.currentTimeMillis() >= timeoutAtTime)
                {
                    throw (publisher == null ? new InvokeTimeoutCommsException(subject) : new InvokeTimeoutException(subject));
                }

                // make the assumption that the node name of the broker we talk to could change between requests/retries
                // and thus change the reply-to as needed
                String replyTo = message.getStringProperty(ConnectorClient.JMS_REPLY_TO_PROPERTY);
                if (replyTo != null) // not applicable to requests that have no reply-to !
                {
                    int index = replyTo.indexOf("::");
                    index += index == -1 ? 1 : 2;
                    message.setStringProperty(ConnectorClient.JMS_REPLY_TO_PROPERTY, m_localRoutingNodeName + "::" + replyTo.substring(index));
                }

                // SNC00082747 - may throw MgmtMsgTooBigException
                m_msgSizeValidator.validate(message, subject);
                
                synchronized (requestArray)
                {
                    // a reply could have been received after an earlier send that we thought had failed but did actually get through,
                    // in which case we can give up now
                    if (requestArray[ConnectorClient.REQUEST_REPLY_MESSAGE_INDEX] != null)
                    {
                        return;
                    }
                }
                
                synchronized(publisher) // note if this is null it will be handled in the exception handing code below
                {
                    publisher.publish(new progress.message.jimpl.Topic(subject, true), message, m_deliveryMode, publisher.getPriority(), ttl);
                }
                m_lastPublishTime = System.currentTimeMillis();

                if (DEBUG_TRACE_PUBLISH)
                {
                    System.out.println("DurableConnector.publish: request published at " + m_lastPublishTime);
                }
            }
            catch(JMSException e)
            {
                if (DEBUG_TRACE_PUBLISH)
                {
                    System.out.println("DurableConnector.publish: Waiting to retry due to: " + e.getClass().getName());
                }
                if (e.getLinkedException() instanceof EUnauthorizedClient)
                {
                    throw e;
                }

                if (m_isCleanupPending)
                {
                    return;
                }

                if (m_connectionProxy != null)
                {
                    m_connectionProxy.onException(e);
                }

                synchronized(this)
                {
                    if (DEBUG_TRACE_PUBLISH)
                    {
                        if (traceEx == null || !traceEx.getClass().isInstance(e.getLinkedException()))
                        {
                            traceEx = e.getLinkedException();
                            traceEx.printStackTrace();
                        }
                    }
                    try { wait(250); }
                    catch(InterruptedException ie1) { if (DEBUG_TRACE_PUBLISH)
                    {
                        System.out.println("DurableConnector.publish: InterruptedException thrown while waiting after JMSException....");
                    } }
                }

                if (m_isCleanupPending)
                {
                    return;
                }

                continue;
            }
            catch(NullPointerException npe)
            {
                if (DEBUG_TRACE_PUBLISH)
                {
                    System.out.println("DurableConnector.publish: subject = " + subject + ", request arr target = " + requestArray[ConnectorClient.REQUEST_TARGET_INDEX]);
                }
                if (m_permanentFailureException != null)
                {
                    throw m_permanentFailureException;
                }
                if (DEBUG_TRACE_PUBLISH)
                {
                    System.out.println("DurableConnector: Waiting to retry due to: " + npe.getClass().getName());
                }
                if (m_isCleanupPending)
                {
                    return;
                }
                synchronized(this)
                {
                    if (DEBUG_TRACE_PUBLISH)
                    {
                        if (traceEx == null || !traceEx.getClass().isInstance(npe))
                        {
                            traceEx = npe;
                            traceEx.printStackTrace();
                        }
                    }
                    try { wait(250); } catch(InterruptedException ie2) { if (DEBUG_TRACE_PUBLISH)
                    {
                        System.out.println("DurableConnector.publish: InterruptedException thrown while waiting after NullPointerException....");
                    } }
                }
                if (m_isCleanupPending)
                {
                    return;
                }
                continue;
            }
            break;
        }
    }

    // subscriptionName can be null, making this a non-durable subscription. See
    // mf.comm.jms.ConnectorClient.connect
    public IConsumer subscribe(String subject, String subscriptionName, MessageListener msgListener, IGlobalComponentListener globalComponentListener)
    throws JMSException
    {
        if (DEBUG_TRACE_SUBSCRIBE)
        {
            System.out.println("DurableConnector.subscribe: subject = " + subject);
        }

        Subscription subscription = new Subscription(subject, subscriptionName, msgListener, globalComponentListener);

        if (DurableConnector.this.m_connectionProxy != null)
        {
            DurableConnector.this.m_connectionProxy.subscribe(subscription);
        }

        return subscription;
    }

    /**
     * return the reference to the ConnectThread instance created by the DurableConnector instance
     */
    Thread getConnectThread() { return m_connectThread; }

    /**
     * Gets the node name of the broker we're connected to
     */
    String getLocalRoutingNodeName()
    {
        // if currently connected, simply return whatever "m_localRoutingNodeName" is
        // currently set to (it will be set properly each time the connection state changes
        // to one of the various flavors of "connected").
        String localRoutingNodeName = null;
        short currentConnectionState = DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
        switch (currentConnectionState)
        {
            case ConnectionStateManager.CONNECTED:
                localRoutingNodeName = DurableConnector.this.m_localRoutingNodeName;
                break;
            default:
                localRoutingNodeName = null;  // if not connected, return null
        }

        return localRoutingNodeName;
    }

    /**
     * Determines if the broker we're connected to is enterprise licensed
     */
    public boolean isConnectionEnterpriseEnabled()
    {
        // if currently connected, simply return whatever "m_isEnterprise" is
        // currently set to (it will be set properly each time the connection state changes
        // to one of the various flavors of "connected").
    	// If not-connected return the last connected setting.
    	return m_connectionEnterpriseEnabled;
    }

    /**
     * Gets the node name of the broker we're connected to
     */
    public boolean isConnected()
    {
        short currentState = this.m_stateManager.getState((Object)ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
        if (currentState == ConnectionStateManager.CONNECTED)
        {
            return true;
        }

        return false;
    }

    private synchronized boolean logFailure(Exception currentException)
    {
        if (currentException instanceof JMSException && ((JMSException)currentException).getLinkedException() != null)
        {
            currentException = ((JMSException)currentException).getLinkedException();
        }

        Exception lastException = m_lastConnectionException;
        boolean retryLogged = m_connectionRetryLogged;

        boolean log = false;
        if (lastException == null)
        {
            log = true;
        }
        else
        if (!(currentException.getClass().getName().equals(lastException.getClass().getName())))
        {
            log = true;
        }
        else
        if (currentException.getMessage() != null && !currentException.getMessage().equals(lastException.getMessage()))
        {
            log = true;
        }

        // don't report if we're going to be terminated anyway
        if (m_isCleanupPending)
        {
            return false;
        }

        if (log)
        {
            if ((m_consumer.getTraceMask() & IConnectorClient.TRACE_CONNECTION_FAILURES) > 0)
            {
                m_consumer.logMessage(m_consumerType + " connect failure, trace follows...", currentException, Level.TRACE);
            }
            else
            {
                StringBuffer sb = new StringBuffer(m_consumerType + " connect failure: ");
                if (currentException instanceof EUnknownBrokerHost)
                {
                    sb.append(currentException.getMessage());
                }
                else
                if (currentException instanceof ENetworkFailure)
                {
                    sb.append(currentException.getMessage());
                }
                else
                {
                    sb.append(currentException.toString());
                }
                m_consumer.logMessage(sb.toString(), Level.WARNING);
            }
            retryLogged = true;
            m_consumer.logMessage("..." + m_consumerType.toLowerCase() + " connect failed, retrying...", Level.INFO);
        }
        else // only log the retry if tracing is switched on
        if (retryLogged && (m_consumer.getTraceMask() & IConnectorClient.TRACE_CONNECTION_FAILURES) > 0)
        {
            m_consumer.logMessage("..." + m_consumerType.toLowerCase() + " connect retry failed, retrying...", Level.TRACE);
        }

        // update the last recorded exceptions / retry flags
        m_lastConnectionException = currentException;
        m_connectionRetryLogged = retryLogged;

        return log;
    }

    /**
     * Remove the given subscription.
     */
    private void unsubscribe(Subscription subscription)
    {
        // remove/close the subscription (we're not interested in any exceptions/errors).
        ConnectionProxy connectionProxy = m_connectionProxy;
        if (connectionProxy != null)
        {
            try
            {
                connectionProxy.unsubscribe(subscription);
            }
            catch (Throwable e) { }
        }
    }

    /**
     * Called by JMSConnectorServer.
     * For container FT, if no components set fault detection connection,
     * JMSConnectorServer calls this method to get the MF connection.
     * @return The MF connection
     */
    public Connection getConnection() { return m_connectionProxy.getTopicConnection(); }

    // This is the method invoked from the DurableConnector constructor (only)
    private TopicConnection initialConnect()
    throws JMSException,BadParamConnectionException
    {
        // request connection state be set to indicate that connection attempt is being made
        short currentState = this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
        if (currentState == ConnectionStateManager.CONNECT)
        {
            try
            {
                m_stateManager.requestStateChange(currentState, ConnectionStateManager.CONNECTING, ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
            }
            catch (RecoverableStateChangeException sce)
            {
                // none of the state controller implementations used here actually throw
                // a RecoverableStateChangeException.  This try-catch block is here
                // simply to satisfy the constraints imposed on the state controller by
                // the signature of the IStateController interface (which includes a throws clause
                // for the RecoverableStateChangeException).
            }
            catch (NonRecoverableStateChangeException sce)
            {
                // none of the state controller implementations used here actually throw
                // a NonRecoverableStateChangeException.  This try-catch block is here
                // simply to satisfy the constraints imposed on the state controller by
                // the signature of the IStateController interface (which includes a throws clause
                // for the NonRecoverableStateChangeException).
            }
        }
        else
        {
            throw new RuntimeException("Unexpected connection state before connect attempt; connection state == " + currentState);
        }

        if (DEBUG_TRACE_CONNECTION_STATE)
        {
            System.out.println("DurableConnector.initialConnect: just before invoking createTopicConnection, currentState = " + this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME));
        }

        TopicConnection connection = null;
        try
        {
            // create the topic connection
            connection = (TopicConnection)m_consumer.getConnectionFactory().createTopicConnection();

            // request connection state be set to indicate that connection attempt was successful
            currentState = this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
            if (DEBUG_TRACE_CONNECTION_STATE)
            {
                System.out.println("DurableConnector.initialConnect: after createTopicConnection, currentState = " + currentState);
            }
            if (currentState == ConnectionStateManager.CONNECTING)
            {
                try
                {
                    m_stateManager.requestStateChange(currentState, ConnectionStateManager.CONNECTED, ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);

                    if (DEBUG_TRACE_CONNECTION_STATE)
                    {
                        System.out.println("DurableConnector.initialConnect: after requesting state change, current conn State = " + this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME));
                    }
                }
                catch (RecoverableStateChangeException sce)
                {
                    // none of the state controller implementations used here actually throw
                    // a RecoverableStateChangeException.  This try-catch block is here
                    // simply to satisfy the constraints imposed on the state controller by
                    // the signature of the IStateController interface (which includes a throws clause
                    // for the RecoverableStateChangeException).
                    if (DEBUG_TRACE_CONNECTION_STATE)
                    {
                        System.out.println("DurableConnector.initialConnect: there's no way that we should be here...");
                    }
                }
                catch (NonRecoverableStateChangeException sce)
                {
                    // none of the state controller implementations used here actually throw
                    // a NonRecoverableStateChangeException.  This try-catch block is here
                    // simply to satisfy the constraints imposed on the state controller by
                    // the signature of the IStateController interface (which includes a throws clause
                    // for the NonRecoverableStateChangeException).
                    if (DEBUG_TRACE_CONNECTION_STATE)
                    {
                        System.out.println("DurableConnector.initialConnect: there's no way that we should be here...");
                    }
                }
            }
            else
            {
                if (DEBUG_TRACE_CONNECTION_STATE)
                {
                    System.out.println("DurableConnector.initialConnect: a RuntimeException before connect attempt...");
                }
                throw new RuntimeException("Unexpected connection state after successful connect attempt; connection state == " + currentState);
            }

            m_localRoutingNodeName = ((progress.message.jclient.Connection)connection).getRoutingNodeName();
            m_connectionEnterpriseEnabled = ((progress.message.jclient.Connection)connection).isEnterpriseEnabled();

            // log that the connection was established
            DurableConnector.this.m_consumer.logMessage(m_consumerType + " connection (re)established (" +
                                                               ((progress.message.jimpl.Connection)connection).getZConnection().getSocket().toString() +
                                                               ')', Level.INFO);

            DurableConnector.this.m_lastConnectionException = null;
            DurableConnector.this.m_connectionRetryLogged = false;

            if (DEBUG_TRACE_CONNECTION_STATE)
            {
                System.out.println("DurableConnector.initialConnect: before exiting method, currentState = " + this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME));
            }
        }
        catch(JMSException je)
        {
            if (DEBUG_TRACE_CONNECTION_STATE)
            {
                System.out.println("DurableConnector.initialConnect: during createTopicConnection attempt, JMSException thrown, je = " + je.getMessage());
            }
            Exception e = je.getLinkedException();
            String classname = "";
            if (e != null)
            {
                classname = e.getClass().getName();
            }
            if (e == null ||
                e instanceof EInvalidSubjectSyntax ||
                e instanceof EParameterIsNull ||
                e instanceof EInvalidUserId ||
                e instanceof EInvalidApplicationId ||
                e instanceof EUserAlreadyConnected ||
                !classname.startsWith("progress.message"))
            {
                if (DEBUG_TRACE_CONNECTION_STATE)
                {
                    System.out.println("DurableConnector.initialConnect: reset to initial state, e = " + e);
                }

                // reset connection state to initial state
                currentState = this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
                resetConnectionStateOnException(currentState);

                try { if (connection != null) connection.close(); } catch(Throwable t) { } // try to cleanup

                if (e instanceof EUserAlreadyConnected)
                {
                    throw new BadParamConnectionException("User already connected", e);
                }
                else
                {
                    throw new BadParamConnectionException(e == null ?
                        "Unknown connection setup failure; connection state=" + currentState :
                        (e.getMessage() == null || e.getMessage().length() == 0 ? "" : (e.getMessage() + "; ")) + "connection state=" + currentState, e);
                }
            }
            else
            if (e instanceof ESecurityPolicyViolation ||
                e instanceof ESecurityGeneralException ||
                e instanceof EPasswordExpired ||
                e instanceof EUnauthorizedClient ||
                e instanceof EAnonymousConnectionDisallowed)
            {
                if (DEBUG_TRACE_CONNECTION_STATE)
                {
                    System.out.println("DurableConnector.initialConnect: doubt that we are here..., e = " + e);
                }
                // reset connection state to initial state
                currentState = this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
                resetConnectionStateOnException(currentState);

                // For security violation, let exception be thrown
                try { if (connection != null) connection.close(); } catch(Throwable t) { } // try to cleanup
                MFSecurityException securityException = new MFSecurityException("Failed to connect to the management broker due to authentication failure; connection state=" + currentState);
                securityException.setLinkedException(e);
                throw securityException;
            }
            else
            {
                // reset connection state to initial state
                currentState = this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
                resetConnectionStateOnException(currentState);

                if (DEBUG_TRACE_CONNECTION_STATE)
                {
                    System.out.println("DurableConnector.initialConnect: simply rethrowing the exception...");
                }
                throw je;
            }
        }

        return connection;
    }

    //
    // This is the method that will be invoked from the new ConnectThread (which has checked and set the state appropriately...)
    //
    private TopicConnection subsequentConnect()
    throws JMSException,BadParamConnectionException
    {
        // It is expected that the caller has checked and set the connection state prior to calling,
        // and will do so after the method returns.

        // create the topic connection
        TopicConnection connection = (TopicConnection)m_consumer.getConnectionFactory().createTopicConnection();

        m_localRoutingNodeName = connection.getRoutingNodeName();
        m_connectionEnterpriseEnabled = ((progress.message.jclient.Connection)connection).isEnterpriseEnabled();


        return connection;
    }

    private void resetConnectionStateOnException(short currentState)
    {
        // reset connection state:
        //    if state is "CONNECTING_TO_PRIMARY__DISCONNECTED_FROM_SECONDARY", then the same state
        //    transition call may be made whether FT is enabled or not;
        //    the other state transitions will only occur FT has been enabled
        if (currentState == ConnectionStateManager.CONNECTING)
        {
            try
            {
                m_stateManager.requestStateChange(currentState, ConnectionStateManager.CONNECT, ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
            }
            catch (RecoverableStateChangeException sce)
            {
                // none of the state controller implementations used here actually throw
                // a RecoverableStateChangeException.  This try-catch block is here
                // simply to satisfy the constraints imposed on the state controller by
                // the signature of the IStateController interface (which includes a throws clause
                // for the RecoverableStateChangeException).
            }
            catch (NonRecoverableStateChangeException sce)
            {
                // none of the state controller implementations used here actually throw
                // a NonRecoverableStateChangeException.  This try-catch block is here
                // simply to satisfy the constraints imposed on the state controller by
                // the signature of the IStateController interface (which includes a throws clause
                // for the NonRecoverableStateChangeException).
            }
        }
        else if ((currentState == ConnectionStateManager.CONNECTED))
        {
            // handle the case where AM/DS FT is NOT enabled and a failure occurs immediately
            // after a connection is set up (and, for example, *before* the context is set on the new connection)
            try
            {
                m_stateManager.requestStateChange(currentState, ConnectionStateManager.CONNECT, ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
            }
            catch (RecoverableStateChangeException sce) {}
            catch (NonRecoverableStateChangeException sce) {}
        }
        else if ((currentState == ConnectionStateManager.CONNECT))
        {
            // handle the case where AM/DS FT is NOT enabled and a failure occurs immediately
            // after a connection is set up (and, for example, *before* the context is set on the new connection)
            try
            {
                m_stateManager.requestStateChange(currentState, ConnectionStateManager.CONNECT, ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
            }
            catch (RecoverableStateChangeException sce) {}
            catch (NonRecoverableStateChangeException sce) {}
        }
        else
        {
            throw new RuntimeException("Unexpected connection state after unsuccessful connect attempt; connection state=" + currentState);
        }
    }
    


    private final static class BadParamConnectionException
    extends JMSException
    {
        private BadParamConnectionException(String msg, Exception e)
        {
            super(msg);
            super.setLinkedException(e);
        }
    }

    private final class Subscription
    implements IConsumer
    {
        Topic topic;
        String subscriptionName;
        private MessageListener listener;
        IGlobalComponentListener globalComponentListener;

        private Subscription(String subject, String subscriptionName, MessageListener listener, IGlobalComponentListener globalComponentListener)
        {
            try
            {
                this.topic = new progress.message.jimpl.Topic(subject, true);
            }
            catch(JMSException e)
            {
                MFRuntimeException re = new MFRuntimeException();
                re.setLinkedException(e);
                throw re;
            }
            this.subscriptionName = subscriptionName;
            this.listener = listener;
            this.globalComponentListener = globalComponentListener;
        }

        //
        // IConsumer interface
        //

        @Override
        public void close()
        {
            DurableConnector.this.unsubscribe(this);
        }
    }

    private final class ConnectThread
    extends Thread
    {
        private boolean needsToConnect;

        private boolean isActive = true;
        private ConnectThread(ConnectionProxy connectionProxy)
        {
            super("Durable Connect [" + DurableConnector.this.m_consumerType + "]");

            // If the attempt to create the connection in DurableConnector's constructor
            // (made in order to ensure that the connection parameters were valid before creating this thread)
            // was successful, then the connection's "connection" will be non-null.
            // If that attempt failed due to invalid connection parameters (such as an invalid user) then
            // this constructor should not have been invoked. If, however, the attempt failed due to
            // another cause such as, for example, the broker was not started yet,
            // then we'll get here with the primary Connections "connection" will be "null"
            // and we'll keep retrying.
            if (connectionProxy.getTopicConnection() == null)
            {
                needsToConnect = true;
            }
            else
            {
                needsToConnect = false;
            }

            super.setDaemon(true);
        }

        @Override
        public void run()
        {
            TopicConnection connection = null;
            ConnectionProxy connProxy = null;

            // keep thread active until client/server is shutting down
            while (stayConnected())
            {
                boolean needToSetProxy = false;  // flag to indicate if the context of the connection proxy object will need to be set

                if (needsToConnect)  // a connection attempt is necessary...
                {
                    needToSetProxy = true;  // will need to set the context of the corresponding ConnectionProxy instance if a connection is successfully established...

                    // switch on the cases that should be handled
                    short currentConnState = DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);

                    if (DEBUG_TRACE_CONNECTION_STATE)
                    {
                        System.out.println("DurableConnector.ConnectThread.run: currentConnState = " + currentConnState);
                    }

                    connection = null;
                    try
                    {
                        switch (currentConnState)
                        {
                            case ConnectionStateManager.CONNECT:
                                if (DEBUG_TRACE_CONNECTION_STATE)
                                {
                                    System.out.println("DurableConnector.ConnectThread.run: create connection, current connection state = " + currentConnState);
                                }
                                connection = createConnection(currentConnState);
                                connProxy = DurableConnector.this.m_connectionProxy;
                                break;
                            default:
                                if (DEBUG_TRACE_CONNECTION_STATE)
                                {
                                    System.out.println("DurableConnector.ConnectThread.run: Unexpected connection state during connect attempt, current connection state = " + currentConnState);
                                }
                                if ((DurableConnector.this.m_consumer.getTraceMask() & IConnectorClient.TRACE_CONNECTION_STATE_CHANGES) > 0)
                                {
                                    DurableConnector.this.m_consumer.logMessage("Unexpected " + m_consumerType.toLowerCase() + " connection state during connect attempt, state=" + currentConnState, Level.TRACE);
                                }
                                throw new RuntimeException("Unexpected " + m_consumerType.toLowerCase() + " connection state encountered while attempting to establish connection in background, state=" + currentConnState);
                        }
                    }
                    catch (final BadParamConnectionException e)
                    {
                        // We cannot recover from this by keep trying to connect
                        Thread failureHandler = new Thread("Durable Connector - Failure Handler")
                        {
                            @Override
                            public void run()
                            {
                                if (DurableConnector.this.m_isCleanupPending)
                                {
                                    return;
                                }
                                DurableConnector.this.m_consumer.onFailure(e);
                            }
                        };
                        failureHandler.setDaemon(true);
                        failureHandler.start();
                        return;
                    }
                    catch (final MFSecurityException e)
                    {
                        // We cannot recover from this by keep trying to connect
                        Thread securityHandler = new Thread("Durable Connector - Security Failure Handler")
                        {
                            @Override
                            public void run()
                            {
                                if (DurableConnector.this.m_isCleanupPending)
                                {
                                    return;
                                }
                                DurableConnector.this.m_consumer.onFailure(e);
                            }
                        };
                        securityHandler.setDaemon(true);
                        securityHandler.start();
                        return;
                    }
                    catch (JMSException je)
                    {
                        short currConnState = DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
                        if (DEBUG_TRACE_CONNECTION_STATE)
                        {
                            System.out.println("DurableConnector.ConnectThread.run: JMS exception thrown, curr connection state = " + currConnState + ", je = " + je.getMessage());
                        }

                        resetConnectionStateOnException(currConnState);

                        if (DEBUG_TRACE_CONNECTION_STATE)
                        {
                            System.out.println("DurableConnector.ConnectThread.run: after reset on exception, current conn state = " + currConnState);
                        }

                        // get out of connection loop if we're going to be terminated anyway (server side condition only)
                        if (DurableConnector.this.m_isCleanupPending)
                        {
                            break;
                        }

                        // Invoke the disconnect handler (if one is registered).
                        Thread disconnectThread = new Thread("Durable Connector - Disconnect Handler")
                        {
                            @Override
                            public void run()
                            {
                                if (DurableConnector.this.m_isCleanupPending)
                                {
                                    return;
                                }
                                DurableConnector.this.m_consumer.onDisconnect();
                            }
                        };
                        disconnectThread.setDaemon(true);
                        disconnectThread.start();
                    }

                    // If connect attempt was successful, set this thread's flag accordingly.
                    // Otherwise, sleep for 1 second and try, try again...
                    if (DEBUG_TRACE_CONNECTION_STATE)
                    {
                        System.out.println("DurableConnector.ConnectThread.run: connection = " + connection + ", curr conn state = " + DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME) + ", needToSetProxy = " + needToSetProxy);
                    }

                    if (connection == null)
                    {
                        try { Thread.sleep(1000); } catch (InterruptedException e) { }
                        continue;  // try again, right?
                    }
                }
                else
                {
                    // if the connection had already been established earlier, there is no need to
                    // to initialize the corresponding connection proxy object...
                    needToSetProxy = false;
                }

                // If we arrived at this point then either the connection was just established,
                // or the connection had already been established ("needsToConnect=false").
                // If the former is the case, we need to establish the context of the
                // appropriate ConnectionProxy object.
                // We'll want to notify any waiting threads.
                if (needToSetProxy)
                {
                    try
                    {
                        connProxy.setupContext(connection);
                    }
                    catch(JMSException e)
                    {
                        if (DEBUG_TRACE_CONNECTION_STATE)
                        {
                            System.out.println("DurableConnector.ConnectThread.run: JMSException thrown while trying to establish context, connection state = " + DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME) + ", e = " + e.getMessage());
                        }

                        final Exception ex = e;
                        final Exception le = e.getLinkedException();

                        if (le != null)
                        {
                            if (DEBUG_TRACE_CONNECTION_STATE)
                            {
                                System.out.println("DurableConnector.ConnectThread.run: Linked exception message = " + le.getMessage());
                            }

                            // I'm pretty sure the check below is meant to catch the case where either a durable
                            // subscriber is already connected, or the permissions are not set for
                            // access.
                            if (le instanceof progress.message.client.EUserAlreadyConnected || le instanceof progress.message.client.EUnauthorizedClient)
                            {
                                // this is a fatal condition and we need to bail...
                                {
                                   if (!(le instanceof progress.message.client.EUserAlreadyConnected))
                                {
                                    DurableConnector.this.m_consumer.logMessage(m_consumerType + "connection: Unauthorized access", Level.SEVERE);
                                }

                                    DurableConnector.this.m_permanentFailureException = e;

                                    // Since we are running in a separate thread, the simplest way to terminate
                                    // is to halt (exit will try to cleanup and might fail because the context is not built yet)
                                    Thread duplicateHandler = new Thread("Durable Connector - Duplicate Handler")
                                    {
                                        @Override
                                        public void run()
                                        {
                                            DurableConnector.this.m_consumer.onFailure(le);
                                        }
                                    };
                                    duplicateHandler.setDaemon(true);
                                    duplicateHandler.start();

                                    return;
                                }
                            }
                            else
                            {
                                // Sonic00025144:  Experience has now shown that there could be some other
                                // linked exceptions thrown *after* the connection has been established but
                                // *before* the context (subscribers, subscriptions, etc.) is set
                                // (for example, a "java.net.NoRouteToHostException" wrapped in a
                                // JMSException).  So, rather than treat this case as a fatal condition,
                                // a message will be logged to indicate a failure, and attempts to re-establish
                                // a connection will continue.

                                // log the failure
                                DurableConnector.this.logFailure(le);

                                // Sonic00025144: handle case where connection is dropped immediately
                                // after it is established (e.g. before context can be set)
                                // reset connection state to initial state
                                short currentState = DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
                                DurableConnector.this.resetConnectionStateOnException(currentState);
                                continue;  // try, try again...
                            }
                        }
                        else if (ex instanceof IllegalStateException)
                        {
                            // log the failure
                            DurableConnector.this.logFailure(ex);

                            // Sonic00025144: handle case where connection is dropped immediately
                            // after it is established (e.g. before context can be set)
                            // reset connection state to initial state
                            short currentState = DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
                            DurableConnector.this.resetConnectionStateOnException(currentState);
                            continue;  // try, try again...
                        }
                        else  // not a LinkedException - treat it as a fatal condition...
                        {
                            DurableConnector.this.m_permanentFailureException = e;
                            final String errMsg = e.getMessage();
                            // Since we are running in a separate thread, the simplest way to terminate
                            // is to halt (exit will try to cleanup and might fail because the context is not built yet)
                            Thread duplicateHandler = new Thread("Durable Connector - Duplicate Handler")
                            {
                                @Override
                                public void run()
                                {
                                    DurableConnector.this.m_consumer.logMessage(errMsg, ex, Level.SEVERE);
                                    DurableConnector.this.m_consumer.onFailure(ex);
                                }
                            };
                            duplicateHandler.setDaemon(true);
                            duplicateHandler.start();
                            return;
                        }
                    }
                    finally
                    {
                        synchronized(m_connectThreadLockObj)
                        {
                            connectThreadLocked = false;
                            m_connectThreadLockObj.notifyAll();
                        }
                        synchronized(DurableConnector.this)
                        {
                            DurableConnector.this.notifyAll();
                        }
                    }
                }
                else  // proxy context already set - just notify any waiting thread(s)
                {
                    synchronized(m_connectThreadLockObj)
                    {
                        connectThreadLocked = false;
                        m_connectThreadLockObj.notifyAll();
                    }
                    synchronized(DurableConnector.this)
                    {
                        DurableConnector.this.notifyAll();
                    }
                }

                // if we arrived at this point, then a connection and the proper context
                // have been successfully established. So, we'll notify folks about the
                // reconnect and wait here until some state change is signaled...

                // create anonymous thread to inform of reconnection,
                // if and only if connect attempt was successful
                short currConnState = DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
                if (currConnState == ConnectionStateManager.CONNECTED)
                {
                    Thread reconnectHandler = new Thread("Durable Connector - Reconnect Handler")
                    {
                        @Override
                        public void run()
                        {
                            if (DurableConnector.this.m_isCleanupPending)
                            {
                                return;
                            }
                            DurableConnector.this.m_consumer.onReconnect(m_localRoutingNodeName);
                        }
                    };
                    reconnectHandler.setDaemon(true);
                    reconnectHandler.start();
                }

                synchronized (DurableConnector.this)
                {
                    try
                    {
                        // Sonic00025144: make a last check, before the wait, to see
                        // if a connection needs to be re-established, as it is possible
                        // for the connection to be dropped between when the lock on the
                        // DurableConnector is released and when the lock on the
                        // ConnectThread is grabbed.
                        needsToConnect = isConnectAttemptRequired();

                        // wait until one of the connection state controllers signals on this
                        // thread [if there is currently no need to connect].
                        if (!needsToConnect)
                        {
                            DurableConnector.this.wait();

                            if (DEBUG_TRACE_CONNECTION_STATE)
                            {
                                System.out.println("DurableConnector.ConnectorThread.run: ConnectThread notified, current connection state = " + DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME));
                            }

                            if (!stayConnected())
                            {
                                break;
                            }

                            // check to see if a connection needs to be re-established
                            needsToConnect = isConnectAttemptRequired();
                        }

                        if (DEBUG_TRACE_CONNECTION_STATE)
                        {
                            System.out.println("DurableConnector.ConnectorThread.run: lost connection, need to reconnect = " + needsToConnect);
                        }
                    }
                    catch (InterruptedException ie)
                    {
                        if ((DurableConnector.this.m_consumer.getTraceMask() & IConnectorClient.TRACE_CONNECTION_STATE_CHANGES) > 0)
                        {
                            DurableConnector.this.m_consumer.logMessage("InterruptedException thrown while " + m_consumerType.toLowerCase() + " ConnectThread waiting, state=" + DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME), Level.TRACE);
                        }
                        if (DEBUG_TRACE_CONNECTION_STATE)
                        {
                            System.out.println("DurableConnector.ConnectorThread.run: InterruptedException thrown while ConnectThread waiting, state=" + DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME));
                        }
                    } // until notified due to some change in status
                }
            }  // end of "while"

            // since we are out of the "stayConnected()" loop, the client/server must be shutting down.
            // so, attempt to clean up the ConnectionProxy object(s)
            if (DurableConnector.this.m_connectionProxy != null)
            {
                DurableConnector.this.m_connectionProxy.cleanup();
            }
        }

        private TopicConnection createConnection(short currentState) throws JMSException
        {
            // request connection state be set to indicate that connection attempt is being made
            try
            {
                m_stateManager.requestStateChange(currentState, ConnectionStateManager.CONNECTING, ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
            }
            catch (RecoverableStateChangeException sce)
            {
                // none of the state controller implementations used here actually throw
                // a RecoverableStateChangeException.  This try-catch block is here
                // simply to satisfy the constraints imposed on the state controller by
                // the signature of the IStateController interface (which includes a throws clause
                // for the RecoverableStateChangeException).
            }
            catch (NonRecoverableStateChangeException sce)
            {
                // none of the state controller implementations used here actually throw
                // a NonRecoverableStateChangeException.  This try-catch block is here
                // simply to satisfy the constraints imposed on the state controller by
                // the signature of the IStateController interface (which includes a throws clause
                // for the NonRecoverableStateChangeException).
            }

            TopicConnection connection = null;
            try
            {
                // create the topic connection
                connection = DurableConnector.this.subsequentConnect();

                // request connection state be set to indicate that connection attempt was successful
                short currentConnState = DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
                if (currentConnState == ConnectionStateManager.CONNECTING)
                {
                    try
                    {
                        m_stateManager.requestStateChange(currentConnState, ConnectionStateManager.CONNECTED, ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
                    }
                    catch (RecoverableStateChangeException sce)
                    {
                        // none of the state controller implementations used here actually throw
                        // a RecoverableStateChangeException.  This try-catch block is here
                        // simply to satisfy the constraints imposed on the state controller by
                        // the signature of the IStateController interface (which includes a throws clause
                        // for the RecoverableStateChangeException).
                    }
                    catch (NonRecoverableStateChangeException sce)
                    {
                        // none of the state controller implementations used here actually throw
                        // a NonRecoverableStateChangeException.  This try-catch block is here
                        // simply to satisfy the constraints imposed on the state controller by
                        // the signature of the IStateController interface (which includes a throws clause
                        // for the NonRecoverableStateChangeException).
                    }
                }
                else
                {
                    throw new RuntimeException("Unexpected connection state after successful connect attempt; connection state == " + currentConnState);
                }

                // log that the connection was established
                DurableConnector.this.m_consumer.logMessage(m_consumerType + " connection (re)established (" +
                                                                   ((progress.message.jimpl.Connection)connection).getZConnection().getSocket().toString() +
                                                                   ')', Level.INFO);

                DurableConnector.this.m_lastConnectionException = null;
                DurableConnector.this.m_connectionRetryLogged = false;
            }
            catch(JMSException je)
            {
                if (DEBUG_TRACE_CONNECTION_STATE)
                {
                    System.out.println("DurableConnector.createConnection: JMSException, linked exception....je.msg = " + je.getMessage());
                }

                Exception e = je.getLinkedException();

                if (je instanceof IllegalStateException)
                {
                    handleIllegalStateException((IllegalStateException)je);
                    return null; // null out connection
                }
                else if ((e != null) && ((e instanceof progress.message.client.ENetworkFailure) ||
                                          (e instanceof progress.message.client.EConnectionLimitExceeded)))
                {
                    // for a network exception, reset the connection state appropriately
                    // in preparation for another connect attempt
                    DurableConnector.this.logFailure(je);

                    // need to reset the connection state change here.
                    try
                    {
                        m_stateManager.requestStateChange(ConnectionStateManager.CONNECTING, ConnectionStateManager.CONNECT, ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
                    }
                    catch (RecoverableStateChangeException sce)
                    {
                        // none of the state controller implementations used here actually throw
                        // a RecoverableStateChangeException.  This try-catch block is here
                        // simply to satisfy the constraints imposed on the state controller by
                        // the signature of the IStateController interface (which includes a throws clause
                        // for the RecoverableStateChangeException).
                    }
                    catch (NonRecoverableStateChangeException sce)
                    {
                        // none of the state controller implementations used here actually throw
                        // a NonRecoverableStateChangeException.  This try-catch block is here
                        // simply to satisfy the constraints imposed on the state controller by
                        // the signature of the IStateController interface (which includes a throws clause
                        // for the NonRecoverableStateChangeException).
                    }

                    // return
                    return connection;  // should be "null" in this case...
                }

                // for other (non-network) exceptions, handle each exception as appropriate
                handleConnectException(je);  // "true" indicates that the connection is to be cleaned up...
            }

            return connection;
        }

        private void handleConnectException(JMSException je) throws JMSException
        {
            Exception e = je.getLinkedException();

            String classname = "";
            if (e != null)
            {
                classname = e.getClass().getName();
            }

            if (e == null ||
                e instanceof EInvalidSubjectSyntax ||
                e instanceof EParameterIsNull ||
                e instanceof EInvalidUserId ||
                e instanceof EInvalidApplicationId ||
                e instanceof EUserAlreadyConnected ||
                !classname.startsWith("progress.message"))
            {
                // reset connection state to initial state
                short currentState = DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
                DurableConnector.this.resetConnectionStateOnException(currentState);

                // TODO: determine if it is indeed correct to throw an exception in all cases
                //       in the event of one of the above exceptions.
                throw new BadParamConnectionException(e == null ? "Unknown connection setup failure; connection state = " + currentState : e.getMessage() + "; connection state = " + currentState, e);
            }
            else if (e instanceof ESecurityPolicyViolation ||
                     e instanceof ESecurityGeneralException ||
                     e instanceof EPasswordExpired ||
                     e instanceof EUnauthorizedClient ||
                     e instanceof EAnonymousConnectionDisallowed)
            {
                // reset connection state to initial state
                short currentState = DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
                resetConnectionStateOnException(currentState);

                // For security violation, let exception be thrown
                MFSecurityException securityException = new MFSecurityException("Failed to connect to the management broker due to authentication failure; connection state=" + currentState);
                securityException.setLinkedException(e);
                throw securityException;
            }
            else
            {
                throw je;
            }
        }

        private void handleIllegalStateException(javax.jms.IllegalStateException ex)
        {
            // log the failure
            DurableConnector.this.logFailure(ex);

            // Sonic00025144: handle case where connection is dropped immediately
            // after it is established (e.g. before exception handler can be set)
            // reset connection state to initial state
            short currentState = DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
            DurableConnector.this.resetConnectionStateOnException(currentState);
        }

        /**
         * Tests if we should try to connect and maintain context on this connection.
         */
        private boolean stayConnected()
        {
            return this.isActive;
        }

        private void cleanup()
        {
            synchronized(m_connectThreadLockObj)
            {
                this.isActive = false;
                connectThreadLocked = false;
                m_connectThreadLockObj.notifyAll();
            }
            synchronized(DurableConnector.this)
            {
                DurableConnector.this.notifyAll();
            }

            try
            {
                this.join(2000);
            }
            catch(InterruptedException ie)
            {
                if (DEBUG_TRACE_CONNECTION_STATE)
                {
                    System.out.println("DurableConnector.ConnectThread.cleanup: InterruptedException thrown while joined...");
                }
            } // ignore

            if (this.isAlive())
            {
                // interrupt the ConnectThread if it is still marked as being alive
                this.interrupt();
            }
        }

        private boolean isConnectAttemptRequired()
        {
            short currentConnectionState = DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
            boolean connectAttemptRequired = false;

            switch (currentConnectionState)
            {

                case ConnectionStateManager.CONNECTED:
                    if (!DurableConnector.this.m_connectionProxy.isConnected())
                    {
                        connectAttemptRequired = true;
                    }
                    break;
                case ConnectionStateManager.CONNECTING:
                    break;  // if in "connecting" state, connect thread is currently making connect attempt...
                case ConnectionStateManager.CONNECT:
                    connectAttemptRequired = true;
                    break;
                case ConnectionStateManager.DISCONNECTING:
                    connectAttemptRequired = false;  // client is shutting down - don't want to initiate reconnect attempt...
                    break;
                default:
                    break;
            }

            return connectAttemptRequired;
        }
    }

    private final class ConnectionProxy
    implements ExceptionListener, ConnectionStateChangeListener
    {
        private TopicConnection connection;
        private ProxyExceptionListener proxyExceptionListener;
        private ProxyConnectionStateChangeListener proxyConnectionStateChangeListener;

        private TopicSession session;
        private TopicPublisher publisher;

        private HashMap subscriptions = new HashMap();

        private boolean isActive = true;

        private ConnectionProxy(TopicConnection connection)
        {
            this.connection = connection;
        }

        public TopicConnection getTopicConnection() { return this.connection; }

        public TopicPublisher getTopicPublisher() { return this.publisher; }

        public void setupContext(TopicConnection connection)
        throws JMSException
        {
            synchronized(DurableConnector.this)
            {
                // Sonic00027623: ensure any old listeners don't fire unitended actions
                if (this.proxyExceptionListener != null)
                {
                    this.proxyExceptionListener.deactivate();
                }
                if (this.proxyConnectionStateChangeListener != null)
                {
                    this.proxyConnectionStateChangeListener.active = false;
                }
                // Sonic00027623: We create a new listener each time a connection is reestablshed, this way
                //                onException calls from old connections cannot screw up any new connections
                //                that are created on another thread
                this.proxyExceptionListener = new ProxyExceptionListener();
                this.proxyConnectionStateChangeListener = new ProxyConnectionStateChangeListener();
                this.connection = connection;
                this.session = null;

                // we now have a valid connection so establish some context
                this.session = this.connection.createTopicSession(false, Session.DUPS_OK_ACKNOWLEDGE);
                createSubscriptions();
                // Sonic00027623: Don't set the exception listener until it is useful. Issues that occur up to this point
                //                will have been handled by exception handlers. The only other issues should be those that
                //                occur due to failure of a ping or publish of a request and these will get handled by
                //                the exception listener we are setting here.
                TopicPublisher publisher = this.session.createPublisher(null);
                publisher.setDeliveryMode(m_deliveryMode);
                connection.setExceptionListener(this.proxyExceptionListener);
                ((progress.message.jclient.Connection)connection).setConnectionStateChangeListener(new ProxyConnectionStateChangeListener());
                this.publisher = publisher;
                this.isActive = true;
            }

            if (DEBUG_TRACE_PUBLISH)
            {
                System.out.println("DurableConnector.ConnectProxy.setupContext: Created publisher on new connection");
            }
        }

        private void cleanup()
        {
            // no longer staying connected, so see what we can cleanup

            // reset state
            this.resetForReconnectAttempt();

            // mark as inactive
            this.isActive = false;
        }

        private void resetForReconnectAttempt()
        {
            synchronized(DurableConnector.this)
            {
                // Sonic00027623: ensure any old listeners don't fire unitended actions
                if (this.proxyExceptionListener != null)
                {
                    this.proxyExceptionListener.deactivate();
                }
                if (this.proxyConnectionStateChangeListener != null)
                {
                    this.proxyConnectionStateChangeListener.active = false;
                }

                // probably pointless, but make attempt to cleanup connection...
                progress.message.jimpl.Connection connection = (progress.message.jimpl.Connection)this.connection;
                if (connection != null)
                {
                    if (connection.getConnectionState() < Constants.FAILED)
                    {
                        try { connection.stop(); } catch(Throwable e) { } // ignore

                        // attempt to clean up subscriptions
                        Object[] subscriptions = this.subscriptions.keySet().toArray();
                        for (int i = 0; i < subscriptions.length; i++)
                        {
                            Subscription subscription = (Subscription)subscriptions[i];
                            Object[] details = (Object[])this.subscriptions.get(subscription);
                            try { ((TopicSubscriber)details[1]).close(); } catch(Throwable e) { } // ignore
                            if (subscription.subscriptionName != null)
                             {
                                // only do for durable subscriptions
                                try { ((TopicSession)details[0]).unsubscribe(subscription.subscriptionName); } catch(Throwable e) { } // ignore
                            }
                        }
                    }

                    // probably also pointless, but attempt to close the connection just to be safe...
                    try { connection.close(); } catch(Throwable e) { } // ignore
                }

                // clear up references
                this.publisher = null;
                this.session = null;
                this.connection = null;
            }
        }

        private void subscribe(Subscription subscription)
        throws JMSException
        {
            if (DEBUG_TRACE_SUBSCRIBE)
            {
                System.out.println("DurableConnector.ConnectionProxy.subscribe: isActive = " + isActive + ", subscription = " + subscription.subscriptionName);
            }

            synchronized(DurableConnector.this)
            {
                if (this.isActive)
                {
                    this.subscriptions.put(subscription, new Object[] { null, null});

                    if (this.session != null)
                    {
                        createSubscriptions();
                    }
                }
            }
        }

        private void unsubscribe(Subscription subscription)
        {
            if (DEBUG_TRACE_SUBSCRIBE)
            {
                System.out.println("DurableConnector.ConnectionProxy.unsubscribe: isActive = " + isActive + ", subscription = " + subscription.subscriptionName);
            }

            synchronized(DurableConnector.this)
            {
                if (this.isActive)
                {
                    Object[] details = (Object[])this.subscriptions.remove(subscription);
                    if (details != null)
                    {
                        try
                        {
                            ((TopicSubscriber)details[1]).close();
                        }
                        catch (Throwable e)
                        {
                            //e.printStackTrace(); // only uncomment to debug issues
                        } // ignore
                        try
                        {
                            ((TopicSession)details[0]).unsubscribe(subscription.subscriptionName);
                        }
                        catch (Throwable e)
                        {
                            //e.printStackTrace(); // only uncomment to debug issues
                        } // ignore
                    }
                }
            }
        }

        /**
         * Goes through the list and if any of the subscription are not on the current session
         * then update them.
         */
        private void createSubscriptions()
        throws JMSException
        {
            synchronized(DurableConnector.this)
            {
                String subscriptionName = null;
                IGlobalComponentListener globalComponentListener = null;
                boolean stopped = false;

                try
                {
                    // loop through and see if we need to recreate the subscriptions
                    // can't set up the subscriptions until the connection has been made and
                    // session established - at that time attempts will be made to (re)create any
                    // subscriptions
                    if (this.session != null)
                    {
                        this.connection.stop();
                        stopped = true;
                    }

                    Object[] entries = this.subscriptions.entrySet().toArray();
                    for (int i = 0; i < entries.length; i++)
                    {
                        Map.Entry entry = (Map.Entry)entries[i];
                        Subscription subscription = (Subscription)entry.getKey();
                        Object[] details = (Object[])entry.getValue();
                        TopicSession session = (TopicSession)details[0];
                        TopicSubscriber subscriber = (TopicSubscriber)details[1];

                        // try to cleanup any old stuff
                        if (subscriber != null)
                        {
                            if (session == this.session)
                            {
                                continue;
                            }

                            try { subscriber.close(); } catch(Throwable e) { }
                        }

                        // if the session is not set then we can't actually make the subscriptions now ..
                        // however they will get made when the connection is established
                        if (this.session == null)
                        {
                            continue;
                        }

                        if (DEBUG_TRACE_SUBSCRIBE)
                        {
                            System.out.println("ConnectionProxy.createSubscriptions: attempt subscribe: topic=" + subscription.topic.getTopicName() + " subname=" + subscription.subscriptionName);
                        }

                        subscriptionName = subscription.subscriptionName;
                        globalComponentListener = subscription.globalComponentListener;

                        // make the subscription
                        if (subscription.subscriptionName == null)
                        {
                            subscriber = this.session.createSubscriber(subscription.topic);
                        }
                        else // durable
                        {
                            // we make use of durable subscription expiration only for the case where a container
                            // dies (a smooth shutdown causes the subscription to be removed, dying does not) and
                            // never comes back - if we did not use durable subsciption expiration, the subscription
                            // would never get cleaned up
                            // expiration occurs after 1 hour currently
                            try
                            {
                                subscriber = ((progress.message.jclient.TopicSession)this.session).createDurableSubscriber(subscription.topic, subscription.subscriptionName, DURABLE_SUBSCRIPTION_TTL);
                            }
                            catch(JMSException e)
                            {
                                Exception le = e.getLinkedException();
                                if ((le != null) && (le instanceof EUserAlreadyConnected) && globalComponentListener != null)
                                {
                                    StringTokenizer st = new StringTokenizer(subscriptionName, "/");
                                    st.nextToken();
                                    globalComponentListener.globalComponentAlreadyExists(st.nextToken());
                                    details[0] = null;
                                    details[1] = null;
                                    continue;
                                }
                                else
                                {
                                    throw e;
                                }
                            }
                        }

                        if (DEBUG_TRACE_SUBSCRIBE)
                        {
                            System.out.println("ConnectionProxy.createSubscriptions: succeed subscribe: topic=" + subscription.topic.getTopicName() + " subname=" + subscription.subscriptionName + " listener=" + subscription.listener);
                        }

                        subscriber.setMessageListener(subscription.listener);

                        // now update our structures
                        details[0] = this.session;
                        details[1] = subscriber;
                    }
                }
                catch(JMSSecurityException e) // should only be when client does not have permission to subscribe
                {
                    throw e;
                }
                catch(JMSException e)
                {
                    Exception le = e.getLinkedException();
                    if ((le != null) && (le instanceof EUserAlreadyConnected))
                    {
                        DurableConnector.this.m_consumer.logMessage(m_consumerType + "connection: Identity already in use: " + subscriptionName, Level.SEVERE);

                        // if we got here, then the container startup should be aborted and the exception rethrown
                        DurableConnector.this.m_consumer.logMessage("Aborting container", Level.SEVERE);
                        throw e;
                    }

                    // else other exceptions should be picked up when we try to reconnect and if
                    // we do reconnect the subscriptions will be recreated
                    return; // when we reconnect again the subscriptions will be recreated
                }
                finally
                {
                    if (this.session != null && stopped)
                    {
                        this.connection.start();
                    }
                }
            }
        }

        /**
         * Tests if we should try to connect and maintain context on this connection.
         */
        private boolean isConnected()
        {
            synchronized(DurableConnector.this)
            {
                return (this.publisher != null);
            }
        }

        //
        // ProxyExceptionListener queues the exception received from the connection and process them one by one. The goal is to prevent
        // the concurrent processing of execptions that cause the SNC00072239 deadlock
        //

        private class ProxyExceptionListener
        implements ExceptionListener
        {
            private final int MAX_LIST_SIZE = 50; //An Arbitrary large number - much larger than is actually needed.
                                                    //We limit the size to make sure some unusual condition will cause the
                                                    // explosion of this queue
            private volatile boolean m_activeObject;
            private ArrayList m_exceptionList;

            ProxyExceptionListener()
            {
                m_activeObject = true;
                m_exceptionList = new ArrayList();

                Thread exceptionProcessor = new Thread("com.sonicsw.mf.comm.jms.ConnectionProxy.ProxyExceptionListener Exception Processor thread")
                {
                    @Override
                    public void run()
                    {
                        while(m_activeObject)
                        {
                            processExceptions();
                        }
                    }
                };

                exceptionProcessor.setDaemon(true);
                exceptionProcessor.start();
            }

            @Override
            public void onException(JMSException exception)
            {
                synchronized(this)
                {
                    if (m_activeObject)
                    {
                        m_exceptionList.add(new ConnException(exception, ConnectionProxy.this.connection));

                        //Remove an old exception - will not be needed anyway
                        if (m_exceptionList.size() >= MAX_LIST_SIZE)
                        {
                            m_exceptionList.remove(0);
                        }

                        notifyAll();
                    }
                }
            }

            private void processExceptions()
            {
                ConnException connException = null;
                synchronized(this)
                {
                    while (m_exceptionList.isEmpty())
                    {
                        try
                        {
                            wait();
                            if (!m_activeObject)
                            {
                                return;
                            }
                        }
                        catch (InterruptedException e)
                        {
                            return;
                        }
                    }

                    connException = (ConnException)m_exceptionList.remove(0);
                }

                if (m_activeObject)
                {
                    ConnectionProxy.this.onException(connException.m_exception, connException.m_connectionProducer);
                }
            }

            private synchronized void deactivate()
            {
                m_activeObject = false;
                notifyAll();
            }

            @Override
            public void finalize()
            {
                deactivate();
            }
        }

        private class ConnException
        {
            private JMSException m_exception;
            private TopicConnection m_connectionProducer;

            ConnException (JMSException exception, TopicConnection connectionProducer)
            {
                m_exception = exception;
                m_connectionProducer = connectionProducer;
            }
        }

        private void onException(JMSException exception, TopicConnection exceptionProducer)
        {
            synchronized(DurableConnector.this)
            {
                if (exceptionProducer == null || exceptionProducer != ConnectionProxy.this.connection)
                {
                    return;
                }
                else
                {
                    onException(exception);
                }
            }
        }

        @Override
        public void onException(JMSException exception)
        {
            synchronized(DurableConnector.this)
            {
                if (this.connection == null)
                {
                    return;
                }

                if (DEBUG_TRACE_PUBLISH || DEBUG_TRACE_SUBSCRIBE || DEBUG_TRACE_CONNECTION_STATE)
                {
                    System.out.println("ConnectionProxy.onException: JMSException, e = " + exception.getMessage());
                }

                if (!DurableConnector.this.m_isCleanupPending)
                {
                    Exception currentException = exception;
                    if (exception.getLinkedException() != null)
                    {
                        currentException = exception.getLinkedException();
                    }

                    DurableConnector.this.m_consumer.logMessage(m_consumerType + " connection dropped", Level.WARNING);
                    if ((DurableConnector.this.m_consumer.getTraceMask() & IConnectorClient.TRACE_CONNECTION_FAILURES) > 0)
                    {
                        DurableConnector.this.m_consumer.logMessage(m_consumerType + " connection failure, trace follows...", currentException, Level.TRACE);
                    }
                    DurableConnector.this.m_consumer.logMessage("...trying to reconnect...", Level.INFO);
                }

                if (DEBUG_TRACE_PUBLISH || DEBUG_TRACE_SUBSCRIBE || DEBUG_TRACE_CONNECTION_STATE)
                {
                    System.out.println("DurableConnector.ConnectionProxy.onException: before update, current state = " + DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME));
                }

                // may need to reset the connection state, if the connection has been lost/dropped
                updateConnectionState();

                if (DEBUG_TRACE_PUBLISH || DEBUG_TRACE_SUBSCRIBE || DEBUG_TRACE_CONNECTION_STATE)
                {
                    System.out.println("DurableConnector.ConnectionProxy.onException: after update, current state = " + DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME));
                }

                //TODO: determine if we need to null out references to connection, session, and publisher,
                //      and possibly set "isActive" to null if this exception handler is invoked.
                //      Are all connection exceptions the result of total connection failure? - check
                //      JMS spec!!!
                //DONE: The answer is "no, not all connection exceptions are thrown as the
                //      the result of total connection failure" (per the JMS spec)."
                
                // this will wake up the connect thread
                DurableConnector.this.notifyAll();
            }

            Exception e = exception.getLinkedException();

            if (exception instanceof IllegalStateException ||
                ((e != null) && ((e instanceof progress.message.client.EConnectionLimitExceeded) || (e instanceof progress.message.client.ENetworkFailure) || (e instanceof java.io.IOException))))
            {
                // Do NOT want to invoke registered exception listener in the event of
                // a dropped connection,as:
                //   (a) DurableConnector's ConnectThread will try to re-establish connection
                //   (b) invoking ConnectorClient's exception/disconnect listener might
                //       (mistakenly) shutdown container/client
                // However, we do want to invoke any registered IConnectionListener's "onDisconnect" method.
                IConnectionListener listener = DurableConnector.this.m_consumer.getConnectionListener();
                if (listener != null)
                {
                    listener.onDisconnect();
                }
            }
            else
            {
                // alert any ExceptionListener that may have been registered
                // with the ConnectorClient
                DurableConnector.this.m_consumer.onFailure(exception);
            }

        }

        //
        // ConnectionStateChangeListener interface
        //

        private class ProxyConnectionStateChangeListener
        implements ConnectionStateChangeListener
        {
            private boolean active = true;

            @Override
            public void connectionStateChanged(int state)
            {
                if (active)
                {
                    // need to update the node name on an FT reconnect
                    ConnectionProxy.this.connectionStateChanged(state);
                }
            }
        }

        @Override
        public void connectionStateChanged(int state)
        {
            if (state == Constants.ACTIVE)
            {
                DurableConnector.this.m_localRoutingNodeName = ((progress.message.jclient.Connection)connection).getRoutingNodeName();
                DurableConnector.this.m_connectionEnterpriseEnabled = ((progress.message.jimpl.Connection)connection).getZConnection().isEnterpriseEdition();
            }
        }

        private void updateConnectionState()
        {
            short currentConnState = DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
            if (DEBUG_TRACE_CONNECTION_STATE)
            {
                System.out.println("DurableConnector.ConnectionProxy.updateConnectionState: currentConnState = " + currentConnState);
            }
            boolean success = false;
            switch (currentConnState)
            {
                case ConnectionStateManager.CONNECT:
                    break;  // no need to change connection state
                case ConnectionStateManager.DISCONNECTING:
                    break;  // must be shutting down - do not change connection state
                case ConnectionStateManager.CONNECTING:
                case ConnectionStateManager.CONNECTED:
                default:
                    try
                    {
                        success = DurableConnector.this.m_stateManager.requestStateChange(currentConnState,ConnectionStateManager.CONNECT,ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME);
                        if (success)
                        {
                            DurableConnector.this.m_connectionProxy.resetForReconnectAttempt();
                        }
                    }
                    catch(RecoverableStateChangeException nrsce)
                    {
                        // this exception never gets thrown from the
                        // ConnectionAndServicesStateManager implementation of
                        // IStateManager
                    }
                    catch(NonRecoverableStateChangeException nrsce)
                    {
                        // this exception never gets thrown from the
                        // ConnectionAndServicesStateManager implementation of
                        // IStateManager
                    }
            }

            //TODO: figure out what to do if "success" is false (i.e. state change request failed...)

            if (DEBUG_TRACE_CONNECTION_STATE)
            {
                System.out.println("DurableConnector.ConnectionProxy.updateConnectionState: prior to return, currentConnState = "  + DurableConnector.this.m_stateManager.getState(ConnectionStateManager.CONNECTOR_CLIENT_CLASSNAME));
            }

        }
    }

    //
    // IStateListener implementation(s)
    //
    public final class ConnectionStateListener
    implements IStateListener
    {
        @Override
        public void stateChanging(short currentState, short intendedState)
        {
            // trace message
            if ((m_consumer.getTraceMask() & IConnectorClient.TRACE_CONNECTION_STATE_CHANGES) > 0)
            {
                DurableConnector.this.m_consumer.logMessage(DurableConnector.this.m_consumerType +
                                                            " connection state changing from state=" + currentState + " to state= " + intendedState, Level.TRACE);
            }
        }

        @Override
        public void stateChanged(short previousState, short currentState)
        {
            // trace message
            if ((m_consumer.getTraceMask() & IConnectorClient.TRACE_CONNECTION_STATE_CHANGES) > 0)
            {
                DurableConnector.this.m_consumer.logMessage(DurableConnector.this.m_consumerType +
                                                            " connection state successfully changed from state=" + previousState + " to state= " + currentState, Level.TRACE);
            }
        }

        @Override
        public void stateChangeFailed(short currentState, short intendedState)
        {
            // trace message
            if ((m_consumer.getTraceMask() & IConnectorClient.TRACE_CONNECTION_STATE_CHANGES) > 0)
            {
                DurableConnector.this.m_consumer.logMessage(DurableConnector.this.m_consumerType +
                                                            " connection state failed to change from state=" + currentState + " to state= " + intendedState, Level.TRACE);
            }
        }
    }
}
