package com.sonicsw.mf.comm.jms;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

import javax.jms.BytesMessage;
import javax.jms.JMSException;
import javax.jms.Message;

import com.sonicsw.mf.comm.IDurableConnectorConsumer;
import com.sonicsw.mf.common.MgmtMsgTooBigException;
import com.sonicsw.mf.common.runtime.Level;

public class MessageSizeValidator
{
    public static final String MESSAGE_SIZE_MAX_MB_PROPERTY = "sonicsw.mf.msgSizeMaxMb";
    private static final int MESSAGE_SIZE_MAX_BYTES;

    public static final String MESSAGE_SIZE_WARN_MB_PROPERTY = "sonicsw.mf.msgSizeWarnMb";
    private static final int MESSAGE_SIZE_WARN_BYTES;
    
    public static final String MESSAGE_SIZE_MAX_WARNINGS_PROPERTY = "sonicsw.mf.msgSizeMaxWarnings";
    private static final int MAX_LOG_WARNINGS;
    
    public static final String MESSAGE_SIZE_MAX_ERRORS_PROPERTY = "sonicsw.mf.msgSizeMaxErrors";
    private static final int MAX_LOG_ERRORS;
    
    public static final String MESSAGE_SIZE_MAX_DUMPS_PROPERTY = "sonicsw.mf.msgSizeMaxDumps";
    private static final int MAX_MSG_DUMPS;
    
    public static final String MESSAGE_SIZE_DUMP_DIR_PROPERTY = "sonicsw.mf.msgSizeDumpDir";
    public static final String MESSAGE_SIZE_DUMP_PREFIX_PROPERTY = "sonicsw.mf.msgSizeDumpPrefix";
    private static final String MESSAGE_DUMP_PREFIX;  // full path prefix, inc' directory (if set) and filename prefix
    private static final String MESSAGE_DUMP_EXTENSION = ".mdump";
    
    
    /* Read configuration from system properties:
     *  -Dsonicsw.mf.msgSizeMaxMb         -- default 9.9
     *  -Dsonicsw.mf.msgSizeWarnMb        -- default 8.0
     *  -Dsonicsw.mf.msgSizeMaxWarnings   -- default 25
     *  -Dsonicsw.mf.msgSizeMaxErrors     -- default 25
     *  -Dsonicsw.mf.msgSizeMaxDumps      -- default 1
     *  -Dsonicsw.mf.msgSizeDumpDir       -- unset (<current working directory>)
     *  -Dsonicsw.mf.msgSizeDumpPrefix    -- "MgmtMsg"
     */
    static
    {
        float mb;
        mb = getFloatProperty(MESSAGE_SIZE_MAX_MB_PROPERTY, (float)9.9);
        MESSAGE_SIZE_MAX_BYTES = (int)(1024.0 * 1024.0 * mb);

        // warning threshold mustn't be higher than the max size
        mb = getFloatProperty(MESSAGE_SIZE_WARN_MB_PROPERTY, (float)8.0);
        int warnBytes = (int)(1024.0 * 1024.0 * mb);
        MESSAGE_SIZE_WARN_BYTES = (warnBytes < MESSAGE_SIZE_MAX_BYTES) ? warnBytes : MESSAGE_SIZE_MAX_BYTES;

        MAX_LOG_WARNINGS = Integer.getInteger(MESSAGE_SIZE_MAX_WARNINGS_PROPERTY, 25);
        MAX_LOG_ERRORS = Integer.getInteger(MESSAGE_SIZE_MAX_ERRORS_PROPERTY, 25);
        MAX_MSG_DUMPS = Integer.getInteger(MESSAGE_SIZE_MAX_DUMPS_PROPERTY, 1);

        String dir = System.getProperty(MESSAGE_SIZE_DUMP_DIR_PROPERTY);
        String filePrefix = System.getProperty(MESSAGE_SIZE_DUMP_PREFIX_PROPERTY, "MgmtMsg");
        MESSAGE_DUMP_PREFIX = (dir != null ? (dir + File.separatorChar) : "") + filePrefix;
    }
    
    private static float getFloatProperty(String propName, float defaultVal)
    {
        float result = (float)-1.0;
        
        try
        {
            String floatStr;
            floatStr = System.getProperty(propName);
            if (floatStr != null)
            {
                result = Float.parseFloat(floatStr);
            }
        }
        catch (Exception e)
        {
            // ignore - use default value
            //e.printStackTrace();  // just for testing
        }
        
        return (result < 0) ? defaultVal : result;
    }
    
    
    private static int s_warningCount = 0;
    private static int s_errorCount = 0;
    private static int s_dumpCount = 0;
    
    private static Object s_countLock = new Object();  // protect access to the static counters
    private static Object s_fileLock = new Object();  // allow only one message dump file to be written at a time, mainly to avoid filename clashes

    private IDurableConnectorConsumer m_consumer;


    public MessageSizeValidator(IDurableConnectorConsumer consumer)
    {
        m_consumer = consumer;
    }

    public void validate(Message jmsMessage, String subject)
    throws JMSException, MgmtMsgTooBigException
    {
        if (!(jmsMessage instanceof progress.message.jclient.Message))
        {
            return;
        }

        progress.message.jclient.Message message = (progress.message.jclient.Message)jmsMessage;
        int bodySize = message.getBodySize();

        // If the message doesn't exceed the thresholds we're done
        if (bodySize < MESSAGE_SIZE_WARN_BYTES)
        {
            return;
        }
        
        // Test which threshold has been exceeded and choose appropriate warning/error message  
        boolean warningOnly;
        String formatStr;
        if (bodySize > MESSAGE_SIZE_MAX_BYTES)
        {
            warningOnly = false;
            formatStr = "Management message in excess of the %1.1fMb limit was generated. The message was rejected.";
        }
        else
        {
            warningOnly = true;
            formatStr = "Management message in excess of the %1.1fMb warning threshold was generated.";
        }
        
        // Use static counters to determine whether to log this warning/error,
        // and whether to dump the message body to a file
        // Note: for a mgmt client the warning/error will only appear on stderr
        // if -Dsonicsw.mf.logConnectionMessages=true (ref' ConnectorClient)
        boolean doLog;
        boolean doDump;
        boolean lastLog;
        boolean lastDump;
        synchronized (s_countLock)
        {
            if (warningOnly)
            {
                doLog = (++s_warningCount <= MAX_LOG_WARNINGS);
                lastLog = s_warningCount == MAX_LOG_WARNINGS;
            }
            else
            {
                doLog = (++s_errorCount <= MAX_LOG_ERRORS);
                lastLog = s_errorCount == MAX_LOG_ERRORS;
            }

            doDump = (doLog && (++s_dumpCount <= MAX_MSG_DUMPS));
            lastDump = s_dumpCount == MAX_MSG_DUMPS;
        }

        // Format warning/error message
        float limitMb = (warningOnly ? MESSAGE_SIZE_WARN_BYTES : MESSAGE_SIZE_MAX_BYTES) / (float)1048576.0;  // 1048576 == 1Mb
        String errorStr = String.format(formatStr, limitMb);
        
        // Log warning/error and dump message content
        if (doLog)
        {
            StringBuilder detailBuff = new StringBuilder(errorStr);
            HashMap<String, String> displayProps = getDisplayProperties(message, subject);
            displayProps.put("Msg body", Integer.toString(bodySize) + " bytes");
            for (Map.Entry<String, String> prop : displayProps.entrySet())
            {
                detailBuff.append("\n    ");
                detailBuff.append(prop.getKey());
                detailBuff.append(": ");
                detailBuff.append(prop.getValue());
            }
            
            if (doDump)
            {
                String dumpfile = writeMsgToFile(message);
                if (dumpfile != null)
                {
                    detailBuff.append("\nMessage dumped to " + dumpfile);
                }
            }
            
            m_consumer.logMessage(detailBuff.toString(), warningOnly ? Level.WARNING : Level.SEVERE);
            
            if (lastLog)
            {
                m_consumer.logMessage("Reached max log limit for large mgmt message "
                                      + (warningOnly ? "warnings" : "errors") + ", limit is "
                                      + (warningOnly ? MAX_LOG_WARNINGS : MAX_LOG_ERRORS),
                                      Level.INFO);
            }
            
            if (lastDump)
            {
                m_consumer.logMessage("Reached max msg dump limit for large mgmt messages, limit is "
                                      + MAX_MSG_DUMPS,
                                      Level.INFO);
            }
        }

        // Throw exception if max size limit exceeded
        if (!warningOnly)
        {
            throw new MgmtMsgTooBigException(errorStr);
        }
    }
    
    /**
     * Extracts a subset of 'useful' mgmt properties from the message
     * for diagnostic use
     *  
     * @param message Mgmt message
     * @return Map containing a subset of the message's properties
     */
    private HashMap<String, String> getDisplayProperties(Message message, String subject)
    {
        HashMap<String, String> displayProps = new LinkedHashMap<String, String>();

        try
        {
            displayProps.put("Subject", subject);
            
            displayProps.put(ConnectorClient.JMS_COMMS_TYPE_PROPERTY, message.getStringProperty(ConnectorClient.JMS_COMMS_TYPE_PROPERTY));
            
            String typeStr;
            short type = message.getShortProperty(ConnectorClient.JMS_CONTENT_TYPE_PROPERTY);
            switch (type)
            {
                case ConnectorClient.NOTIFICATION_CONTENT_TYPE :
                    typeStr = "notification"; break;
                case ConnectorClient.REQUEST_CONTENT_TYPE :
                    typeStr = "request"; break;
                case ConnectorClient.REPLY_CONTENT_TYPE :
                    typeStr = "reply"; break;
                default :
                    typeStr = "unknown";
            }
            
            if (message.getBooleanProperty(ConnectorClient.JMS_ONEWAY_REQUEST_PROPERTY))
            {
                typeStr += " (oneway)";
            }
            
            displayProps.put(ConnectorClient.JMS_CONTENT_TYPE_PROPERTY, typeStr);
            
            displayProps.put(ConnectorClient.JMS_REQUEST_TARGET_PROPERTY, message.getStringProperty(ConnectorClient.JMS_REQUEST_TARGET_PROPERTY));

            displayProps.put(ConnectorClient.JMS_REPLY_TO_PROPERTY, message.getStringProperty(ConnectorClient.JMS_REPLY_TO_PROPERTY));
            
            displayProps.put(ConnectorClient.JMS_REQUEST_OPERATION_PROPERTY, message.getStringProperty(ConnectorClient.JMS_REQUEST_OPERATION_PROPERTY));
        }
        catch (Throwable t)
        {
            m_consumer.logMessage("Diagnostics failure: unable to get mgmt message properties", t, Level.WARNING);
        }
        
        return displayProps;
    }
    
    private String writeMsgToFile(Message message)
    {
        String genFilename = null;
        String fullFilename = null;
        OutputStream os = null;
        Throwable error = null;

        synchronized (s_fileLock)
        {

            try
            {
                String op = message.getStringProperty(ConnectorClient.JMS_REQUEST_OPERATION_PROPERTY);
                genFilename = MESSAGE_DUMP_PREFIX + '-' + System.currentTimeMillis() + '-' + op + MESSAGE_DUMP_EXTENSION;
                File dumpFile = new File(genFilename);
                fullFilename = dumpFile.getAbsolutePath();

                BytesMessage bytesMsg = (BytesMessage)message;
                bytesMsg.reset(); // make the message readable

                if (dumpFile.exists())
                {
                    throw new IOException("File already exists: " + fullFilename);
                }

                os = new BufferedOutputStream(new FileOutputStream(dumpFile));

                byte[] buff = new byte[16384];
                int length = bytesMsg.readBytes(buff);
                while (length > 0)
                {
                    os.write(buff, 0, length);
                    length = bytesMsg.readBytes(buff);
                }
            }
            catch (Throwable t)
            {
                error = t;
                fullFilename = null;
            }
            finally
            {
                try
                {
                    if (os != null)
                    {
                        os.close();
                    }
                }
                catch (IOException e)
                {
                }
            }
        }

        if (error != null)
        {
            m_consumer.logMessage("Diagnostics failure: unable to dump mgmt message to '" + genFilename + "'", error, Level.WARNING);
        }
        
        return fullFilename;
    }
}
