package com.sonicsw.mf.common;

import java.util.ArrayList;

import javax.management.MBeanAttributeInfo;
import javax.management.MBeanNotificationInfo;
import javax.management.MBeanOperationInfo;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.common.config.IElementChange;
import com.sonicsw.mf.common.config.IFSElementChange;
import com.sonicsw.mf.common.config.INamingNotification;
import com.sonicsw.mf.common.info.IManagementInfo;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.runtime.IComponentState;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;

public abstract class AbstractComponent
implements IComponent
{
    protected short m_state = IComponentState.STATE_UNKNOWN;
    protected long m_startTime = 0;
    protected IComponentContext m_context;
    public int m_traceMask = 0;

    private INotification m_lastStateNotification = null;
    
    private static final String DEFAULT_TRACE_MASK_VALUES = "1=verbose,2=set attributes,4=operation invoked,8=notification sent";
    private static final String ENABLE_METRICS_ATTR = "ENABLE_METRICS";

    //
    // IComponent interface
    //

    /**
     * @see com.sonicsw.mf.common.IComponent#init(IComponentContext)
     */
    @Override
    public void init(IComponentContext context)
    {
        m_context = context;
        m_state = IComponentState.STATE_OFFLINE;
    }

    /**
     * The default implementaion sets the state to online, sends a state notification and
     * resets the uptime to 0. Component implementations will typically call this super
     * implementation if they override this method.
     *
     * @see com.sonicsw.mf.common.IComponent#start()
     */
    @Override
    public synchronized void start()
    {
        setState(IComponentState.STATE_ONLINE);
        m_startTime = System.currentTimeMillis();
    }

    /**
     * The default implementaion sets the state to offline, sends a state notification and
     * resets the uptime to 0. Component implementations will typically call this super
     * implementation if they override this method.
     *
     * @see com.sonicsw.mf.common.IComponent#stop()
     */
    @Override
    public synchronized void stop()
    {
        setState(IComponentState.STATE_OFFLINE);
        m_startTime = 0;
    }

    /**
     * @see com.sonicsw.mf.common.IComponent#destroy()
     */
    @Override
    public void destroy()
    {
        m_state = IComponentState.STATE_UNKNOWN;

        // help GC
        m_context = null;
    }

    /**
     * @see com.sonicsw.mf.common.IComponent#getState()
     */
    @Override
    public Short getState() { return new Short(m_state); }

    /**
     * @see com.sonicsw.mf.common.IComponent#getStateString()
     */
    @Override
    public String getStateString() { return IComponentState.STATE_TEXT[m_state]; }

    /**
     * @see com.sonicsw.mf.common.IComponent#getUptime()
     */
    @Override
    public Long getUptime()
    {
        long uptime = m_startTime == 0 ? 0 : System.currentTimeMillis() - m_startTime;
        return new Long(uptime);
    }

    /**
     * @see com.sonicsw.mf.common.IComponent#enableMetrics(IMetricIdentities[])
     */
    @Override
    public void enableMetrics(IMetricIdentity[] ids) { }

    /**
     * @see com.sonicsw.mf.common.IComponent#disableMetrics(IMetricIdentities[])
     */
    @Override
    public void disableMetrics(IMetricIdentity[] ids) { }

    /**
     * @see com.sonicsw.mf.common.IComponent#getInstanceMetricNames(IMetricIdentity)
     */
    @Override
    public String[] getInstanceMetricNames(IMetricIdentity id) { return null; }

    /**
     * @see com.sonicsw.mf.common.IComponent#getManagementInfo()
     */
    @Override
    public IManagementInfo[] getManagementInfo() { return new IManagementInfo[0]; }

    /**
     * @see com.sonicsw.mf.common.IComponent#getAttributeInfos()
     */
    @Override
    public MBeanAttributeInfo[] getAttributeInfos()
    {
        // default implementation handles legacy code that supports only getManagementInfo
        IManagementInfo[] mgmtInfos = getManagementInfo();

        if (mgmtInfos == null || mgmtInfos.length == 0)
        {
            return null;
        }

        ArrayList infos = new ArrayList();

        for (int i = 0; i < mgmtInfos.length; i++)
        {
            if (mgmtInfos[i] instanceof MBeanAttributeInfo)
            {
                infos.add(mgmtInfos[i]);
            }
        }

        if (infos.isEmpty())
        {
            return null;
        }

        return (MBeanAttributeInfo[])infos.toArray(IEmptyArray.EMPTY_ATTRIBUTE_INFO_ARRAY);
    }

    /**
     * @see com.sonicsw.mf.common.IComponent#getOperationInfos()
     */
    @Override
    public MBeanOperationInfo[] getOperationInfos()
    {
        // default implementation handles legacy code that supports only getManagementInfo
        IManagementInfo[] mgmtInfos = getManagementInfo();

        if (mgmtInfos == null || mgmtInfos.length == 0)
        {
            return null;
        }

        ArrayList infos = new ArrayList();

        for (int i = 0; i < mgmtInfos.length; i++)
        {
            if (mgmtInfos[i] instanceof MBeanOperationInfo)
            {
                infos.add(mgmtInfos[i]);
            }
        }

        if (infos.isEmpty())
        {
            return null;
        }

        return (MBeanOperationInfo[])infos.toArray(IEmptyArray.EMPTY_OPERATION_INFO_ARRAY);
    }


    /**
     * @see com.sonicsw.mf.common.IComponent#getNotificationInfos()
     */
    @Override
    public MBeanNotificationInfo[] getNotificationInfos()
    {
        // default implementation handles legacy code that supports only getManagementInfo
        IManagementInfo[] mgmtInfos = getManagementInfo();

        if (mgmtInfos == null || mgmtInfos.length == 0)
        {
            return null;
        }

        ArrayList infos = new ArrayList();

        for (int i = 0; i < mgmtInfos.length; i++)
        {
            if (mgmtInfos[i] instanceof MBeanNotificationInfo)
            {
                infos.add(mgmtInfos[i]);
            }
        }

        if (infos.isEmpty())
        {
            return null;
        }

        return (MBeanNotificationInfo[])infos.toArray(IEmptyArray.EMPTY_NOTIFICATION_INFO_ARRAY);
    }

    /**
     * @see com.sonicsw.mf.common.IComponent#getTraceMask()
     */
    @Override
    public Integer getTraceMask() { return new Integer(m_traceMask); }

    /**
     * @see com.sonicsw.mf.common.IComponent#setTraceMask(Integer)
     */
    @Override
    public void setTraceMask(Integer traceMask)
    {
        m_traceMask = traceMask.intValue();
        if (m_traceMask < 0)
        {
            m_traceMask = 0;
        }
    }

    /**
     * @see com.sonicsw.mf.common.IComponent#getTraceMaskValues()
     */
    @Override
    public String getTraceMaskValues() { return DEFAULT_TRACE_MASK_VALUES; }

    /**
     * @see com.sonicsw.mf.common.IComponent#handleElementChange(IElementChange)
     */
    @Override
    public synchronized void handleElementChange(IElementChange elementChange) { }

    @Override
    public void handleFSNamingNotification(INamingNotification notification) {}

    @Override
    public void handleFileChange(IFSElementChange fileChange) {}

    //
    // internal
    //

    protected void setState(short state)
    {
        if (m_state != state)
        {
            m_state = state;

            switch(state)
            {
                case IComponentState.STATE_ONLINE:
                    sendStateNotification(state, Level.INFO, INotification.INFORMATION_TYPE, false);
                    break;
                case IComponentState.STATE_OFFLINE:
                    sendStateNotification(state, Level.INFO, INotification.INFORMATION_TYPE, true);
                    break;
                default:
                    break;
            }
        }
    }

    protected void sendStateNotification(int state, int severityLevel, short logType)
    {
        sendStateNotification(state, severityLevel, logType, false);
    }

    private void sendStateNotification(int state, int severityLevel, short logType, boolean checkLastNotification)
    {
        if (checkLastNotification && m_lastStateNotification!= null && m_lastStateNotification.getEventName().equals(IComponentState.STATE_TEXT[state]))
         {
            return; // don't send again!
        }
        
        INotification notification =
            m_context.createNotification(INotification.SYSTEM_CATEGORY, INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY], IComponentState.STATE_TEXT[state], severityLevel);

        // set the log type as required
        notification.setLogType(logType);

        m_context.sendNotification(notification);
        
        m_lastStateNotification = notification;
    }

    protected boolean matchesMetricPattern(String metricType, String metricPattern)
    {
        if (metricType == null || metricPattern == null || metricType.length() < 1 || metricPattern.length() < 1)
        {
            throw new IllegalArgumentException();
        }

        // straight match ?
        if (metricType.equals(metricPattern))
        {
            return true;
        }

        // any metric ?
        if (metricPattern.equals("*"))
        {
            return true;
        }

        // is there a wildcard suffix
        if (metricPattern.endsWith("*"))
        {
            String prefix = metricPattern.substring(0, metricPattern.length() - 2);
            if (metricType.startsWith(prefix))
            {
                return true;
            }
        }

        return false;
    }

    protected void validateOnline()
    {
        if (m_state != IComponentState.STATE_ONLINE)
        {
            throw new IllegalStateException(m_context.getComponentName().getComponentName() + " is not online");
        }
    }
}
