package com.sonicsw.mf.common;

import javax.management.MBeanAttributeInfo;
import javax.management.MBeanNotificationInfo;
import javax.management.MBeanOperationInfo;

import com.sonicsw.mf.common.config.IElementChange;
import com.sonicsw.mf.common.config.IElementChangeHandler;
import com.sonicsw.mf.common.config.IFSElementChange;
import com.sonicsw.mf.common.config.INamingNotification;
import com.sonicsw.mf.common.info.IManagementInfo;
import com.sonicsw.mf.common.metrics.IMetricIdentity;

/**
 * The IComponent interface is the base interface for all Application and
 * Framework components and contains methods that are common to all types
 * of component. IApplicationComponent and IFrameworkComponent are
 * derived from this interface.
 */
public interface IComponent
extends IElementChangeHandler
{
    /**
     * To be used for bitwise comparison of the components TraceMask
     * when determining if verbose debug information should be logged.
     */
    public static final int TRACE_VERBOSE = 1;
    /**
     * To be used for bitwise comparison of the components TraceMask
     * when determining if setting of component attributes should be logged.
     */
    public static final int TRACE_ATTRIBUTES = 2;
    /**
     * To be used for bitwise comparison of the components TraceMask
     * when determining if invoking of component operations should be logged.
     */
    public static final int TRACE_OPERATIONS = 4;
    /**
     * To be used for bitwise comparison of the components TraceMask
     * when determining if sending of component notifications should be logged.
     */
    public static final int TRACE_NOTIFICATIONS = 8;

    //
    // Methods
    //

    /**
     * Initialize the component. If the component is already initialized
     * then this call should be ignored.
     */
    public void init(IComponentContext context);

    /**
     * Start the component. If the component is starting or started, or the
     * component is being destroyed, this call will be ignored.
     *
     * NOTE: Implementations should be synchronized.
     */
    public void start();

    /**
     * Stop the component. If the component is stopping or stopped, or the
     * component is being destroyed, this call will be ignored.
     *
     * NOTE: Implementations should be synchronized.
     */
    public void stop();

    /**
     * Destroy the component. If the component is already destroyed or was not
     * initialized then this call should be ignored.
     */
    public void destroy();

    /**
     * Get the current state of this component
     */
    public Short getState();

    /**
     * Get the current state of this component
     */
    public String getStateString();

    /**
     * Get the time in milliseconds since this component was started
     */
    public Long getUptime();

    /**
     * Enables the metrics for the requested metric identities. If some or all of
     * the requested metric identities are already enabled, then those identities
     * will be ignored.
     *
     * @param ids A set of metric identities to be disabled which may include
     *            one or more individual metric identities and/or identities
     *            that are branches of the metrics tree (in which case all the
     *            leaves (individual metrics) of that branch will be disbaled).
     */
    public void enableMetrics(IMetricIdentity[] ids);

    /**
     * Disables the metrics for the requested metric identities. If some or all of
     * the requested metric identities are already disabled or are not dynamic
     * metrics, then those identities will be ignored.
     *
     * @param ids A set of metric identities to be disabled which may include
     *            one or more individual metric identities and/or identities
     *            that are branches of the metrics tree (in which case all the
     *            leaves (individual metrics) of that branch will be disbaled).
     */
    public void disableMetrics(IMetricIdentity[] ids);

    /**
     * Gets the instance names of instances to which the instance metric identity
     * applies.
     *
     * @param id The instance metric parent identity.
     *
     * @return An array of instance names. If there are no instances the component should
     *         a zero length array. If the metric identity is either unknown or not that
     *         of an instance metric, then the component should return null;
     */
    public String[] getInstanceMetricNames(IMetricIdentity id);

    /**
     * Get the meta data describing the management attributes, operations and
     * notifications the component supports.
     *
     * @see InfoFactory
     * @deprecated
     */
    public IManagementInfo[] getManagementInfo();

    /**
     * Get the JMX meta data describing the management attributes the component supports.
     *
     * @return Array of attribute infos or null if there are none.
     */
    public MBeanAttributeInfo[] getAttributeInfos();

    /**
     * Get the JMX meta data describing the management operations the component supports.
     *
     * @return Array of operation infos or null if there are none.
     */
    public MBeanOperationInfo[] getOperationInfos();

    /**
     * Get the JMX meta data describing the management notifications the component supports.
     *
     * @return Array of notification infos or null if there are none.
     */
    public MBeanNotificationInfo[] getNotificationInfos();

    /**
     * Get the mask used to specify runtime debugging capabilities.
     *
     * @see #setTraceMask(Integer)
     */
    public Integer getTraceMask();

    /**
     * Sets the mask used to specify runtime debugging capabilities.
     *
     * The mask is a bit mask that defines which debug trace messages should
     * be logged by the component. Components log debug messages using
     * their context [IComponentContext.logMessage()]. All debug looging
     * should provide a log severity of Level.TRACE. A mask value of 0 (zero)
     * or less indicates to the component that it should log no debug messages.
     *
     * Possible mask values are presented using the comma delimited string
     * obtained using getTraceMaskValues().
     *
     * The following list of values are reserved and used by components which
     * extend the AbstractComponent (some default debug logging is provided
     * using those values):
     *
     * 1 - verbose           Intended to be combined with other mask values to
     *                       provide a more detailed output.
     *                       By itself (i.e. a TraceMask of value 1) will not
     *                       produce any debug output.
     * 2 - set attributes    When an attribute exposed by the component is set
     *                       a message will be logged that indicates the name of
     *                       the attribute being set.
     *                       When combined with the "detailed" mask value, the
     *                       before and after value will be logged (using toString()
     *                       to display the values).
     * 4 - operation invoked When a management operation is invoked a message will
     *                       be logged indicating the invoked operation name and
     *                       signature.
     *                       When combined with the "detailed" mask value, the
     *                       arguments and return values will be logged (using
     *                       toString() to display the values).
     * 8 - notification sent When a notification is published a message will be
     *                       logged indicating the notification type and severity.
     *                       When combined with the "detailed" mask value, the
     *                       notification attributes, if any, will be logged.
     *
     * @see #getTraceMask()
     * @see #getTraceMaskValues()
     * @see com.sonicsw.mf.common.IComponentContext
     * @see com.sonicsw.mf.common.AbstractComponent
     * @see com.sonicsw.mf.common.runtime.Level#TRACE
     */
    public void setTraceMask(Integer traceMask);

    /**
     * Gets the textual description of possible debug mask values for this
     * component. The description is provided as a comma seperated list of
     * <mask value>=<short description> values.
     *
     * Components that extend AbstractComponent will have the following
     * returned by getTraceMask():
     *
     * "1=verbose,2=set attributes,4=operation invoked,8=notification sent"
     *
     * Components that extend AbstractComponent can return their specific
     * debug mask values by implementing:
     *
     *   super.getMaskValues() + <additional mask values list>
     */
    public String getTraceMaskValues();


    /**
     * @see com.sonicsw.mf.common.IElementChangeHandler
     *
     * NOTE: Implementations should be synchronized.
     */
    @Override
    public void handleElementChange(IElementChange elementChange);

    public void handleFSNamingNotification(INamingNotification notification);

    /**
     * Handle a change in a file element.
     *
     * @param fileChange Description of the change to the file element
     *
     */

    public void handleFileChange(IFSElementChange fileChange);

}
