package com.sonicsw.mf.common;

import java.util.Date;

import javax.naming.Context;

import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.metrics.IMetricInfo;
import com.sonicsw.mf.common.metrics.manager.IMetricsRegistrar;
import com.sonicsw.mf.common.runtime.ICanonicalName;
import com.sonicsw.mf.common.runtime.INotification;

/**
 * The IComponentContext interface context for any MF component.
 *
 * It provides access to information and services the component needs
 * to run. It also acts as a factory for certain kinds of objects.
 */
public interface IComponentContext
{
	public static final int	NOTAVAILABLE = -1;
	public static final int	MAYBENOTAVAILABLE = 0;
	public static final int	AVAILABLE = 1;

    public void restartContainer()
    throws Exception;


    /**
     * Gets the component name associated with this context.
     */
    public ICanonicalName getComponentName();

    /**
     * Sets configuration information for the component in a reliable fashion.
     *
     * @param configuration The configuration to be set.
     */
    public void setRuntimeConfiguration(IDirElement configuration);

    /**
     * Gets configuration information for the component.
     *
     * @param acceptChanges If true, the component wishes to receive incremental changes
     *                      to its configuration (not all components will be capable of
     *                      handling configuration changes).
     */
    public IElement getConfiguration(boolean acceptsChanges);

    /**
     * Gets configuration information for the named configuration. A components configuration
     * information may include references to other pieces of configuration information where
     * references provide a stringified form of referenced configuration identity.
     *
     * @param configID      The name of a configuration element as known by the Directory Service
     *                      (IIdentity.getName()).
     * @param acceptChanges If true, the component wishes to receive incremental changes
     *                      to its configuration (not all components will be capable of
     *                      handling configuration changes).
     */
    public IElement getConfiguration(String configID, boolean acceptChanges);

     /**
     * Gets configuration information for the named configuration. A components configuration
     * information may include references to other pieces of configuration information where
     * references provide a stringified form of referenced configuration identity.
     *
     * @param configID      The name of a configuration element as known by the Directory Service
     *                      (IIdentity.getName()).
     * @param acceptChanges If true, the component wishes to receive incremental changes
     *                      to its configuration (not all components will be capable of
     *                      handling configuration changes).
     * @param alwaysFromDS  if true, the container will alwasy go to the DS for the configuration and
     *                      never look in its cache.
     */
    public IElement getConfiguration(String configID, boolean acceptChanges, boolean alwaysFromDS);

    /**
     * Check if the element specified by the logical name exists in the cache and/or the DS.  This method shall not wait for 
     * the DS to come online if the container is in its boot phase (as other API that goes to the DS does).
     * 
     * @param path	The logical name of a configuration element as known by the Directory Service
     * @return		AVAILABLE, if the element found in the cache or in the DS; 
     * 				NOTAVAILABLE, if not in the cache and not in the DS; 
     * 				MAYBENOTAVAILABLE, if not found in the cache and the DS is unreachable. 
     */
    public int checkFSConfiguration(String path); 
    
    /**
     * Gets configuration information for the named configuration. A components configuration
     * information may include references to other pieces of configuration information where
     * references provide a stringified form of referenced configuration identity.
     *
     * @param path      The logical name of a configuration element as known by the Directory Service
     *
     * @param notify     If true, the component wishes to receive incremental changes
     *                      to its configuration (not all components will be capable of
     *                      handling configuration changes).
     */

    public IElement getFSConfiguration(String path, boolean notify);

    /**
     * Gets configuration information for the named configuration. A components configuration
     * information may include references to other pieces of configuration information where
     * references provide a stringified form of referenced configuration identity.
     *
     * @param path      The logical name of a configuration element as known by the Directory Service
     *
     * @param notify     If true, the component wishes to receive incremental changes
     *                      to its configuration (not all components will be capable of
     *                      handling configuration changes).
     * @param alwaysFromDS If true, the container will always fetch the configuration from the DS; it will not
     *                     look in its cache for the configuration
     */

    public IElement getFSConfiguration(String path, boolean notify, boolean alwaysFromDS);

    /** Lets a component register interest in changes to a file stored in the Directory Service
     *
     * @param path Logical name of the file element.
     *
     * @return The file envelope element.
     */

    public IElement registerFileChangeInterest(String path);
    
    /**
     *  Lets a component unregister the interest in changes to a file in the Directory Service
     *  @param path Logical name or sonicfs url of the file element
     *  
     */
    public void unregisterFileChangeInterest(String path);

    /** Returns the file on disk corresponding to path
     *
     * @param path name of the file. If path starts with sonicfs:///, getLocalFile uses the rest if path
     * as the logical name of a file. In this case path can name a file that is stored in the DS or a file contained
     * in an expandable jar file in the DS. If path does not start with sonicfs:///, getLocalFile looks for a file
     * on disk  that exactly matches path
     *
     * @throws MFException if path contains an invalid URL or the persistent cache has not been enabled
     * @return The file corresponding to path.
     */

    public java.io.File getLocalFile(String path) throws MFException;

    /**
     * Gets configuration information for the named configuration list.
     *
     * @param configIDs     The names of a configuration elements
     * @param acceptChanges Each acceptChanges item corresponds to a configID item in the configIDs list.
     *                      If true, the component wishes to receive incremental changes
     *                      to these configuration.
     */
    public IElement[] getConfigurations(String[] configIDs, boolean[] acceptChanges);

    /**
     * Gets configuration information for the named configuration list.
     *
     * @param paths      The logical names of configuration elements
     * @param notify     If true, the component wishes to receive incremental change notifications
     *                      about all the elements in paths (not all components will be capable of
     *                      handling configuration changes).
     */

    public IElement[] getFSConfigurations(String[] paths, boolean notify);


    /**
     * Gets all the configuration elements under a directory.
     *
     * @param dirName       The name of a Directory Service directory.
     *
     * @param acceptChanges If true, the component wishes to receive incremental changes
     *                      to its configuration (not all components will be capable of
     *                      handling configuration changes).
     *
     */
    public IElement[] getConfigurations(String dirName, boolean acceptChanges);

    /**
     * Create a notification.
     *
     * @param category    Each notification belongs to a major category. Permissible values are
     *                    defined in INotification (see <>_CATEGORY constants).
     * @param subCategory Each notification has a minor category.
     * @param eventName   Each notification type is unique by category/sub category/event name. Event names
     *                    are descriptive of what event generated the notification.
     * @param severity    Each notification has a severity associated with it. Severities are used by
     *                    client applications to prioritize multiple notifications. Additionally the severity
     *                    of certain notifications will be used to derive an aggregated state for a container.
     *                    Permissible values are defined in INotification (see <>_SEVERITY constants).
     *
     * @return An notification object.
     */
    public INotification createNotification(short category, String subCategory, String eventName, int severityLevel);

    /**
     * Send the given notification to interested parties.
     */
    public void sendNotification(INotification notification);

    /**
     * Log a message to the system log mechanism. Some logMessages will be as a result of an error
     * condition on the component. Such errors should also be registered with the context.
     *
     * @see #registerErrorCondition(String, int)
     */
    public void logMessage(String message, int severity);

     /**
     * Log a message to the system log mechanism. Some logMessages will be as a result of an error
     * condition on the component. Such errors should also be registered with the context.
     *
     * @param message The description associted with the error condition.
     * @param exception The Exception thrown
     * @param severity The severity of the error - see Level for values.
     * @see #registerErrorCondition(String, int)
     */
    public void logMessage(String message, Throwable exception, int severity);

     /**
     * Log a message to the system log mechanism. Some logMessages will be as a result of an error
     * condition on the component. Such errors should also be registered with the context.
     *
     * @param exception  The Exception thrown
     * @param severity The severity of the error - see Level for values.
     * @see #registerErrorCondition(String, int)
     */
    public void logMessage(Throwable exception, int severity);

    /**
     * Indicates the error condition for the component. A component can be in any
     * of the defined states (online, offline, starting, stopping) and still have
     * an error condition. The error condition is cleared either by calling
     * clearErrorCondition() on the context or administratively through the
     * management API.
     *
     * A new error condition will not override a previously set error condition with
     * a higher severity.
     *
     * @param errorDescription The description associted with the error condition.
     * @param errorSeverity    The severity of the error - see Level
     *                         for values.
     *
     * @return A false will be returned if the error was not registered due to a higher
     *         severity error condition already having been registered (and not cleared).
     *
     * @see #clearErrorCondition()
     * @see com.sonicsw.mf.common.runtime.Level
     */
    public boolean registerErrorCondition(String errorDescription, int errorLevel);

    /**
     * Clears any error condition previously registered for the component. Sets the
     * current registered error condition to IComponentState.UKNOWN_SEVERITY.
     * Components are responsible for clearing an error condition if it should be
     * cleared automatically (e.g. on component restart).
     *
     * @see #registerErrorCondition(String, int)
     * @see com.sonicsw.mf.common.runtime.IComponentState
     */
    public void clearErrorCondition();

    /**
     * Schedule a task to be executed at the given date/time.
     */
    public void scheduleTask(Runnable task, Date startTime);

    /**
     * Remove the proviously scheduled task from the list of scheduled tasks.
     */
    public void cancelTask(Runnable task);

    /**
     * A component initailizes metrics management to tell the management framework that it has some metric(s)
     * to expose through the component's external management API. The framework will ensure metrics management
     * is only initialized once during component initialization (IComponent.init()).
     *
     * @param metricInfos An array of meta-data for all the metrics the component is capable of
     *                    supporting.
     *
     * @return An instance of a metrics registrar (on which metrics may be regsitered).
     */
    public IMetricsRegistrar initMetricsManagement(IMetricInfo[] metricInfos);

    /**
     * Fires the configuration change handlers - typically called from the component's handleElementChange() method.
     * @see com.sonicsw.mf.common.config.IAttributeSet#registerAttributeChangeHandler(Object, IAttributeChangeHandler)
     * @see com.sonicsw.mf.common.config.IAttributeSet#unregisterAttributeChangeHandler(Object)
     * @see com.sonicsw.mf.common.config.IAttributeSet#unregisterAttributeChangeHandler(Object, String)
     * @see com.sonicsw.mf.common.config.IAttributeSet#registerAttributeChangeHandler(Object, String, IAttributeChangeHandler)
     * @see com.sonicsw.mf.common.config.IAttributeList#registerAttributeChangeHandler(Object, IAttributeChangeHandler)
     * @see com.sonicsw.mf.common.config.IAttributeList#unregisterAttributeChangeHandler(Object)
     * @see com.sonicsw.mf.common.config.IAttributeList#unregisterAttributeChangeHandler(Object, int)
     * @see com.sonicsw.mf.common.config.IAttributeList#registerAttributeChangeHandler(Object, int, IAttributeChangeHandler)
     */
    public void fireAttributeChangeHandlers();

    /**
     * Returns the attribute set for the deployment parameters associated with the component.
     *
     * @return An instance of an IAttributeSet that contains the deployment parameters associated with the component.
     */
    public IAttributeSet getDeploymentParameters();
    
    /**
     * Returns the container configuration element.
     * 
     * @param acceptChanges If true, the component wishes to receive incremental changes
     *                      to the container configuration.
     *  
     * @return the container configuration
     */
    public IElement getContainerConfiguration(boolean acceptsChanges);

    /**
     * Get the private sub directory for a component in the container. The directory name is a directory
     * under the given base directory with the name <domain>.<container>.<component>
     * @param baseDir The parent directory for the storage directory.
     * @return The name of the private sub directory for the calling component. The directory is created if it doesn't exist.
     */

    public String getPrivateSubDir(String baseDir);
    /**
     * Set the fault detect connection in the fault tolerance conntainer.
     * If the container is not a FT container, no operation and return null.
     * <p>
     * 
     * @param connection Must be an instance of javax.jms.Connection
     * @return An instance of a progress.message.jclient.ConnectionStateChangeListener (but the caller must cast it as this)
     * @throws Exception
     */
    public Object setFaultDetectionConnection(Object jmsConnection) throws Exception;

    /**
     * Returns a JNDI context to lookup JNDI objects stored in the DS.
     * @return context to be used by components to lookup JNDI objects stored in the DS
     */

    public Context getInitialContext();
  }
